package com.rocogz.syy.order.client;

import com.rocogz.syy.common.PageTable;
import com.rocogz.syy.common.response.Response;
import com.rocogz.syy.order.client.req.GiftLogQuery;
import com.rocogz.syy.order.constant.*;
import com.rocogz.syy.order.constant.trace.BussiSystemEnum;
import com.rocogz.syy.order.constant.trace.BussiTypeEnum;
import com.rocogz.syy.order.constant.trace.CallApiEnum;
import com.rocogz.syy.order.dto.*;
import com.rocogz.syy.order.dto.appoint.AppointAdminSearchParamDto;
import com.rocogz.syy.order.dto.appoint.AppointWriteOffDto;
import com.rocogz.syy.order.dto.appoint.AppointWriteOffPreparedDataDto;
import com.rocogz.syy.order.dto.appoint.CreateAppointOrderDto;
import com.rocogz.syy.order.dto.evaluate.AdminSearchEvaluationParamDto;
import com.rocogz.syy.order.dto.evaluate.OrderEvaluateSummaryStatistic;
import com.rocogz.syy.order.dto.goods.*;
import com.rocogz.syy.order.dto.inspection.CarInspectionAdminSearchParamsDTO;
import com.rocogz.syy.order.dto.inspection.CarInspectionOrderCreateDTO;
import com.rocogz.syy.order.dto.insurancegift.*;
import com.rocogz.syy.order.dto.oil.*;
import com.rocogz.syy.order.dto.pay.OrderPayDetailDto;
import com.rocogz.syy.order.dto.pay.WxPayRefundParamDto;
import com.rocogz.syy.order.dto.pay.WxPayUnifiedOrderParamDto;
import com.rocogz.syy.order.dto.pay.WxPayUnifiedOrderResultDto;
import com.rocogz.syy.order.dto.peccancy.CreatePeccancyOrderDto;
import com.rocogz.syy.order.dto.peccancy.PeccancyAdminSearchParamDto;
import com.rocogz.syy.order.dto.peccancy.PeccancyRecordTotalDto;
import com.rocogz.syy.order.dto.peccancy.QueryPeccancyConditionsDto;
import com.rocogz.syy.order.dto.trace.ApiRequestLogStatisticsResponseDto;
import com.rocogz.syy.order.dto.withdrawals.*;
import com.rocogz.syy.order.entity.evaluate.OrderEvaluate;
import com.rocogz.syy.order.entity.goods.GoodsOrder;
import com.rocogz.syy.order.entity.goods.GoodsOrderBestowInfo;
import com.rocogz.syy.order.entity.goods.GoodsOrderDeliveryAddress;
import com.rocogz.syy.order.entity.goods.GoodsOrderPointFeedback;
import com.rocogz.syy.order.entity.inspection.CarInspectionOrder;
import com.rocogz.syy.order.entity.insuranceGift.*;
import com.rocogz.syy.order.entity.oil.OilDepositRecord;
import com.rocogz.syy.order.entity.oil.OrderOilDeposit;
import com.rocogz.syy.order.entity.orders.OrderLog;
import com.rocogz.syy.order.entity.orders.OrderPayment;
import com.rocogz.syy.order.entity.orders.OrderPaymentRecord;
import com.rocogz.syy.order.entity.orders.SaleOrder;
import com.rocogz.syy.order.entity.pay.OrderWxPayPaymentFlow;
import com.rocogz.syy.order.entity.pay.WxPayRecord;
import com.rocogz.syy.order.entity.pay.WxPayRefundRecord;
import com.rocogz.syy.order.entity.peccancy.OrderPeccancyOrder;
import com.rocogz.syy.order.entity.peccancy.OrderPeccancyQueryRecord;
import com.rocogz.syy.order.entity.peccancy.OrderPeccancyRecord;
import com.rocogz.syy.order.entity.trace.OrderErrorTrace;
import com.rocogz.syy.order.entity.trace.OrderThirdTrace;
import com.rocogz.syy.order.entity.withdrawals.WithdrawalsFeeRateConfig;
import com.rocogz.syy.order.entity.withdrawals.WithdrawalsFeeRateConfigLog;
import com.rocogz.syy.order.entity.withdrawals.WithdrawalsRecord;
import org.springframework.cloud.openfeign.SpringQueryMap;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;

/**
 * 订单接口
 *
 * @author：zhangmin
 * @since：2020/3/25
 */
@SuppressWarnings("all")
public interface IOrderClientService {
    /**
     * 查询订单支付明细
     * @param orderCode
     * @return
     */
    @GetMapping("/order/paymentRecord/listByOrderCode")
    List<OrderPaymentRecord> listOrderPaymentRecordByOrderCode(@RequestParam("orderCode") String orderCode);

    @GetMapping("/order/paymentRecord/listByOrderCodeAndPayWay")
    List<OrderPaymentRecord> listOrderPaymentRecordByOrderCodeAndPayWay(
            @RequestParam("orderCode") String orderCode,
            @RequestParam(value = "payWay", required = false) String payWay);

    /**
     * 创建订单
     *
     * @param order
     */
    @PostMapping("/api/order/create")
    Response createOrder(@RequestBody SaleOrder order);

    /**
     * 后台管理->订单搜索
     *
     * @param searchParamDto
     */
    @GetMapping("/api/order/adminSearch")
    PageTable<SaleOrder> adminSearch(@SpringQueryMap AdminSearchParamDto searchParamDto);

    /**
     * 获取订单主信息
     *
     * @param orderCode 订单编号
     */
    @GetMapping("/api/order/getOrderMainInfo/{orderCode}")
    Response<SaleOrder> getOrderMainInfo(@PathVariable("orderCode") String orderCode);

    /**
     * 查询订单主信息,如果需要其他信息也一起加载
     *
     * @param orderCode     订单编号
     * @param otherInfoList 需要加载的其他信息选项,用来实现按需加载,如果需要全部其他信息,则传入 ALL
     **/
    @GetMapping("/api/order/getOrderWithOtherInfo/{orderCode}")
    Response<SaleOrder> getOrderWithOtherInfo(@PathVariable("orderCode") String orderCode,
                                              @RequestParam("otherInfoList") List<OtherInfoOption> otherInfoList);

    /**
     * 查询订单主信息,如果需要其他信息也一起加载
     *
     * @param identifyCode  订单标识号
     * @param otherInfoList 需要加载的其他信息选项,用来实现按需加载,如果需要全部其他信息,则传入 ALL
     **/
    @GetMapping("/api/order/getOrderWithOtherByIdentifyCode/{identifyCode}")
    Response<SaleOrder> getOrderWithOtherByIdentifyCode(@PathVariable("identifyCode") String identifyCode,
                                                        @RequestParam("otherInfoList") List<OtherInfoOption> otherInfoList);

    /**
     * 车主端小程序 搜索订单
     *
     * @param userCode
     * @return
     */
    @GetMapping("/api/order/weappSearch")
    PageTable<SaleOrder> weappSearch(@RequestParam("userCode") String userCode,
                                     @RequestParam(value = "status", required = false) String status,
                                     @RequestParam(value = "orderTypeList", required = false) List<String> orderTypeList,
                                     @RequestParam(value = "limit", defaultValue = "10") Integer limit,
                                     @RequestParam(value = "page", defaultValue = "1") Integer page);

    /**
     * 商户端小程序 搜索订单
     *
     * @param storeCode          门店编号[必须]
     * @param keyword：订单号/车牌号/券码 [选传]
     * @param orderTypeList      订单类型列表[选传]
     * @param statusList         订单状态列表 [选传]
     * @param appointStartTime   预约开始时间[选传]
     * @param appointEndTime     预约开始结束时间[选传]
     */
    @GetMapping("/api/order/merchantWeappSearch")
    PageTable<SaleOrder> merchantWeappSearch(@RequestParam(value = "storeCode") String storeCode,
                                             @RequestParam(value = "keyword", required = false) String keyword,
                                             @RequestParam(value = "orderTypeList", required = false) List<String> orderTypeList,
                                             @RequestParam(value = "statusList", required = false) List<String> statusList,
                                             @RequestParam(value = "appointStartTime", required = false) String appointStartTime,
                                             @RequestParam(value = "appointEndTime", required = false) String appointEndTime,
                                             @RequestParam(value = "limit", defaultValue = "10") Integer limit,
                                             @RequestParam(value = "page", defaultValue = "1") Integer page);

    /**
     * 逻辑删除订单
     *
     * @param orderCode 订单号
     * @param userCode  如果是weapp端调用,则需要传入用户编号,用来判断是否是车主自己的订单
     * @return
     */
    @GetMapping("/api/order/delete/{orderCode}")
    Response<String> deleteLogical(@PathVariable("orderCode") String orderCode,
                                   @RequestParam(value = "userCode", required = false) String userCode);

    /**
     * 关闭 超时未支付的订单
     *
     * @param orderCode 订单号
     * @return 如果关单成功, 则返回需要退回的用户券码
     */
    @GetMapping("/api/order/cancelOrder/{orderCode}")
    Response<CancelOrderResultDto> cancelOrder(@PathVariable("orderCode") String orderCode);

    /**
     * 后台管理取消订单
     *
     * @param orderCode 订单号
     * @param updateUser 更新人
     * @return 如果关单成功 response
     */
    @GetMapping("/api/order/adminCancelOrder/{orderCode}")
    public Response<CancelOrderResultDto> adminCancelOrder(@PathVariable("orderCode") String orderCode,
                                                           @RequestParam("updateUser") String updateUser);

    /**
     * 订单的操作日志列表
     *
     * @param orderCode 订单编号
     * @return 返回结果按照id倒序排列
     */
    @GetMapping("/api/order/log/{orderCode}")
    Response<List<OrderLog>> findLogList(@PathVariable("orderCode") String orderCode);

    /**
     * 定时把过了过期时间的订单设置为已过期
     *
     * @return
     */
    @GetMapping("/api/order/scheduleExpireOrder")
    Response<String> scheduleExpireOrder();

    /**
     * 定时查询 需要现金支付并支付超时的订单号(主要解决rabbitmq延迟任务中消息丢失,mq宕机等问题导致的数据不一致行问题)
     *
     * @return
     */
    @GetMapping("/api/order/scheduleFindCashPayTimeoutOrderCodes")
    Response<List<String>> scheduleFindCashPayTimeOutOrderCodes();

    /**
     * 统计订单数量
     *
     * @param orderCountParamDto orderCountParamDto
     * @return the response
     */
    @PostMapping("/api/order/count")
    Response<List<OrderCountResultDto>> saleOrderCount(@RequestBody OrderCountParamDto orderCountParamDto);

    // ======  订单统计 相关接口 start   ===========>

    /**
     * 统计 门店/车主 订单数量, 统计结果 按照订单状态分组
     *
     * @param storeCode 门店编号 [可选]
     * @param userCode  用户编号 [可选]
     * @return Map<订单状态                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                               ,                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                               订单数量>
     */
    @GetMapping("/api/order/statis/countGroupByStatus")
    Response<Map<String, Long>> countOrderNumGroupByStatus(@RequestParam(value = "storeCode", required = false) String storeCode,
                                                           @RequestParam(value = "userCode", required = false) String userCode);

    /**
     * 商户端 统计销售业绩
     *
     * @param statisOptionList
     */
    @PostMapping("/api/order/statis/summarize")
    Response<Map<StatisTimeRangeEnum, OrderSummaryStatistic>> summarizeOrderByStoreCode(@RequestBody List<OrderStatisOption> statisOptionList);

    /**
     * 商户端: 统计今日最新：未核销订单数量、预约数、评论数
     *
     * @param storeCode
     */
    @GetMapping("/api/order/statis/countNewest/{storeCode}")
    Response<Map<String, Long>> countNewestOrderByStoreCode(@PathVariable("storeCode") String storeCode);

    // ======  订单核销 相关接口 start   ===========>

    /**
     * 第二阶段：核销之前数据预校验,只有校验通过后才能进入核销页面
     *
     * @param orderDto
     */
    @PostMapping("/api/order/writeOff/validateBeforeWriteOff")
    Response<String> validateBeforeWriteOff(@RequestBody OrderWriteOffDto orderDto);

    /**
     * 第三阶段：真正提交核销 & 创建订单：（包括保存订单前数据校验,生成订单、保存订单）
     *
     * @param orderDto
     * @return 订单创建成功后，会返回订单对象 SaleOrder
     */
    @PostMapping("/api/order/writeOff/createOrder")
    Response<SaleOrder> createOrderWhenWriteOff(@RequestBody OrderWriteOffDto orderDto);

    /**
     * 更新订单状态 & 保存操作日志
     * 注意：此接口只针对单个订单的状态更新
     *
     * @param logList （orderCode 与 afterStatus 必须传入）
     */
    @PostMapping("/api/order/writeOff/updateOrderStatus")
    Response<String> updateStatusWithSaveLog(@RequestBody List<OrderLog> logList);

    /**
     * 第一步：在车主端 创建积分兑换服务订单
     *
     * @param orderDto
     * @return
     */
    @PostMapping("/api/order/centExchange/createCentExchangeOrder")
    Response<SaleOrder> createCentExchangeOrder(@RequestBody OrderCreateCentExchangeDto orderDto);

    /**
     * 第二步：在车主端 积分支付兑换服务订单
     *
     * @param payDto
     */
    @PostMapping("/api/order/centExchange/payCentExchangeOrder")
    Response payCentExchangeOrder(@RequestBody OrderCentPayDto payDto);

    // ======  订单评价 相关接口 start   ===========>

    /**
     * @param evaluate(传入参数：订单号、评价来源,评价人名,评价内容,评分,印象标签，图片，kpi)
     * @return 返回创建成功的评价对象
     */
    @PostMapping("/api/order/evaluate/create")
    Response<OrderEvaluate> createEvaluate(@RequestBody OrderEvaluate evaluate);

    /**
     * 获取评价信息
     *
     * @param orderCode     订单号
     * @param loadOtherFlag null或0: 不加载其他评价信息
     *                      1: 加载评价其他信息（标签、kpi、前端可见图片）
     *                      2: 加载评价其他信息（标签、kpi、全部图片）
     */
    @GetMapping("/api/order/evaluate/{orderCode}")
    Response<OrderEvaluate> getEvaluateByOrderCode(@PathVariable("orderCode") String orderCode,
                                                   @RequestParam(value = "loadOtherFlag", required = false) Integer loadOtherFlag);

    /**
     * 删除评价
     *
     * @param orderCode  订单号
     * @param updateUser 删除操作人
     * @return
     */
    @GetMapping("/api/order/evaluate/delete/{orderCode}")
    Response<String> deleteEvaluate(@PathVariable("orderCode") String orderCode,
                                    @RequestParam(value = "updateUser", required = false) String updateUser);

    /**
     * 精选/取消精选
     *
     * @param orderCode  订单号
     * @param updateUser 删除操作人
     * @return
     */
    @GetMapping("/api/order/evaluate/togglePerfect/{orderCode}")
    Response<String> toggleEvaluatePerfect(@PathVariable("orderCode") String orderCode,
                                           @RequestParam(value = "updateUser", required = false) String updateUser);

    /**
     * 设置 可见/ 不可见
     *
     * @param orderCode  订单号
     * @param updateUser 删除操作人
     * @return
     */
    @GetMapping("/api/order/evaluate/toggleVisibility/{orderCode}")
    Response<String> toggleEvaluateVisibility(@PathVariable("orderCode") String orderCode,
                                              @RequestParam(value = "updateUser", required = false) String updateUser);

    /**
     * 管理后台 搜索评价
     *
     * @param searchParamDto
     */
    @GetMapping("/api/order/evaluate/adminSearch")
    PageTable<OrderEvaluate> adminSearchEvaluation(@SpringQueryMap AdminSearchEvaluationParamDto searchParamDto);

    /**
     * 车主端小程序 搜索评价
     *
     * @param code            门店编号或 服务项目编号 等
     * @param statisWay       统计方式，按什么维度统计
     * @param visible         是否会员可见,null 表示不限制
     * @param perfect         是否精选，null表示不限制
     * @param impressionLabel 要查询的印象标签,空表示不限制
     * @param
     */
    @GetMapping("/api/order/evaluate/weappSearch")
    PageTable<OrderEvaluate> weappSearchEvaluation(@RequestParam(value = "code", required = false) String code,
                                                   @RequestParam(value = "statisWay", required = false) EvaluateStatisOption.EvaluateStatisWay statisWay,
                                                   @RequestParam(value = "visible", required = false) Boolean visible,
                                                   @RequestParam(value = "perfect", required = false) Boolean perfect,
                                                   @RequestParam(value = "impressionLabel", required = false) String impressionLabel,
                                                   @RequestParam(value = "limit", defaultValue = "20") int limit,
                                                   @RequestParam(value = "page", defaultValue = "1") int page);

    /**
     * 车主端小程序 搜索评价,统计评价数量
     *
     * @param code      门店编号或 服务项目编号 等
     * @param statisWay 统计方式，按什么维度统计
     * @param visible   是否会员可见,null 表示不限制
     * @param perfect   是否精选，null表示不限制
     */
    @GetMapping("/api/order/evaluate/weappSearchCount")
    Response<Integer> weappSearchCountEvaluation(@RequestParam(value = "code", required = false) String code,
                                                 @RequestParam(value = "statisWay", required = false) EvaluateStatisOption.EvaluateStatisWay statisWay,
                                                 @RequestParam(value = "visible", required = false) Boolean visible,
                                                 @RequestParam(value = "perfect", required = false) Boolean perfect,
                                                 @RequestParam(value = "impressionLabel", required = false) String impressionLabel);

    // 评价统计相关
    @PostMapping("/api/order/evaluate/statis/summarize")
    Response<Map<String, OrderEvaluateSummaryStatistic>> summarizeEvaluation(@RequestBody List<EvaluateStatisOption> statisOptionList);

    //电子加油券 相关接口 start   ===========>

    /**
     * 加油电子券下订单
     *
     * @return 创建成功时返回订单对象
     */
    @PostMapping("/api/order/oil/createOrder")
    Response<SaleOrder> createOilTicketOrder(@RequestBody OilOrderCreateDto orderDto);

    /**
     * 第一步：代金券+现金  加油电子券下订单
     *
     * @param orderDto
     */
    @PostMapping("/api/order/oil/createWithCashOrder")
    Response<SaleOrder> createWithCashOilTicketOrder(@RequestBody OilOrderWithCashCreateDto orderDto);

    /**
     * 使用电子券支付 加油电子券订单
     *
     * @return 支付成功时返回OilOrderPayedResultDto对象, 对象中每个属性都有值
     */
    @PostMapping("/api/order/oil/payOrder")
    Response<OilOrderPayedResultDto> payOilTicketOrder(@RequestBody OilOrderPayDto payDto);

    /**
     * 使用现金支付 加油订单
     *
     * @param payDto
     * @return
     */
    @PostMapping("/api/order/oil/cashPayOrder")
    Response payOilTickOrderByCash(@RequestBody OrderCashPayDto payDto);

    /**
     * 当系统5次自动尝试以后还是不能创建油卡交易单，则可以等油卡系统恢复正常后，手工尝试再去创建油卡订单
     *
     * @param orderCode 车服订单号
     */
    @GetMapping("/api/order/oil/manualCreateOilTrade/{orderCode}")
    Response<String> manualIssueCreateOilTrade(@PathVariable("orderCode") String orderCode);

    /**
     * 当油卡系统充值交易订单创建成功后,调用此方法更新此系统状态
     *
     * @param orderCode 车服订单号
     * @param tradeNo   油卡系统充值交易号
     * @return
     */
    @PostMapping("/api/order/oil/deposit/updateDepositOrder")
    Response<String> updateDepositWhenOilCreatedOrder(@RequestParam("orderCode") String orderCode,
                                                      @RequestParam("tradeNo") String tradeNo);

    /**
     * 查找待补发尝试的交易单
     *
     * @return
     */
    @GetMapping("/api/order/oil/deposit/findPendingRetryDeposit")
    Response<List<OrderOilDeposit>> findPendingRetryDeposit();

    /**
     * 获得充值单详情
     *
     * @param tradeNo       交易号
     * @param isLoadRecords 是否加载充值记录流水
     * @return
     */
    @GetMapping("/api/order/oil/deposit/getDepositOrderDetail/{tradeNo}")
    Response<OrderOilDeposit> getDepositOrderDetailByTradeNo(@PathVariable("tradeNo") String tradeNo,
                                                             @RequestParam(value = "isLoadRecords", required = false) boolean isLoadRecords);

    /**
     * 更新尝试次数和失败原因
     *
     * @param deposit
     * @return
     */
    @PostMapping("/api/order/oil/deposit/updateRetryTimes")
    Response<String> updateRetryTimesAndReason(@RequestBody OrderOilDeposit deposit);

    /**
     * 加油电子券 兑换到账详情
     *
     * @param userCouponCode
     */
    @GetMapping("/api/order/oil/deposit/exchangeDetail/{userCouponCode}")
    Response<OilTicketExchangeDetailDto> getExchangeDetail(@PathVariable("userCouponCode") String userCouponCode);

    /**
     * 保存成功充值的 充值记录
     *
     * @param depositRecord 充值记录
     * @return 返回成功处理的交易流水号bizNo和状态码
     * （1）普通处理成功返回状态码： ResponseCode.SUCCEED
     * （2）充值完成处理成功返回状态码：OrderOilDeposit.DEPOSIT_FINISHED_RESP_CODE
     */
    @PostMapping("/api/order/oil/depositRecord/save")
    Response<OilDepositRecord> saveDepositRecord(@RequestBody OilDepositRecord depositRecord);

    /**
     * 油卡订单：手工补发 券使用情况消息
     *
     * @param orderCode
     */
    @GetMapping("/api/order/oil/depositRecord/manualResendExchangeMsg/{orderCode}")
    Response<String> manualResendExchangeMsg(@PathVariable("orderCode") String orderCode);

    /**
     * 车主端小程序 搜索充值记录,按照时间倒序排列
     *
     * @param userCode      会员编号
     * @param orderCode     车服订单号
     * @param depositMobile 充值手机号
     */
    @GetMapping("/api/order/oil/depositRecord/weappSearch")
    PageTable<OilDepositRecord> weappSearchDepositRecord(@RequestParam(value = "userCode", required = true) String userCode,
                                                         @RequestParam(value = "orderCode", required = false) String orderCode,
                                                         @RequestParam(value = "depositMobile", required = false) String depositMobile,
                                                         @RequestParam(value = "limit", defaultValue = "20") int limit,
                                                         @RequestParam(value = "page", defaultValue = "1") int page);

    /**
     * 车主端小程序 合计充值到账金额
     *
     * @param userCode      会员编号
     * @param orderCode     车服订单号
     * @param depositMobile 充值手机号
     */
    @GetMapping("/api/order/oil/depositRecord/summarize")
    Response<DepositSummaryStatistic> summarizeDeposit(@RequestParam(value = "userCode", required = true) String userCode,
                                                       @RequestParam(value = "orderCode", required = false) String orderCode,
                                                       @RequestParam(value = "depositMobile", required = false) String depositMobile);

    /**
     * 后台管理->电子劵订单搜索
     *
     * @param searchParamDto
     */
    @GetMapping("/api/order/oil/adminSearch")
    PageTable<SaleOrder> adminOilOrderSearch(@SpringQueryMap OilAdminSearchParamDto searchParamDto);

    /**
     * 后台管理->电子劵订单支付信息信息
     *
     * @param orderCode
     * @return
     */
    @GetMapping("/api/order/oil/getOrderOilPaymentInfo/{orderCode}")
    Response<OrderPayment> getOrderOilPaymentInfo(@PathVariable("orderCode") String orderCode);

    /**
     * 后台管理->电子劵订单支付信息信息
     *
     * @param orderCode
     * @return
     */
    @GetMapping("/api/order/oil/getOrderOilPaymentLogList/{orderCode}")
    Response<OrderPayment> getOrderOilPaymentLogList(@PathVariable("orderCode") String orderCode);

    /**
     * 后台管理->电子劵订单充值记录列表
     *
     * @param orderCode
     * @return
     */
    @GetMapping("/api/order/oil/getOilDepositRecordList/{orderCode}")
    Response<List<OilDepositRecord>> getOilDepositRecordList(@PathVariable("orderCode") String orderCode);

    /**
     * 后台管理-支付事项记录列表
     *
     * @param orderCode
     * @return
     */
    @GetMapping("/api/order/oil/getPayDetailList/{orderCode}")
    Response<List<OrderPayDetailDto>> getPayDetailList(@PathVariable("orderCode") String orderCode);

    //预约订单 相关接口 start   ===========>

    /**
     * 统计门店服务项目在各时间段已预约订单数量
     *
     * @param storeServiceCode                门店服务项目编号
     * @param timeRangeList，时间范围格式：2010-11-12 09:01-11:30
     */
    @GetMapping("/api/order/appoint/countAppintNum")
    Response<Map<String, Integer>> countAppintNum(@RequestParam("storeServiceCode") String storeServiceCode,
                                                  @RequestParam("timeRangeList") List<String> timeRangeList);

    /**
     * 获取已预约的数量
     *
     * @param storeServiceCode 门店服务编号
     * @param startTime        预约开始时间 时间格式：YYYY-MM-dd HH:mm:ss
     * @param endTime          预约结束时间 时间格式：YYYY-MM-dd HH:mm:ss
     */
    @GetMapping("/api/order/appoint/getAppointedNum")
    Response<Integer> getAppointedNum(@RequestParam("storeServiceCode") String storeServiceCode,
                                      @RequestParam("startTime") String startTime,
                                      @RequestParam("endTime") String endTime);

    /**
     * 更新预约订单的会员账号信息 （当绑定车主端触发）
     *
     * @param userCode   会员编号
     * @param openId     会员Openid
     * @param userMobile 会员手机号
     */
    @GetMapping("/api/order/appoint/updateAppointUserInfo")
    Response<String> updateUserInfoByAppoitMobileWhenBound(@RequestParam("userCode") String userCode,
                                                           @RequestParam("openId") String openId,
                                                           @RequestParam("userMobile") String userMobile,
                                                           @RequestParam("miniAppId") String miniAppId);

    /**
     * 创建预约订单
     *
     * @param orderDto
     * @return
     */
    @PostMapping("/api/order/appoint/createAppoint")
    Response<String> createAppointOrder(@RequestBody CreateAppointOrderDto orderDto);

    /**
     * 预约订单核销之前 准备数据
     *
     * @param thirdWriteOffCode 第三方订单核销码
     */
    @GetMapping("/api/order/appoint/prepareDataBeforeWriteOff/{thirdWriteOffCode}")
    Response<AppointWriteOffPreparedDataDto> prepareDataBeforeWriteOffAppointment(@PathVariable("thirdWriteOffCode") String thirdWriteOffCode);

    /**
     * 核销预约订单
     *
     * @param writeOffDto
     * @return
     */
    @PostMapping("/api/order/appoint/doWriteOff")
    Response<SaleOrder> writeOffAppointOrder(@RequestBody AppointWriteOffDto writeOffDto);

    /**
     * 内部取消订单：【车主端/管理端】取消订单
     *
     * @param orderCode  roco车服订单号
     * @param actionUser 操作人
     * @param channel    操作渠道 {@link com.rocogz.syy.order.constant.OrderConstant.DictChannel}
     */
    @GetMapping("/api/order/appoint/internelCancelOrder/{orderCode}")
    Response<String> internelCancelAppointOrder(@PathVariable("orderCode") String orderCode,
                                                @RequestParam(value = "actionUser", required = false) String actionUser,
                                                @RequestParam(value = "channel") String channel);

    /**
     * 外部调用取消订单接口：例如：车务平台发起的取消订单
     *
     * @param thirdOrderCode    人保车务平台订单号
     * @param thirdWriteOffCode 人保车务平台的订单核销码
     */
    @GetMapping("/api/order/appoint/externelCancelOrder/{thirdOrderCode}")
    Response<String> externelCancelAppointOrder(@PathVariable("thirdOrderCode") String thirdOrderCode, @RequestParam("thirdWriteOffCode") String thirdWriteOffCode);

    /**
     * 后台搜索 预约订单
     *
     * @param searchParamDto
     * @return
     */
    @GetMapping("/api/order/appoint/adminSearch")
    PageTable<SaleOrder> adminSearchAppointOrder(@SpringQueryMap AppointAdminSearchParamDto searchParamDto);

    //违章订单相关接口 start   ===========>

    /**
     * 第一步：勾选违章记录 创建违章订单
     *
     * @return 创建成功时返回订单对象
     */
    @PostMapping("/api/order/peccancy/createPeccancyOrder")
    Response<SaleOrder> createPeccancyOrder(@RequestBody CreatePeccancyOrderDto orderDto);

    /**
     * 第二步：现金支付违章订单
     */
    @PostMapping("/api/order/peccancy/cashPayOrder")
    Response payPeccancyOrderByCash(@RequestBody OrderCashPayDto payDto);

    /**
     * 更新违章订单信息
     *
     * @param peccancy
     */
    @PostMapping("/api/order/peccancy/update")
    Response<String> updatePeccancyByOrderCode(@RequestBody OrderPeccancyOrder peccancy);

    /**
     * 查询所有待补偿创建鼎聚订单 违章订单,放入到补发延迟队列（在消息中心消费端启动服务时使用）
     *
     * @return
     */
    @GetMapping("/api/order/peccancy/findPendingRetry")
    Response<List<OrderPeccancyOrder>> findPendingRetryPeccancyOrderList();

    /**
     * 后台管理-违章订单搜索queryBySelfConditions
     *
     * @param searchParamDto
     */
    @GetMapping("/api/order/peccancy/adminSearch")
    PageTable<SaleOrder> adminSearchPeccancyOrder(@SpringQueryMap PeccancyAdminSearchParamDto searchParamDto);

    /**
     * 当系统5次自动尝试以后还是不能创建鼎聚订单，则可以等鼎聚系统正常后，手工尝试再去创建鼎聚订单
     *
     * @param orderCode 车服订单号
     */
    @GetMapping("/api/order/peccancy/manualCreateDingJuOrder/{orderCode}")
    Response<String> manualIssueCreateDingJuOrder(@PathVariable("orderCode") String orderCode);



    //服务调用 跟踪   相关接口 start   ===========>

    @GetMapping("/api/order/trace/error")
    PageTable<OrderErrorTrace> searchErrorTrace(@RequestParam(value = "coreCode", required = false) String coreCode,
                                                @RequestParam(value = "bussType", required = false) BussiTypeEnum bussType,
                                                @RequestParam(value = "callApi", required = false) CallApiEnum callApi,
                                                @RequestParam(value = "system", required = false) BussiSystemEnum system,
                                                @RequestParam(value = "createStartTime", required = false) LocalDateTime createStartTime,
                                                @RequestParam(value = "createEndTime", required = false) LocalDateTime createEndTime,
                                                @RequestParam(value = "limit", defaultValue = "20") int limit,
                                                @RequestParam(value = "page", defaultValue = "1") int page);

    @PostMapping("/api/order/trace/error/save")
    Response<String> saveErrorTrace(@RequestBody OrderErrorTrace callTrace);

    @GetMapping("/api/order/trace/thirdApi")
    PageTable<OrderThirdTrace> searchThirdCallTrace(@RequestParam(value = "coreCode", required = false) String coreCode,
                                                    @RequestParam(value = "callApiPath", required = false) String callApiPath,
                                                    @RequestParam(value = "sourcePlatform", required = false) String sourcePlatform,
                                                    @RequestParam(value = "targetPlatform", required = false) String targetPlatform,
                                                    @RequestParam(value = "createStartTime", required = false) LocalDateTime createStartTime,
                                                    @RequestParam(value = "createEndTime", required = false) LocalDateTime createEndTime,
                                                    @RequestParam(value = "limit", defaultValue = "20") int limit,
                                                    @RequestParam(value = "page", defaultValue = "1") int page);

    @PostMapping("/api/order/trace/thirdApi/save")
    Response<String> saveThirdCallTrace(@RequestBody OrderThirdTrace callTrace);

    /**
     * 主要是配合在折线图中会选择查询的接口，需要进行去重操作，只返回，callApiPath
     *
     * @param callApiPath
     * @param startTime   必填写
     * @param endTime     必填写
     * @param page
     * @param limit
     * @return
     */
    @GetMapping("/api/order/trace/thirdApi/pageQueryDistinctApiRequestInterface")
    PageTable<OrderThirdTrace> pageQueryDistinctApiRequestInterface(@RequestParam(value = "callApiPath", required = false) String callApiPath,
                                                                    @RequestParam(value = "startTime") LocalDateTime startTime,
                                                                    @RequestParam(value = "endTime") LocalDateTime endTime,
                                                                    @RequestParam(value = "page", defaultValue = "1") Integer page,
                                                                    @RequestParam(value = "limit", defaultValue = "20") Integer limit);

    /**
     * 只返回callStartTime，spendMillis,配合折线图组装数据
     *
     * @param callApiPath 类名
     * @param startTime   查询开始时间
     * @param endTime     查询结束时间
     * @return
     */
    @GetMapping("/api/order/trace/thirdApi/getStartTimeAndUsedTimeMillisList")
    Response<List<OrderThirdTrace>> getStartTimeAndUsedTimeMillisList(@RequestParam(value = "callApiPath", required = false) String callApiPath,
                                                                      @RequestParam(value = "startTime") LocalDateTime startTime,
                                                                      @RequestParam(value = "endTime") LocalDateTime endTime);

    /**
     * 返回耗时最大的10个接口，和平均耗时最大的10个接口
     *
     * @param startTime 查询开始时间，以startTime作为对比参考
     * @param endTime   查询结束时间，以startTime作为对比参考
     * @return
     */
    @GetMapping("/api/order/trace/thirdApi/statisticsMaxAndAverageUserTime")
    Response<ApiRequestLogStatisticsResponseDto> statisticsMaxAndAverageUserTimeApiList(@RequestParam(value = "startTime") LocalDateTime startTime,
                                                                                        @RequestParam(value = "endTime") LocalDateTime endTime);
    //微信支付记录 相关接口 start   ===========>

    /**
     * Query wx pay record by roco trade no and status response.
     *
     * @param rocoTradeNo  the roco trade no
     * @param wxTradeState the wx trade state
     * @return the response
     */
    @GetMapping("/api/order/wx-pay-record/queryByRocoTradeNoAndStatus")
    Response<WxPayRecord> queryWxPayRecordByRocoTradeNoAndStatus(@RequestParam("rocoTradeNo") String rocoTradeNo,
                                                                 @RequestParam(value = "wxTradeState", required = false) String wxTradeState);

    /**
     * Query wx pay record by flow code and status response.
     *
     * @param flowCode     the flow code
     * @param wxTradeState the wx trade state
     * @return the response
     */
    @GetMapping("/api/order/wx-pay-record/queryByFlowCodeAndStatus")
    Response<WxPayRecord> queryWxPayRecordByFlowCodeAndStatus(@RequestParam("flowCode") String flowCode,
                                                              @RequestParam(value = "wxTradeState", required = false) String wxTradeState);
    /**
     * Update wx pay record by id response.
     *
     * @param wxPayRecord the wx pay record
     * @return the response
     */
    @PostMapping("/api/order/wx-pay-record/updateById")
    Response updateWxPayRecordById(@RequestBody WxPayRecord wxPayRecord);

    /**
     * 后台管理订单支付记录分页查询列表  yinshihao加
     *
     * @param rocoTradeNo
     * @param wxTransactionId
     * @param wxUserOpenid
     * @param rocoResCode
     * @param rocoTradeType
     * @param wxTradeState
     * @param createTimeStart
     * @param createTimeEnd
     * @param wxTimeEndStart
     * @param wxTimeEndEnd
     * @param limit
     * @param page
     * @return
     */
    @GetMapping("/api/order/wx-pay-record/queryPage")
    PageTable<WxPayRecord> wxPayRecordPageTable(@RequestParam(value = "rocoTradeNo", required = false) String rocoTradeNo,
                                                @RequestParam(value = "wxTransactionId", required = false) String wxTransactionId,
                                                @RequestParam(value = "wxUserOpenid", required = false) String wxUserOpenid,
                                                @RequestParam(value = "rocoResCode", required = false) String rocoResCode,
                                                @RequestParam(value = "rocoTradeType", required = false) String rocoTradeType,
                                                @RequestParam(value = "wxTradeState", required = false) String wxTradeState,
                                                @RequestParam(value = "createTimeStart", required = false) String createTimeStart,
                                                @RequestParam(value = "createTimeEnd", required = false) String createTimeEnd,
                                                @RequestParam(value = "wxTimeEndStart", required = false) String wxTimeEndStart,
                                                @RequestParam(value = "wxTimeEndEnd", required = false) String wxTimeEndEnd,
                                                @RequestParam(value = "limit", defaultValue = "20") int limit,
                                                @RequestParam(value = "page", defaultValue = "1") int page);

    /**
     * 后台管理导出订单支付记录  yinshihao加
     *
     * @param rocoTradeNo
     * @param wxTransactionId
     * @param wxUserOpenid
     * @param rocoResCode
     * @param rocoTradeType
     * @param wxTradeState
     * @param createTimeStart
     * @param createTimeEnd
     * @param wxTimeEndStart
     * @param wxTimeEndEnd
     * @return
     */
    @GetMapping("/api/order/wx-pay-record/wxPayRecordList")
    Response<List<WxPayRecord>> wxPayRecordList(@RequestParam(value = "rocoTradeNo", required = false) String rocoTradeNo,
                                                @RequestParam(value = "wxTransactionId", required = false) String wxTransactionId,
                                                @RequestParam(value = "wxUserOpenid", required = false) String wxUserOpenid,
                                                @RequestParam(value = "rocoResCode", required = false) String rocoResCode,
                                                @RequestParam(value = "rocoTradeType", required = false) String rocoTradeType,
                                                @RequestParam(value = "wxTradeState", required = false) String wxTradeState,
                                                @RequestParam(value = "createTimeStart", required = false) String createTimeStart,
                                                @RequestParam(value = "createTimeEnd", required = false) String createTimeEnd,
                                                @RequestParam(value = "wxTimeEndStart", required = false) String wxTimeEndStart,
                                                @RequestParam(value = "wxTimeEndEnd", required = false) String wxTimeEndEnd);

    /**
     * 后台根据订单编码查询微信支付及退款详情 yinshihao加
     *
     * @param rocoTradeNo
     * @return
     */
    @GetMapping("/api/order/wx-pay-record/getWxPayRecordByTradeNo")
    Response<WxPayRecord> getWxPayRecordByTradeNo(@RequestParam(value = "rocoTradeNo") String rocoTradeNo);

    /**
     * 后台根据订单编码查询微信支付流水 yinshihao加
     *
     * @param businessCode
     * @return
     */
    @GetMapping("/api/order/wx-pay-record/getOrderWxPayPaymentFlow")
    Response<OrderWxPayPaymentFlow> getOrderWxPayPaymentFlow(@RequestParam(value = "businessCode") String businessCode);
    //微信支付记录 相关接口 end   ===========>

    //微信支付API 相关接口 start   ===========>

    /**
     * 统一下单(详见https://pay.weixin.qq.com/wiki/doc/api/jsapi.php?chapter=9_1)
     * 在发起微信支付前，需要调用统一下单接口，获取"预支付交易会话标识"
     * 接口地址：https://api.mch.weixin.qq.com/pay/unifiedorder
     *
     * @param paramDto the param dto
     * @return the response
     */
    @PostMapping("/api/order/pay/wx/unifiedOrder")
    Response<WxPayUnifiedOrderResultDto> unifiedWxPayOrder(@RequestBody WxPayUnifiedOrderParamDto paramDto);

    /**
     * 微信支付下单成功后未立即支付，再次调起支付接口
     *
     * @param outTradeNo roco车服的订单编号
     * @return response
     */
    @GetMapping("/api/order/pay/wx/repayOrder")
    Response<WxPayUnifiedOrderResultDto> repayWxPayOrder(@RequestParam("outTradeNo") String outTradeNo);

    /**
     * <pre>
     * 关闭订单
     * 应用场景
     * 以下情况需要调用关单接口：
     * 1. 商户订单支付失败需要生成新单号重新发起支付，要对原订单号调用关单，避免重复支付；
     * 2. 系统下单后，用户支付超时，系统退出不再受理，避免用户继续，请调用关单接口。
     * 注意：订单生成后不能马上调用关单接口，最短调用时间间隔为5分钟。
     * 接口地址：https://api.mch.weixin.qq.com/pay/closeorder
     * 是否需要证书：   不需要。
     * </pre>
     *
     * @param outTradeNo 商户系统内部的订单号
     * @return the response
     */
    @GetMapping("/api/order/pay/wx/closeOrder/{outTradeNo}")
    Response closeWxPayOrder(@PathVariable("outTradeNo") String outTradeNo);

    /**
     * <pre>
     * 微信支付-申请退款
     * 详见 https://pay.weixin.qq.com/wiki/doc/api/jsapi.php?chapter=9_4
     * 接口链接：https://api.mch.weixin.qq.com/secapi/pay/refund
     * </pre>
     *
     * @param refundParam 请求对象
     * @return the response
     */
    @PostMapping("/api/order/pay/wx/refund")
    Response<String> refund(@RequestBody WxPayRefundParamDto refundParam);

    //微信支付API 相关接口 end   ===========>

    //微信支付退款记录 相关接口 start   ===========>

    /**
     * Query wx pay refund record by roco refund no response.
     *
     * @param rocoRefundNo the roco refund no
     * @return the response
     */
    @GetMapping("/api/order/wx-pay-refund-record/queryByRocoRefundNo")
    Response<WxPayRefundRecord> queryWxPayRefundRecordByRocoRefundNo(@RequestParam("rocoRefundNo") String rocoRefundNo);

    /**
     * Update wx pay refund record by id response.
     *
     * @param wxPayRefundRecord the wx pay refund record
     * @return the response
     */
    @PostMapping("/api/order/wx-pay-refund-record/updateById")
    Response updateWxPayRefundRecordById(@RequestBody WxPayRefundRecord wxPayRefundRecord);

    /**
     * 后台管理订单支付退款记录分页查询列表  yinshihao加
     *
     * @param rocoTradeNo
     * @param rocoRefundNo
     * @param wxRefundId
     * @param wxTransactionId
     * @param wxUserOpenid
     * @param rocoResCode
     * @param rocoTradeType
     * @param wxRefundStatus
     * @param wxRefundRequestSource
     * @param createTimeStart
     * @param createTimeEnd
     * @param wxRefundSuccessTimeStart
     * @param wxRefundSuccessTimeEnd
     * @param limit
     * @param page
     * @return
     */
    @GetMapping("/api/order/wx-pay-refund-record/queryPage")
    PageTable<WxPayRefundRecord> wxPayRefundRecordPageTable(@RequestParam(value = "rocoTradeNo", required = false) String rocoTradeNo,
                                                            @RequestParam(value = "rocoRefundNo", required = false) String rocoRefundNo,
                                                            @RequestParam(value = "wxRefundId", required = false) String wxRefundId,
                                                            @RequestParam(value = "wxTransactionId", required = false) String wxTransactionId,
                                                            @RequestParam(value = "wxUserOpenid", required = false) String wxUserOpenid,
                                                            @RequestParam(value = "rocoResCode", required = false) String rocoResCode,
                                                            @RequestParam(value = "rocoTradeType", required = false) String rocoTradeType,
                                                            @RequestParam(value = "wxRefundStatus", required = false) String wxRefundStatus,
                                                            @RequestParam(value = "wxRefundRequestSource", required = false) String wxRefundRequestSource,
                                                            @RequestParam(value = "createTimeStart", required = false) String createTimeStart,
                                                            @RequestParam(value = "createTimeEnd", required = false) String createTimeEnd,
                                                            @RequestParam(value = "wxRefundSuccessTimeStart", required = false) String wxRefundSuccessTimeStart,
                                                            @RequestParam(value = "wxRefundSuccessTimeEnd", required = false) String wxRefundSuccessTimeEnd,
                                                            @RequestParam(value = "limit", defaultValue = "20") int limit,
                                                            @RequestParam(value = "page", defaultValue = "1") int page);

    /**
     * 后台管理导出订单支付退款记录列表  yinshihao加
     *
     * @param rocoTradeNo
     * @param rocoRefundNo
     * @param wxRefundId
     * @param wxTransactionId
     * @param wxUserOpenid
     * @param rocoResCode
     * @param rocoTradeType
     * @param wxRefundStatus
     * @param wxRefundRequestSource
     * @param createTimeStart
     * @param createTimeEnd
     * @param wxRefundSuccessTimeStart
     * @param wxRefundSuccessTimeEnd
     * @return
     */
    @GetMapping("/api/order/wx-pay-refund-record/wxPayRefundRecordList")
    Response<List<WxPayRefundRecord>> wxPayRefundRecordList(@RequestParam(value = "rocoTradeNo", required = false) String rocoTradeNo,
                                                            @RequestParam(value = "rocoRefundNo", required = false) String rocoRefundNo,
                                                            @RequestParam(value = "wxRefundId", required = false) String wxRefundId,
                                                            @RequestParam(value = "wxTransactionId", required = false) String wxTransactionId,
                                                            @RequestParam(value = "wxUserOpenid", required = false) String wxUserOpenid,
                                                            @RequestParam(value = "rocoResCode", required = false) String rocoResCode,
                                                            @RequestParam(value = "rocoTradeType", required = false) String rocoTradeType,
                                                            @RequestParam(value = "wxRefundStatus", required = false) String wxRefundStatus,
                                                            @RequestParam(value = "wxRefundRequestSource", required = false) String wxRefundRequestSource,
                                                            @RequestParam(value = "createTimeStart", required = false) String createTimeStart,
                                                            @RequestParam(value = "createTimeEnd", required = false) String createTimeEnd,
                                                            @RequestParam(value = "wxRefundSuccessTimeStart", required = false) String wxRefundSuccessTimeStart,
                                                            @RequestParam(value = "wxRefundSuccessTimeEnd", required = false) String wxRefundSuccessTimeEnd);

    //微信支付退款记录 相关接口 end   ===========>

    // ***************************** 违章记录查询 liangyongtong start

    /**
     * 查询最近的一次查询记录
     *
     * @param query 查询条件
     * @return
     */
    @PostMapping("/api/order/peccancy/orderPeccancyQueryRecord/lastest/query")
    Response<OrderPeccancyQueryRecord> lastestQuery(@RequestBody OrderPeccancyQueryRecord query);

    /**
     * 新增或更新最近的一次成功查询记录
     *
     * @param record 待保存记录
     * @return
     */
    @PostMapping("/api/order/peccancy/orderPeccancyQueryRecord/updatelastesttime")
    Response peccancyQueryRecordUpdateLastestTime(@RequestBody OrderPeccancyQueryRecord record);

    /**
     * 更新或者新增违章记录
     *
     * @param records
     * @return
     */
    @PostMapping("/api/order/peccancy/orderPeccancyRecord/list")
    Response peccancyRecord(@RequestBody List<OrderPeccancyRecord> records);

    /**
     * 查询用户下的查询记录
     *
     * @param userCode 会员编码
     * @return
     */
    @GetMapping("/api/order/peccancy/orderPeccancyQueryRecord/records")
    Response<List<OrderPeccancyQueryRecord>> queryPeccancyQueryRecord(@RequestParam(value = "userCode") String userCode);

    /**
     * 查询已存储的违章记录
     *
     * @param query 查询条件
     * @param type  查询类型  未处理  已处理
     * @return
     */
    @PostMapping("/api/order/peccancy/orderPeccancyRecord/peccancyRecordByQueryType")
    Response<List<OrderPeccancyRecord>> queryPeccancyRecord(@RequestBody OrderPeccancyRecord query, @RequestParam(value = "type", required = false) String type);

    @GetMapping("/api/order/peccancy/orderPeccancyRecord/queryByPeccancyIdList")
    Response<List<OrderPeccancyRecord>> queryPeccancyRecordByIdList(@RequestParam("peccancyIdList") List<String> peccancyIdList);

    /**
     * 查询违章记录信息 车牌号
     *
     * @param conditions
     * @return
     */
    @PostMapping("/api/order/peccancy/orderPeccancyRecord/queryBySelfConditions")
    Response<List<OrderPeccancyRecord>> queryPeccancyRecordBySelfConditions(@RequestBody QueryPeccancyConditionsDto conditions);

    /**
     * 统计一下违章记录总数
     *
     * @param query 查询条件
     * @return
     */
    @PostMapping("/api/order/peccancy/orderPeccancyRecord/queryPeccancyRecordTotal")
    Response<PeccancyRecordTotalDto> queryPeccancyRecordTotal(@RequestBody OrderPeccancyRecord query);

    // ***************************** 违章记录查询 liangyongtong end
    // ***************************** 提现记录 ranxiangfu start

    /**
     * 创建提现订单
     *
     * @param orderDto
     * @return
     */
    @PostMapping("/api/order/withdrawals-record/createWithdrawalsOrder")
    Response<SaleOrder> createWithdrawalsOrder(@RequestBody CreateWithdrawalsOrderDto orderDto);

    /**
     * 根据订单编码查询提现订单信息.
     *
     * @param orderCode the order code
     * @return the withdrawals record by order code
     */
    @GetMapping("/api/order/withdrawals-record/getByOrderCode")
    Response<WithdrawalsRecord> getWithdrawalsRecordByOrderCode(@RequestParam("orderCode") String orderCode);

    /**
     * 保存提现订单支付信息
     *
     * @param paymentInfoDto the payment info dto
     * @return the response
     */
    @PostMapping("/api/order/withdrawals-record/savePaymentInfo")
    Response saveWithdrawalsPaymentInfo(@Validated @RequestBody OrderCentPayDto paymentInfoDto);

    /**
     * 提现汇总统计
     *
     * @param userCode
     * @return
     */
    @GetMapping("/api/order/withdrawals-record/summary")
    Response<WithdrawalsSummaryResultDto> getWithdrawalsSummary(@RequestParam("userCode") String userCode,
                                                                @RequestParam("pointType") String pointType);

    /**
     * 统计提现订单中已使用的虚拟货币总金额
     *
     * @param userCode  the user code
     * @param startTime the start time
     * @param endTime   the end time
     * @return withdrawals summary
     */
    @GetMapping("/api/order/withdrawals-record/sumUsedVirtualCurrencyAmount")
    Response<BigDecimal> sumWithdrawalsUsedVirtualCurrencyAmount(@RequestParam("userCode") String userCode,
                                                                 @RequestParam("pointType") String pointType,
                                                                 @RequestParam(value = "startTime", required = false) String startTime,
                                                                 @RequestParam(value = "endTime", required = false) String endTime);

    /**
     * 审核结果回调接口
     *
     * @param pass       the pass
     * @param code       the code
     * @param updateUser the update user
     * @param remark     the remark
     * @return withdrawals summary
     */
    @GetMapping("/api/order/withdrawals-record/approvementCallBack")
    Response withdrawalsApprovementCallBack(@RequestParam("pass") Boolean pass,
                                            @RequestParam("code") String code,
                                            @RequestParam("updateUser") String updateUser,
                                            @RequestParam(value = "remark", required = false) String remark);

    /**
     * 更改提现订单状态
     *
     * @param changeStatusDto the change status dto
     * @return res
     */
    @PostMapping("/api/order/withdrawals-record/changeStatus")
    Response changeWithdrawalsOrderStatus(@RequestBody WithdrawalsOrderChangeStatusDto changeStatusDto);

    /**
     * 搜索服务券订单
     *
     * @param searchParamDto
     * @return
     */
    @GetMapping("/api/order/withdrawals-record/adminSearch")
    PageTable<SaleOrder> adminSearchWithdrawalsOrder(@SpringQueryMap WithdrawalsAdminSearchParamDto searchParamDto);

    /**
     * 标记提现订单成功退回用户的兑换券
     *
     * @param orderCode the order code
     * @return withdrawals summary
     */
    @GetMapping("/api/order/withdrawals-record/successRollbackUserCoupon")
    Response setWithdrawalsOrderSuccessRollbackUserCoupon(@RequestParam("orderCode") String orderCode);

    /**
     * 发送退回用户兑换券的消息到MQ, 传入订单编号或订单编号列表, 已退回的订单不会重复再退
     *
     * @param paramDto the param dto
     * @return withdrawals summary
     */
    @PostMapping("/api/order/withdrawals-record/sendRollbackUserCouponMsg")
    Response sendWithdrawalsOrderRollbackUserCouponMsg(@RequestBody WithdrawalsOrderChangeStatusDto paramDto);

    // ***************************** 提现记录 ranxiangfu end

    // ***************************** 商品订单 ranxiangfu start

    /**
     * 搜索商品订单
     *
     * @param searchParamDto the search param dto
     * @return page table
     */
    @GetMapping("/api/order/goods-order/adminSearch")
    PageTable<SaleOrder> adminSearchGoodsOrder(@SpringQueryMap GoodsAdminSearchParamDto searchParamDto);

    /**
     * 创建商品订单
     *
     * @param orderDto the order dto
     * @return the response
     */
    @PostMapping("/api/order/goods-order/create")
    Response<SaleOrder> createGoodsOrder(@RequestBody CreateGoodsOrderDto orderDto);

    /**
     * 根据订单编码查询提现订单信息.
     *
     * @param orderCode the order code
     * @return the withdrawals record by order code
     */
    @GetMapping("/api/order/goods-order/getByOrderCode")
    Response<GoodsOrder> getGoodsOrderByOrderCode(@RequestParam("orderCode") String orderCode);

    /**
     * 保存商品订单支付信息
     *
     * @param paymentInfoDto the payment info dto
     * @return the response
     */
    @PostMapping("/api/order/goods-order/savePaymentInfo")
    Response saveGoodsOrderPaymentInfo(@RequestBody OrderCentPayDto paymentInfoDto);

    /**
     * 更改商品订单状态
     *
     * @param changeStatusDto the change status dto
     * @return withdrawals summary
     */
    @PostMapping("/api/order/goods-order/changeStatus")
    Response changeGoodsOrderStatus(@RequestBody GoodsOrderChangeStatusDto changeStatusDto);

    /**
     * 现金支付回调接口
     *
     * @param cashPayDto the cash pay dto
     * @return withdrawals summary
     */
    @PostMapping("/api/order/goods-order/cashPay")
    Response<String> payGoodsOrderByCash(@RequestBody OrderCashPayDto cashPayDto);

    /**
     * List goods order by order codes response.
     *
     * @param orderCodeList the order code list
     * @return the response
     */
    @GetMapping("/api/order/goods-order/listGoodsOrderByOrderCodes")
    Response<List<GoodsOrder>> listGoodsOrderByOrderCodes(@RequestParam("orderCodeList") List<String> orderCodeList);

    /**
     * 权益转赠时-领取商品订单
     *
     * @param receiveDto the receive dto
     * @return the response
     */
    @PostMapping("/api/order/goods-order/receive")
    Response<GoodsOrder> receiveGoodsOrder(@RequestBody GoodsOrderReceiveDto receiveDto);

    /**
     * 权益转赠时-更新商品订单收货地址
     *
     * @param address the address
     * @return the response
     */
    @PostMapping("/api/order/goods-order/receiveGoodsOrderWithAddress")
    Response receiveGoodsOrderWithAddress(@RequestBody GoodsOrderDeliveryAddress address);

    /**
     * List goods order express delivery info by bestow code response.
     *
     * @param bestowCode the bestow code
     * @return the response
     */
    @GetMapping("/api/order/goods-order/listGoodsOrderExpressDeliveryInfoByBestowCode")
    Response<List<GoodsOrderExpressDeliveryInfoDto>> listGoodsOrderExpressDeliveryInfoByBestowCode(@RequestParam("bestowCode") String bestowCode);

    /**
     * Agent gift bestow page query page table.
     *
     * @param createUserCode the create user code
     * @param status         the status
     * @param limit          the limit
     * @param page           the page
     * @return the page table
     */
    @GetMapping("/api/order/goods-order/agentGiftBestowPageQuery")
    PageTable<SaleOrder> agentGiftBestowPageQuery(@RequestParam(name = "createUserCode") String createUserCode,
                                                  @RequestParam(name = "status", required = false) String status,
                                                  @RequestParam(name = "limit", defaultValue = "10") Integer limit,
                                                  @RequestParam(name = "page", defaultValue = "1") Integer page);

    /**
     * 统计投保礼赠送商品订单数量
     *
     * @param orderCountParamDto   orderCountParamDto
     * @return the response
     */
    @PostMapping("/api/order/goods-order/agentGiftBestowOrderCount")
    Response<List<OrderCountResultDto>> agentGiftBestowOrderCount(@RequestBody OrderCountParamDto orderCountParamDto);
    /**
     * 统计当前下单机构,团队和代理商各有哪几种  yinshihao加  20201209
     *
     * @return the response
     */
    @GetMapping("/api/order/goods-order/findOrderTeamAndOrgAndAgent")
    Response<Map<String, List<OrderTeamAndOrgAndAgentDto>>> findOrderTeamAndOrgAndAgent();
    // ***************************** 商品订单 ranxiangfu end


    // ***************************** 商品订单赠送信息 ranxiangfu start
    /**
     * Gets goods order bestow info by order code.
     *
     * @param orderCode the order code
     * @return the goods order bestow info by order code
     */
    @GetMapping("/api/order/goods-order-bestow-info/getByOrderCode")
    GoodsOrderBestowInfo getGoodsOrderBestowInfoByOrderCode(@RequestParam(value = "orderCode") String orderCode);
    // ***************************** 商品订单赠送信息 ranxiangfu end


    // ***************************** 商品订单收货地址 ranxiangfu start

    /**
     * 保存或更新商品订单收货地址
     *
     * @param address the address
     * @return the response
     */
    @PostMapping("/api/order/goods-order-delivery-address/saveOrUpdate")
    Response saveOrUpdateGoodsOrderDeliveryAddress(@RequestBody GoodsOrderDeliveryAddress address);

    /**
     * 删除商品订单收货地址
     *
     * @param orderCode the order code
     * @return the response
     */
    @DeleteMapping("/api/order/goods-order-delivery-address/deleteByOrderCode")
    Response deleteGoodsOrderDeliveryAddressByOrderCode(@RequestParam("orderCode") String orderCode);

    /**
     * 获取商品订单收货地址
     * @param orderCode 订单编码
     * @return
     */
    @GetMapping("/api/order/goods-order-delivery-address/getByOrderCde")
    Response<GoodsOrderDeliveryAddress> getGoodsOrderDeliveryAddressByOrderCode(@RequestParam("orderCode") String orderCode);
    // ***************************** 商品订单收货地址 ranxiangfu end

    /*******************************************************liangyongtong 商户投保礼订单 start**************************************************/
    /**
     * 新增投保礼订单
     *
     * @param orderDto 投保礼数据
     * @return
     */
    @PostMapping("/api/merchant/insurancegift/shop/order")
    Response<Map<String, String>> createInsuranceGiftOrder(@RequestBody MerchantInsuranceGiftOrderDto orderDto);

    /**
     * 投保礼详情
     *
     * @param storeCode 商户编码
     * @param mainCode  主订单号
     * @param itemCode  二级订单号
     * @return
     */
    @GetMapping("/api/merchant/insurancegift/shop/order/gift/detail")
    Response<MerchantInsuranceGiftMainOrder> insuranceGiftMainOrderDetail(@RequestParam(value = "storeCode") String storeCode, @RequestParam(value = "mainCode") String mainCode, @RequestParam(value = "itemCode", required = false) String itemCode);

    /**
     * 列表查询投保礼列表
     *
     * @param orderType 订单类型
     * @param accountNumber 账号
     * @param keyword   查询关键字
     * @param status    状态
     * @param storeCode 商户编码
     * @param limit     每页条数
     * @param page      当前页数
     * @return
     */
    @GetMapping("/api/merchant/insurancegift/shop/order/gift/queryPage")
    PageTable<MerchantInsuranceGiftItemOrder> giftMainOrderPageTable(
            @RequestParam(value = "orderType", defaultValue = InsuranceGiftConstant.DictData.ORDER_TYPE_STORE) String orderType,
            @RequestParam(value = "accountNumber", required = false) String accountNumber,
            @RequestParam(value = "keyword", required = false) String keyword,
            @RequestParam(value = "status", required = false) String status,
            @RequestParam(value = "storeCode", required = false) String storeCode,
            @RequestParam(value = "limit", defaultValue = "10") int limit,
            @RequestParam(value = "page", defaultValue = "1") int page);

    /**
     * 列表查询投保礼列表
     *
     * @param orderType 订单类型
     * @param accountNumber 账号
     * @param keyword   查询关键字
     * @param handleStatus    状态
     * @param storeCode 商户编码
     * @param limit     每页条数
     * @param page      当前页数
     * @return
     */
    @GetMapping("/api/merchant/insurancegift/shop/order/gift/queryPageHandlerStatus")
    PageTable<MerchantInsuranceGiftItemOrder> giftItemOrderPageTable(
            @RequestParam(value = "orderType", defaultValue = InsuranceGiftConstant.DictData.ORDER_TYPE_STORE) String orderType,
            @RequestParam(value = "accountNumber", required = false) String accountNumber,
            @RequestParam(value = "keyword", required = false) String keyword,
            @RequestParam(value = "handleStatus", required = false) String handleStatus,
            @RequestParam(value = "storeCode", required = false) String storeCode,
            @RequestParam(value = "limit", defaultValue = "10") int limit,
            @RequestParam(value = "page", defaultValue = "1") int page);

    /**
     * 确认投保礼方案
     *
     * @param storeCode 商户编码
     * @param mainCode  主订单编码
     * @param itemCode  二级订单编码
     * @param giftCodes 选择的商品编码
     * @return
     */
    @GetMapping("/api/merchant/insurancegift/shop/order/gift/confirm")
    Response<List<GiftCancelDto>> confirmGiftOrder(@RequestParam(value = "storeCode", required = false) String storeCode, @RequestParam(value = "mainCode") String mainCode, @RequestParam(value = "itemCode") String itemCode, @RequestParam(value = "giftCodes", required = false) String[] giftCodes, @RequestParam(value = "confirmOpenid", required = false) String confirmOpenid);

    /**
     * 不同意投保礼方案
     * @param storeCode 商户编码
     * @param mainCode 主订单号
     * @param itemCode 二级订单编号
     */
    @GetMapping("/api/merchant/insurancegift/shop/order/gift/disagree")
    Response<Boolean> noConfirmGiftOrder(@RequestParam(value = "storeCode", required = false) String storeCode, @RequestParam(value = "mainCode") String mainCode, @RequestParam(value = "itemCode") String itemCode);

    /**
     * 报价方案支付
     *
     * @param orderPay
     * @return
     */
    @PostMapping("/api/merchant/insurancegift/shop/order/gift/pay")
    Response<WxPayUnifiedOrderResultDto> giftOrderPay(@RequestBody MerchantInsuranceGifOrderPayDto orderPay);

    /**
     * 投保礼二级详情
     *
     * @param itemCode 二级订单号
     * @return
     */
    @GetMapping("/api/merchant/insurancegift/shop/order/gift/secondDetail")
    Response<MerchantInsuranceGiftItemOrder> giftSecondDetail(@RequestParam(value = "itemCode") String itemCode);

    /**
     * 投保礼二级详情
     * @param bestowCode 二级订单号赠送号
     * @return
     */
    @GetMapping("/api/merchant/insurancegift/shop/order/gift/secondDetailByBestowCode")
    Response<MerchantInsuranceGiftItemOrder> giftSecondDetailBybestowCode(@RequestParam(value = "bestowCode") String bestowCode);

    /**
     * 投保礼订单支付结果
     *
     * @param payResult 结果数据
     * @return
     */
    @PostMapping("/api/merchant/insurancegift/shop/order/gift/payresult")
    Response<MerchantInsuranceGiftItemOrder> giftOrderPayResult(@RequestBody OrderCashPayDto payResult);

    /**
     * 投保礼礼品赠送
     *
     * @param itemCode         二级订单编码
     * @param storeAccountName 商户名称
     * @param bestowCode 赠送编码
     * @param onlyGoodSelfTheGift 只有商品且自提
     * @return
     */
    @GetMapping("/api/merchant/insurancegift/shop/order/gift/giving")
    Response getOrderGiving(@RequestParam(value = "itemCode") String itemCode, @RequestParam(value = "storeAccountName") String storeAccountName, @RequestParam(value = "bestowCode") String bestowCode, @RequestParam(value = "onlyGoodSelfTheGift", required = false) Boolean onlyGoodSelfTheGift);

    /**
     * 定时任务处理已确认订单过时未支付设置为作废
     * @return
     */
    @GetMapping("/api/merchant/insurancegift/shop/order/gift/taskforconfirm")
    Response<List<String>> taskToInvalidForConfirmOrder();

    /**
     * 撤销赠送
     * @param storeCode 商户编码
     * @param itemCode 二级订单
     * @return
     */
    @GetMapping("/api/merchant/insurancegift/shop/order/gift/giftrecall")
    Response<MerchantInsuranceGiftItemOrder> giftRecall(@RequestParam(value = "storeCode", required = false) String storeCode, @RequestParam(value = "itemCode") String itemCode, @RequestParam(value = "storeAccountName") String storeAccountName);

    /**
     * 投保礼领取成功
     * @param bestowCode 赠送编码
     * @return
     */
    @GetMapping("/api/merchant/insurancegift/shop/order/gift/giftOrderReceive")
    Response giftOrderReceive(@RequestParam(value = "bestowCode") String bestowCode, @RequestParam(value = "receiveUser", required = false) String receiveUser);

    /**
     * 获取每个订单的数量
     * @param storeCode 商户编码
     * @param accountNumber 创建账号
     * @return
     */
    @GetMapping("/api/merchant/insurancegift/shop/order/gift/countforstatus")
    Response<Map<String, Integer>> giftOrderStatistics(@RequestParam(value = "storeCode", required = false) String storeCode,
                                                       @RequestParam(required = false, value = "accountNumber") String accountNumber,
                                                       @RequestParam(required = false, value = "orderType") String orderType);
    /**
     * 获取每个处理状态的数量
     * @param storeCode 商户编码
     * @param accountNumber 创建账号
     * @return
     */
    @GetMapping("/api/merchant/insurancegift/shop/order/gift/countforhandlestatus")
    Response<Map<String, Integer>> giftOrderHandleStatusStatistics(@RequestParam(value = "storeCode", required = false) String storeCode,
                                                       @RequestParam(required = false, value = "accountNumber") String accountNumber,
                                                       @RequestParam(required = false, value = "orderType") String orderType);

    /**
     * 投保礼开单汇总
     * @param orderType 订单类型
     * @param storeCode 商户编码
     * @param accountNumber 账号
     * @return
     */
    @GetMapping("/api/merchant/insurancegift/shop/order/gift/giftSummary")
    Response<GiftSummaryDto> giftSummary(@RequestParam(value = "orderType") String orderType,
                                         @RequestParam(value = "storeCode", required = false) String storeCode,
                                         @RequestParam(value = "accountNumber", required = false) String accountNumber);

    /**
     * 成功列表查询
     * @param orderType 订单类型
     * @param accountNumber 账号
     * @param keyword 搜索关键字
     * @param status 状态 默认成功
     * @param storeCode 商户编码
     * @param month 月分
     * @param orderBy 排序类型
     * @param startBudget 开始预算
     * @param endBudget 结束预算
     * @param startCost 开始成本
     * @param endCost 结束成本
     * @param limit 每页条数
     * @param page 当前页
     * @return
     */
    @GetMapping("/api/merchant/insurancegift/shop/order/gift/queryPage/more")
    PageTable<MerchantInsuranceGiftItemOrder> giftMainOrderWxPageTable(
            @RequestParam(value = "orderType", defaultValue = InsuranceGiftConstant.DictData.ORDER_TYPE_MEMBER) String orderType,
            @RequestParam(value = "accountNumber", required = false) String accountNumber,
            @RequestParam(value = "keyword", required = false) String keyword,
            @RequestParam(value = "status", required = false, defaultValue = OrderConstant.DictOrderStatus.GIFT_COMPLETED) String status,
            @RequestParam(value = "storeCode", required = false) String storeCode,
            @RequestParam(value = "month", required = false) String month,
            @RequestParam(value = "orderBy", required = false) String orderBy,
            @RequestParam(value = "startBudget", required = false) Double startBudget,
            @RequestParam(value = "endBudget", required = false) Double endBudget,
            @RequestParam(value = "startCost", required = false) Double startCost,
            @RequestParam(value = "endCost", required = false) Double endCost,
            @RequestParam(value = "limit", defaultValue = "10") int limit,
            @RequestParam(value = "page", defaultValue = "1") int page);


    /**
     * 查询复用方案列表
     * @param orderType 订单类型
     * @param accountNumber 账号
     * @param keyword 搜索关键字
     * @param orderBy 排序类型: AI 智能 KARKETASC 市场从低到高  KARKETDESC 市场从高到低 COSTASC 成本从低到高 COSTDESC 成本从高到低
     * @param startMarket 开始预算
     * @param endMarket 结束预算
     * @param startCost 开始成本
     * @param endCost 结束成本
     * @param limit 每页条数
     * @param page 当前页
     * @return
     */
    @GetMapping("/api/merchant/insurancegift/shop/order/gift/plan/queryPage/more")
    PageTable<MerchantInsuranceGiftItemOrder> giftPlanItemOrderWxPageTable(
            @RequestParam(value = "orderType", defaultValue = InsuranceGiftConstant.DictData.ORDER_TYPE_MEMBER) String orderType,
            @RequestParam(value = "accountNumber", required = false) String accountNumber,
            @RequestParam(value = "planType", required = false) String planType,
            @RequestParam(value = "keyword", required = false) String keyword,
            @RequestParam(value = "orderBy", required = false) String orderBy,
            @RequestParam(value = "startMarket", required = false) Double startMarket,
            @RequestParam(value = "endMarket", required = false) Double endMarket,
            @RequestParam(value = "startCost", required = false) Double startCost,
            @RequestParam(value = "endCost", required = false) Double endCost,
            @RequestParam(value = "limit", defaultValue = "10") int limit,
            @RequestParam(value = "page", defaultValue = "1") int page);

    /**
     * 把成功方案加入复用方案
     * @param copyGift
     * @return
     */
    @PostMapping("/api/merchant/insurancegift/shop/order/gift/copy")
    Response copyPlanFromGift(@RequestBody CopyGiftDto copyGift);

    /**
     * 把成功方案加入复用方案
     * @param copyGift
     * @return
     */
    @GetMapping("/api/merchant/insurancegift/shop/order/gift/tovoid")
    Response giftToVoid(@RequestParam(value = "itemCode") String itemCode);

    /**
     * 增加复制次数
     * @param itemCode
     * @return
     */
    @GetMapping("/api/merchant/insurancegift/shop/order/gift/addCopyTime")
    Response giftCopyTimeAdd(@RequestParam(value = "itemCode") String itemCode);

    /**
     * 获取投保礼当前状态各商户或会员的数量
     * @param orderType 投保礼类型
     * @param storeCode 商户编码
     * @param memberNo 会员编码
     * @return
     */
    @GetMapping("/api/merchant/insurancegift/shop/order/gift/current/count")
    Response<Map<String, Long>> giftBeingCount(@RequestParam(value = "orderType") String orderType, @RequestParam(value = "storeCode", required = false) String storeCode, @RequestParam(value = "memberNo", required = false) String memberNo);

    /**
     * 更新礼品状态
     * @param updateDto
     * @return
     */
    @PostMapping("/api/merchant/insurancegift/shop/order/gift/updategiftstatus")
    Response updateGiftStatus(@RequestBody GiftStatusUpdateDto updateDto);

    /**
     * 通过主订单号更新主订单信息
     * @param mainOrder
     * @return
     */
    @PostMapping("/api/merchant/insurancegift/shop/order/gift/updateMainOrderInfo")
    Response updateGiftBaseInfoForMainByMainCode(@RequestBody MerchantInsuranceGiftMainOrder mainOrder);

    /**
     * 通过二级订单号更新二级订单信息
     * @param itemOrder
     * @return
     */
    @PostMapping("/api/merchant/insurancegift/shop/order/gift/updateItemOrderInfo")
    Response updateGiftBaseInfoForItemByItemCode(@RequestBody MerchantInsuranceGiftItemOrder itemOrder);

    /**
     * 删除意向单
     * @param mainCode 主订单号
     * @param userCode 用户号
     * @param orderType 类型
     * @return
     */
    @GetMapping("/api/merchant/insurancegift/shop/order/gift/deleteGiftByMainCode")
    Response deleteGiftByMainCode(@RequestParam(value = "mainCode") String mainCode, @RequestParam(value = "userCode") String userCode, @RequestParam(value = "orderType") String orderType);

    /**
     * 保存审核后赠送失败的订单号
     * @param giftFail
     * @return
     */
    @PostMapping("/api/merchant/insurancegift/shop/order/gift/saveGivingFailOrder")
    Response saveGivingFailOrder(@RequestBody MerchantInsuranceGiftFail giftFail);

    /**
     * 保存方案操作日志
     * @param log
     * @return
     */
    @PostMapping("/api/merchant/insurancegift/shop/order/gift/saveOpLog")
    Response savePlanOpLog(@RequestBody MerchantInsuranceGiftLog log);

    /**
     * 方案操作日志列表
     * @param query
     * @return
     */
    @PostMapping("/api/merchant/insurancegift/shop/order/gift/oplogList")
    Response<List<MerchantInsuranceGiftLog>> queryList(@RequestBody GiftLogQuery query);
    /*******************************************************liangyongtong 商户投保礼订单 end**************************************************/

    // -----------------------------luoyc 商户投保礼方案 start-----------------------------

    /**
     * 商户投保礼方案列表
     *
     * @param pageDto
     * @return
     */
    @PostMapping("/api/merchant/insurancegift/admin/gift/queryPage")
    PageTable<MerchantInsuranceGiftMainOrderAdminDto> adminGiftMainOrderPageTable(@RequestBody MerchantInsuranceGiftMainOrderAdminPageDto pageDto);

    /**
     * 方案详情
     *
     * @param mainCode
     * @return
     */
    @GetMapping("/api/merchant/insurancegift/admin/gift/gainMainOrder/{mainCode}")
    Response<MerchantInsuranceGiftMainOrder> gainMainOrder(@PathVariable("mainCode") String mainCode);

    /**
     * 通过微信订单号获取意向单
     *
     * @param wxCode 微信订单号
     * @return
     */
    @GetMapping("/api/merchant/insurancegift/admin/gift/gainMainOrderByWxCode/{wxCode}")
    Response<MerchantInsuranceGiftMainOrder> gainMainOrderByWxCode(@PathVariable("wxCode") String wxCode);

    /**
     * 礼品方案
     *
     * @param mainCode
     * @param limit
     * @param page
     * @return
     */
    @GetMapping("/api/merchant/insurancegift/admin/gift/queryItemPage")
    PageTable<MerchantInsuranceGiftItemOrder> queryItemPage(
            @RequestParam(value = "mainCode") String mainCode,
            @RequestParam(value = "limit", defaultValue = "20") int limit,
            @RequestParam(value = "page", defaultValue = "1") int page);

    /**
     * 方案明细
     *
     * @param itemCode
     * @return
     */
    @GetMapping("/api/merchant/insurancegift/admin/gift/gainItemDetailList")
    PageTable<MerchantInsuranceGiftItemDetail> gainItemDetailList(
            @RequestParam(value = "itemCode") String itemCode,
            @RequestParam(value = "limit", defaultValue = "20") int limit,
            @RequestParam(value = "page", defaultValue = "1") int page);

    /**
     * 方案明细汇总
     *
     * @param itemCode
     * @return
     */
    @GetMapping("/api/merchant/insurancegift/admin/gift/gainItemDetailCount")
    Response<MerchantInsuranceGiftItemOrderDetailDTO> gainItemDetailCount(@RequestParam(value = "itemCode") String itemCode);

    /**
     * 商户投保礼方案列表
     *
     * @param code
     * @param license
     * @param mobile
     * @param storeCode
     * @param companyName
     * @param status
     * @return
     */
    @GetMapping("/api/merchant/insurancegift/admin/export")
    Response<List<MerchantInsuranceGiftMainOrderAdminDto>> adminExport(
            @RequestParam(value = "code", required = false) String code,
            @RequestParam(value = "license", required = false) String license,
            @RequestParam(value = "mobile", required = false) String mobile,
            @RequestParam(value = "storeCode", required = false) String storeCode,
            @RequestParam(value = "status", required = false) String status,
            @RequestParam(value = "orderType", required = false) String orderType,
            @RequestParam(value = "userName", required = false) String userName,
            @RequestParam(value = "userMobile", required = false) String userMobile,
            @RequestParam(value = "planType", required = false) String planType);

    /**
     * 意向单方案列表导出v2.0
     * @param exportDto
     * @return
     */
    @PostMapping("/api/merchant/insurancegift/admin/newExport")
    Response<List<MerchantInsuranceGiftMainOrderAdminDto>> adminNewExport(@RequestBody MerchantInsuranceGiftMainOrderAdminPageDto exportDto);

    /**
     * 方案详情2.0
     *
     * @param itemCode
     * @return
     */
    @GetMapping("/api/merchant/insurancegift/admin/gift/gainItemOrderInfo/{itemCode}")
    Response<MerchantInsuranceGiftItemOrderDetailInfoAdminDto> gainItemOrderInfo(@PathVariable("itemCode") String itemCode);

    /**
     * 方案生命周期列表
     *
     * @param itemCode
     * @param limit
     * @param page
     * @return
     */
    @GetMapping("/api/merchant/insurancegift/admin/gift/lifecycleLog/{itemCode}")
    public PageTable<MerchantInsuranceGiftLog> pageLifecycleLog(@PathVariable("itemCode") String itemCode,
                                                                @RequestParam(value = "limit", required = true) int limit,
                                                                @RequestParam(value = "page", required = true) int page);
    // -----------------------------luoyc 商户投保礼方案 end-----------------------------

    // -----------------------------liangyongtong 微信支付业务流水 start-----------------------------
    /**
     * 处理超时未支付订单失效
     * @return
     */
    @GetMapping("/api/order/pay/wx/paymentflow/failure")
    Response taskToFailureForOrder();
    // -----------------------------liangyongtong 微信支付业务流水 end-----------------------------

    /**
     * 投保礼方案支付详情
     * @param mainCode
     * @return
     */
    @GetMapping("/api/merchant/insurancegift/admin/gift/gainOrderInsuranceGiftPayment/{mainCode}")
    Response<MerchantInsuranceGiftOrderPaymentDTO> gainOrderInsuranceGiftPayment(@PathVariable("mainCode") String mainCode);

    // -----------------------------ranxiangfu 提现费率设置 start-----------------------------
    /**
     * Page query user agent identity auth page table.
     *
     * @param pageQueryDto the page query dto
     * @return the page table
     */
    @GetMapping("/api/order/withdrawals-fee-rate-config/pageQuery")
    PageTable<WithdrawalsFeeRateConfig> pageQueryWithdrawalsFeeRateConfig(@SpringQueryMap WithdrawalsFeeRatePageQueryDto pageQueryDto);

    /**
     * Save or update withdrawals fee rate config response.
     *
     * @param feeRateConfig the fee rate config
     * @return the response
     */
    @PostMapping("/api/order/withdrawals-fee-rate-config")
    Response saveOrUpdateWithdrawalsFeeRateConfig(@RequestBody WithdrawalsFeeRateConfig feeRateConfig);

    /**
     * Batch save withdrawals fee rate config response.
     *
     * @param feeRateConfigList the fee rate config list
     * @return the response
     */
    @PostMapping("/api/order/withdrawals-fee-rate-config/saveBatch")
    Response batchSaveWithdrawalsFeeRateConfig(@RequestBody List<WithdrawalsFeeRateConfig> feeRateConfigList);

    /**
     * Batch update withdrawals fee rate config response.
     *
     * @param feeRateConfigList the fee rate config list
     * @return the response
     */
    @PostMapping("/api/order/withdrawals-fee-rate-config/updateBatch")
    Response batchUpdateWithdrawalsFeeRateConfig(@RequestBody List<WithdrawalsFeeRateConfig> feeRateConfigList);

    /**
     * Delete withdrawals fee rate config response.
     *
     * @param id the id
     * @return the response
     */
    @DeleteMapping("/api/order/withdrawals-fee-rate-config")
    Response deleteWithdrawalsFeeRateConfig(@RequestParam(value = "id") Long id);

    /**
     * Gets withdrawals fee rate config by mobile.
     *
     * @param mobile       the mobile
     * @param queryDefault the query default
     * @return the withdrawals fee rate config by mobile
     */
    @GetMapping("/api/order/withdrawals-fee-rate-config/getByMobile")
    Response<WithdrawalsFeeRateConfig> getWithdrawalsFeeRateConfigByMobile(@RequestParam(value = "mobile", required = false) String mobile,
                                                                           @RequestParam(value = "status", required = false) String status,
                                                                           @RequestParam(value = "queryDefault", required = false) Boolean queryDefault);

    /**
     * List withdrawals fee rate by mobile and issuing body response.
     *
     * @param mobile              the mobile
     * @param issuingBodyCodeList the issuing body code list
     * @return the response
     */
    @GetMapping("/api/order/withdrawals-fee-rate-config/listByMobileAndIssuingBody")
    Response<Map<String, BigDecimal>> listWithdrawalsFeeRateByMobileAndIssuingBody(
            @RequestParam(value = "mobile", required = false) String mobile,
            @RequestParam(value = "issuingBodyCodeList", required = false) List<String> issuingBodyCodeList);

    @PostMapping("/api/order/withdrawals-fee-rate-config/checkIsExist")
    Response<List<String>> checkIsExist(@RequestBody List<String> mobileList);

    @GetMapping("/api/order/withdrawals-fee-rate-config")
    Response<WithdrawalsFeeRateConfig> getWithdrawalsFeeRateConfig(@RequestParam(value = "id") Integer id);

    @PostMapping("/api/order/withdrawals-fee-rate-config-log")
    Response saveWithdrawalsFeeRateConfigLog(@RequestBody WithdrawalsFeeRateConfigLog feeRateConfigLog);

    /**
     * Batch save withdrawals fee rate config log response.
     *
     * @param feeRateConfigLogList the fee rate config log list
     * @return the response
     */
    @PostMapping("/api/order/withdrawals-fee-rate-config-log/saveBatch")
    Response batchSaveWithdrawalsFeeRateConfigLog(@RequestBody List<WithdrawalsFeeRateConfigLog> feeRateConfigLogList);

    /**
     * Page query withdrawals fee rate config log by mobile page table.
     *
     * @param code  the code
     * @param page  the page
     * @param limit the limit
     * @return the page table
     */
    @GetMapping("/api/order/withdrawals-fee-rate-config-log/pageQuery")
    PageTable<WithdrawalsFeeRateConfigLog> pageQueryWithdrawalsFeeRateConfigLogByCode(
            @RequestParam(value = "code", required = false) String code,
            @RequestParam("page") Integer page,
            @RequestParam("limit") Integer limit);
    // -----------------------------ranxiangfu 提现费率设置 end-----------------------------

    // -----------------------------ranxiangfu 商品订单积分回馈 start-----------------------------
    /**
     * Save or update goods order point feedback response.
     *
     * @param pointFeedback the point feedback
     * @return the response
     */
    @PostMapping("/api/order/goods-order-point-feedback/saveOrUpdate")
    Response saveOrUpdateGoodsOrderPointFeedback(@RequestBody GoodsOrderPointFeedback pointFeedback);

    /**
     * Batch save goods order point feedback response.
     *
     * @param pointFeedbackList the point feedback list
     * @return the response
     */
    @PostMapping("/api/order/goods-order-point-feedback/batchSave")
    Response<List<GoodsOrderPointFeedback>> batchSaveGoodsOrderPointFeedback(@RequestBody List<GoodsOrderPointFeedback> pointFeedbackList);

    /**
     * list goods order point feedback by order code response.
     *
     * @param orderCode the order code
     * @return the response
     */
    @GetMapping("/api/order/goods-order-point-feedback/listByOrderCode")
    Response<List<GoodsOrderPointFeedback>> listGoodsOrderPointFeedbackByOrderCode(@RequestParam("orderCode") String orderCode);

    /**
     * Point feedback summary response.
     *
     * @param memberUserCodeList the member user code list
     * @return the response
     */
    @GetMapping("/api/order/goods-order-point-feedback/summary")
    Response<List<PointFeedbackSummaryDto>> goodsOrderPointFeedbackSummary(@RequestParam("memberUserCodeList") List<String> memberUserCodeList);
    // -----------------------------ranxiangfu 商品订单积分回馈 end-----------------------------



    // -------------------------- dongjuntong 车辆年审订单 start -----------------------



    /**
     * 年检免检券下单
     * @param param
     * @return
     */
    @PostMapping("/api/order/car/inspection/create")
    Response<SaleOrder> createCarInspectionOrder( @RequestBody CarInspectionOrderCreateDTO param);



    /**
     * 现金支付 年审代金券
     * @param orderCashPayDto
     * @return
     */
    @PostMapping("/api/order/car/inspection/cashPayOrder")
    Response<SaleOrder> payCarInspectionOrderByCash( @RequestBody OrderCashPayDto orderCashPayDto);


    /**
     * 查询需要重新发放的年审券列表
     * @param inspectionType 年审券类型
     * @param status 订单状态
     * @return
     */
    @GetMapping("/api/order/car/inspection/retry")
    Response<List<CarInspectionOrder>> findPendingRetryCarInspectionOrder( @RequestParam("inspectionType")String inspectionType,
                                                                @RequestParam("status")String status);


    /**
     * 更新年审订单信息
     * @param carInspectionOrder
     * @return
     */
    @PostMapping("/api/order/car/inspection/update")
    Response<SaleOrder> updateCarInspectionOrder( @RequestBody CarInspectionOrder carInspectionOrder);


    /**
     * 年审订单回调
     * @param orderCode
     * @param state
     * @return
     */
    @PostMapping("/api/order/car/inspection/callBack")
    Response<SaleOrder> carInspectionCallBackHandle(@RequestParam("orderNo") String orderCode,@RequestParam("status")  Integer state);

    /**
     * 后台管理查询
     * @param searchParamDto
     * @return
     */
    @GetMapping("/api/order/car/adminSearch")
     PageTable<SaleOrder> adminSearchCarInspectionOrder(@SpringQueryMap CarInspectionAdminSearchParamsDTO searchParamDto);


    /**
     * 查询年审订单详情
     * @param orderCode 订单编号
     * @return
     */
    @GetMapping("/api/order/car/detail")
    Response<SaleOrder> carInspectionOrderDetail(@RequestParam("orderCode")String orderCode);


    /**
     * 手动创建订单年审订单
     * @param orderCode 订单编号
     * @return
     */
    @GetMapping("/api/order/car/manualCreateDingJuOrder/{orderCode}")
    Response<String> manualCreateDingJuCarInspectionOrder(@PathVariable("orderCode")String orderCode);


    /**
     * 根据用户编号统计订单数量
     * @param userCode
     * @return
     */
    @GetMapping("/api/order/car/count")
    Response<List<OrderCountResultDto>> countCarInspectionOrder(@RequestParam("userCode") String userCode);


    // -------------------------- dongjuntong 车辆年审订单 end -----------------------

}

