package com.rocogz.syy.order.util;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.io.resource.UrlResource;
import cn.hutool.json.JSONObject;
import cn.hutool.json.JSONUtil;
import org.apache.commons.lang3.StringUtils;

import java.net.URL;
import java.time.DayOfWeek;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * 法定节假日工具类
 *
 * @author zhangmin
 */
public final class ChineseHoliday {

    //指定的节假日
    private static String specificHolidayUrl = "https://gameboylv.github.io/ChineseHoliday/data/%s.txt";

    //指定的工作日
    private static String specificWorkdayUrl = "https://gameboylv.github.io/ChineseHoliday/data/%s_w.txt";

    private static Map<Integer, List<LocalDate>> holidayCache = new ConcurrentHashMap<>();

    private static List<LocalDate> holidayOfYear(int year) {

        List<LocalDate> holidayList = new ArrayList<LocalDate>();
        LocalDate firstDayOfYear = LocalDate.of(year, 1, 1);

        int daysOfYear = firstDayOfYear.plusYears(1).minusDays(1).getDayOfYear();

        for (int i = 1; i <= daysOfYear; i++) {

            LocalDate day = firstDayOfYear.withDayOfYear(i);

            if (day.getDayOfWeek() == DayOfWeek.SATURDAY || day.getDayOfWeek() == DayOfWeek.SUNDAY) {
                holidayList.add(day);
            }
        }


        try {
            String holidayUrl = String.format(specificHolidayUrl, year);
            String specificHolidays = getContentFromUrl(holidayUrl);
            String[] specificHolidayArray = specificHolidays.split("\\n");

            for (String holiday : specificHolidayArray) {
                if (StringUtils.isBlank(holiday)) {
                    continue;
                }

                holiday = holiday.trim();

                if (holiday.length() != 4) {
                    continue;
                }

                LocalDate holidayDate = LocalDate.of(year, Integer.parseInt(holiday.substring(0, 2)), Integer.parseInt(holiday.substring(2, 4)));

                if (!holidayList.contains(holidayDate)) {
                    holidayList.add(holidayDate);
                }
            }

            String workDayUrl = String.format(specificWorkdayUrl, year);
            String specificWorkday = getContentFromUrl(workDayUrl);

            for (String workDay : specificWorkday.split("\\n")) {
                if (StringUtils.isBlank(workDay)) {
                    continue;
                }

                workDay = workDay.trim();

                if (workDay.length() != 4) {
                    continue;
                }

                LocalDate workDate = LocalDate.of(year, Integer.parseInt(workDay.substring(0, 2)), Integer.parseInt(workDay.substring(2, 4)));
                if (holidayList.contains(workDate)) {
                    holidayList.remove(workDate);
                }
            }


            Collections.sort(holidayList);


        } catch (Exception ex) {


            holidayList = null;
        }

        return holidayList;
    }


    private static List<LocalDate> getHoliday(int year) {
        if (!holidayCache.containsKey(year)) {
            List<LocalDate> holidayListOfYear = holidayOfYear(year);

            if (CollectionUtil.isNotEmpty(holidayListOfYear)) {
                holidayCache.put(year, holidayListOfYear);
            }
        }

        return holidayCache.get(year);
    }


    /**
     * 获取指定日期的下一个工作日
     *
     * @param date
     * @return
     */
    public static LocalDate getNextWorkingDate(LocalDate date) {

        LocalDate nextDate = date.plusDays(1);

        while (!isWorkingDay(nextDate)) {
            nextDate = nextDate.plusDays(1);
        }
        return nextDate;
    }


    public static boolean isWorkingDay(LocalDate date) {

        List<LocalDate> holidayList = getHoliday(date.getYear());

        if (CollectionUtil.isNotEmpty(holidayList)) {
            return !holidayList.contains(date);
        }

        //如果 holidayList ==null,说明 调用 specificHolidayUrl 地址发生错误，例如该网站被关闭了

        //继续 通过  holiday.dreace.top 网站查询是否是工作日

        boolean isWorkingDay = true;

        try {
            isWorkingDay = !isHolidayDayByDreace(date);
        } catch (Exception ex) {

            //如果 通过 网站：https://holiday.dreace.top 查询是否工作日，也出现异常，例如：该网站也被关闭了，则通过java api方式直接返回是否是工作日

            isWorkingDay = isWorkdayByLocal(date);
        }

        return isWorkingDay;
    }

    /**
     * 获取预计到账时间
     * 如果是工作日：
     * <= 16:00,当天 20:00到账
     * 否则 16:00之后,下个工作日：20:00点到账
     *
     * @param createTime
     * @return
     */
    public static String getPredictReceiptTime(LocalDateTime createTime) {

        LocalDate createDate = createTime.toLocalDate();

        boolean workingDay = isWorkingDay(createDate);

        LocalTime currentTime = createTime.toLocalTime();

        //下午16点
        LocalTime sixtyTime = LocalTime.of(16, 0, 0);

        String predictReceiptTime = null;

        if (workingDay && !currentTime.isAfter(sixtyTime)) {
            //如果是工作日 &&  <=16点,则预计 当前20:00到账
            predictReceiptTime = createDate.toString() + " 20:00";
        } else {
            //下个工作日 20:00到账
            predictReceiptTime = getNextWorkingDate(createDate).toString() + " 20:00";

        }

        return predictReceiptTime;
    }


    /**
     * 参考： https://github.com/Dreace/ChinaHolidayAPI
     * https://holiday.dreace.top/?date=2023-01-02
     *
     * @param date
     * @return 返回是否是法定节假日
     */
    private static Boolean isHolidayDayByDreace(LocalDate date) throws Exception {
        String url = "https://holiday.dreace.top/?date=" + date.toString();
        /**
         {
         "date": "2023-01-02",
         "isHoliday": true,
         "note": "元旦",
         "type": "假日"
         }
         */

        String dateJson = getContentFromUrl(url);

        JSONObject jsonObj = JSONUtil.parseObj(dateJson);

        Boolean isHoliday = (Boolean) jsonObj.get("isHoliday");

        return Boolean.TRUE.equals(isHoliday);

    }


    /**
     * 判断指定日期是否为工作日（周一到周五）
     *
     * @param date 要判断的日期
     * @return 是否为工作日
     */
    private static boolean isWorkdayByLocal(LocalDate date) {
        // 获取该日期是周几
        DayOfWeek dayOfWeek = date.getDayOfWeek();
        // 判断是否为周六或周日
        if (dayOfWeek == DayOfWeek.SATURDAY || dayOfWeek == DayOfWeek.SUNDAY) {
            return false;
        }
        return true;
    }


    private static String getContentFromUrl(String url) throws Exception {
        UrlResource urlRes = new UrlResource(new URL(url));
        String content = urlRes.readUtf8Str();
        if (StringUtils.isNotBlank(content)) {
            content = content.trim();
        }

        return content;
    }

}