package com.rocogz.syy.order.util;

import java.time.DayOfWeek;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.time.temporal.TemporalAdjusters;

/**
 * <dl>
 * <dd>Description: DateTimeUtils</dd>
 * <dd>Company: 大诚若谷信息技术有限公司</dd>
 * <dd>@date：2020/5/12 9:40</dd>
 * <dd>@author：ranxiangfu</dd>
 * </dl>
 */
public class DateTimeUtil {
    private DateTimeUtil(){}

    private static final String DATE_FORMAT = "yyyy-MM-dd";
    private static final String DATE_NO_DELIMITER_FORMAT = "yyyyMMdd";
    private static final String DATE_HOUR_FORMAT = "yyyy-MM-dd HH";
    private static final String DATE_HOUR_MINUTE_FORMAT = "yyyy-MM-dd HH:mm";
    private static final String DATE_TIME_FORMAT = "yyyy-MM-dd HH:mm:ss";
    private static final String DATE_TIME_NO_DELIMITER_FORMAT = "yyyyMMddHHmmss";

    public static final DateTimeFormatter DATE_TIME_FORMATTER = DateTimeFormatter.ofPattern(DATE_TIME_FORMAT);
    public static final DateTimeFormatter DATE_FORMATTER = DateTimeFormatter.ofPattern(DATE_FORMAT);
    public static final DateTimeFormatter DATE_TIME_NO_DELIMITER_FORMATTER = DateTimeFormatter.ofPattern(DATE_TIME_NO_DELIMITER_FORMAT);

    public static final String TIME_DAY_START = " 00:00:00";
    public static final String TIME_DAY_END = " 23:59:59";
    /**
     * 获取某周的开始日期
     *
     * @param offset 0本周，1下周，-1上周，依次类推
     * @return
     */
    public static LocalDate weekStart(int offset) {
        LocalDate localDate = LocalDate.now().plusWeeks(offset);
        return localDate.with(DayOfWeek.MONDAY);
    }

    /**
     * 获取某周的结束日期
     *
     * @param offset 0本周，1下周，-1上周，依次类推
     * @return
     */
    public static LocalDate weekEnd(int offset) {
        LocalDate localDate = LocalDate.now().plusWeeks(offset);
        return localDate.with(DayOfWeek.SUNDAY);
    }

    /**
     * 获取某月的开始日期
     *
     * @param offset 0本月，1下个月，-1上个月，依次类推
     * @return
     */
    public static LocalDate monthStart(int offset) {
        return LocalDate.now().plusMonths(offset).with(TemporalAdjusters.firstDayOfMonth());
    }

    /**
     * 获取某月的结束日期
     *
     * @param offset 0本月，1下个月，-1上个月，依次类推
     * @return
     */
    public static LocalDate monthEnd(int offset) {
        return LocalDate.now().plusMonths(offset).with(TemporalAdjusters.lastDayOfMonth());
    }

    /**
     * 获取某季度的开始日期
     * 季度一年四季， 第一季度：1月-3月， 第二季度：4月-6月， 第三季度：7月-9月， 第四季度：10月-12月
     *
     * @param offset 0本季度，1下个季度，-1上个季度，依次类推
     * @return
     */
    public static LocalDate quarterStart(int offset) {
        final LocalDate date = LocalDate.now().plusMonths(offset * 3);
        //当月
        int month = date.getMonth().getValue();
        int start = 0;
        if (month <= 3) {
            //第一季度
            start = 1;
        } else if (month <= 6) {
            //第二季度
            start = 4;
        } else if (month <= 9) {
            //第三季度
            start = 7;
        } else {
            //第四季度
            start = 10;
        }
        return date.plusMonths(start - month).with(TemporalAdjusters.firstDayOfMonth());
    }

    /**
     * 获取某季度的结束日期
     * 季度一年四季， 第一季度：1月-3月， 第二季度：4月-6月， 第三季度：7月-9月， 第四季度：10月-12月
     *
     * @param offset 0本季度，1下个季度，-1上个季度，依次类推
     * @return
     */
    public static LocalDate quarterEnd(int offset) {
        final LocalDate date = LocalDate.now().plusMonths(offset * 3);
        //当月
        int month = date.getMonth().getValue();
        int end = 0;
        if (month <= 3) {
            //第一季度
            end = 3;
        } else if (month <= 6) {
            //第二季度
            end = 6;
        } else if (month <= 9) {
            //第三季度
            end = 9;
        } else {
            //第四季度
            end = 12;
        }
        return date.plusMonths(end - month).with(TemporalAdjusters.lastDayOfMonth());
    }

    /**
     * 获取某年的开始日期，某年的第1天
     *
     * @param offset 0今年，1明年，-1去年，依次类推
     * @return
     */
    public static LocalDate yearStart(int offset) {
        return LocalDate.now().plusYears(offset).with(TemporalAdjusters.firstDayOfYear());
    }

    /**
     * 获取某年的结束日期,某年的最后1天
     *
     * @param offset 0今年，1明年，-1去年，依次类推
     * @return
     */
    public static LocalDate yearEnd(int offset) {
        return LocalDate.now().plusYears(offset).with(TemporalAdjusters.lastDayOfYear());
    }
}
