package com.rocogz.redis;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.script.DefaultRedisScript;
import org.springframework.data.redis.core.script.RedisScript;
import org.springframework.stereotype.Component;

import java.util.Arrays;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;

/**
 * Created by zhangmin on 2019/5/12.
 * redis实现的分布式锁
 * <pre>
 *     (1)如果要用默认锁，则直接@Autowired注解注入到系统中就行了
 *     (2）如果要使用非默认锁，则直接 new RedisDistributedLock("您自己的锁名字")
 *     使用方法：在减库存业务方法中：
 *     public void decreaseStock(int stockNum) {
 *         RedisDistributedLock rLock = new RedisDistributedLock("您自己的锁名字");
 *         try{
 *               rLock.lock();
 *               //自己的扣库存逻辑
 *         }finally{
 *               rLock.unlock();
 *         }
 *     }
 * </pre>
 */
@SuppressWarnings("all")
@Component
public class RedisDistributedLock implements Lock {

    private final static String DEFAULT_LOCK_NAME = "redisDistributedLock";

    /**
     * 释放锁的lua脚本,只有持有锁的线程，才能解锁
     */
    private final static String script =  "if redis.call('get', KEYS[1]) == ARGV[1] then return redis.call('del', KEYS[1]) else return 0 end";

    private static RedisScript<Long> redisScript = new DefaultRedisScript<>(script, Long.class);

    /**
     * 锁的默认有效期 10秒
     */
    private final long LOCK_DEFAULT_EXPIRE_TIME= 10_000;

    /**
     * 重试获得锁的 时间间隔，毫秒数
     */
    private final long RETRY_INTERVAL_MILLISECOND = 20;

    private String lockName;

    private ThreadLocal<String> lockValueHolder = new ThreadLocal<>();

    /**
     * 占有锁的线程
     */
    private Thread executiveOwnerThread;

    @Autowired
    private RedisService redisService;

    public RedisDistributedLock() {
        this(DEFAULT_LOCK_NAME);
    }

    public RedisDistributedLock(String lockName) {
        this.lockName = lockName;
    }

    public RedisDistributedLock(String lockName, RedisService redisService) {
        this.lockName = lockName;
        this.redisService = redisService;
    }

    /**
     * 获得锁，如果没有获得到锁，则一直等待
     */
    @Override
    public void lock() {
        while(!tryLock()) {
            try {
                TimeUnit.MILLISECONDS.sleep(RETRY_INTERVAL_MILLISECOND);
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
        }
    }

    /**
     * 返回是否成功获得锁,如果获得不到锁 则直接返回false
     * @return
     */
    @Override
    public boolean tryLock() {
        return tryAcquire(LOCK_DEFAULT_EXPIRE_TIME,TimeUnit.MILLISECONDS);
    }

    /**
     * 尝试获得锁，最多等待time时间，如果获得不到 返回false
     * @param waitTime
     * @param unit
     * @return
     */
    @Override
    public boolean tryLock(long waitTime, TimeUnit unit) {
        long base = System.currentTimeMillis();
        long elapsedTime = 0;
        //要等待的毫秒数
        long waitMillis = unit.toMillis(waitTime);
        while(!tryAcquire(waitTime,unit)) {
            long delay = waitMillis - elapsedTime;
            if (delay <= 0) {
                 return false;
            }
            try {
                TimeUnit.MILLISECONDS.sleep(RETRY_INTERVAL_MILLISECOND);
            } catch (InterruptedException e) {
            }
            elapsedTime = System.currentTimeMillis() - base;
        }

        return true;
    }

    /**
     * @param ttl 锁的有效期时间
     * @param unit 锁的有效期时间单位
     * @return
     */
    private boolean tryAcquire(long ttl, TimeUnit unit) {
        String lockValue = UUID.randomUUID().toString();
        Thread currentThread = Thread.currentThread();
        if(this.redisService.setIfAbsent(lockName, lockValue, ttl, unit)) {
            lockValueHolder.set(lockValue);
            setExecutiveOwnerThread(currentThread);
            return true;
        }
        else if(currentThread == executiveOwnerThread) {
            //可重入
            return true;
        }
        return false;
    }

    @Override
    public void unlock() {
        if(this.lockValueHolder.get() == null) {
            return;
        }

        Long deleteKeyNum = this.redisService.execute(redisScript, Arrays.asList(lockName),lockValueHolder.get());
        if(deleteKeyNum>0) {
            this.lockValueHolder.remove();
        }
    }

    @Override
    public Condition newCondition() {
        return null;
    }

    @Override
    public void lockInterruptibly() throws InterruptedException {
    }

    private void setExecutiveOwnerThread(Thread executiveOwnerThread) {
        this.executiveOwnerThread = executiveOwnerThread;
    }
}
