package com.rocogz.swagger.spring.boot.autoconfigure;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.*;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.stereotype.Controller;
import org.springframework.util.StringUtils;
import springfox.documentation.builders.ApiInfoBuilder;
import springfox.documentation.builders.PathSelectors;
import springfox.documentation.builders.RequestHandlerSelectors;
import springfox.documentation.service.ApiInfo;
import springfox.documentation.service.Contact;
import springfox.documentation.spi.DocumentationType;
import springfox.documentation.spring.web.plugins.Docket;
import springfox.documentation.swagger2.annotations.EnableSwagger2;

import static com.rocogz.swagger.spring.boot.autoconfigure.SwaggerProperties.CompanyCopyRight;

/**
 * 自动配置成功后访问：http://localhost:port/swagger-ui.html
 * 使用说明：
 * （1）导入 swagger-starter 依赖包以后,在配置application.properties文件中：配置如下：
 *
 *      swagger.basePackage=com.rocogz.demo.controller  //您的controller基础包
 *
 * (2)swagger在导入依赖后,默认开启生成api文档功能,在生产环境应该禁用swagger功能,禁用swagger需配置如下:
 *     swagger.enabled=false
 *
 *  swagger自动配置
 * @author zhangmin
 * @date 2020/1/15
 */
@ConditionalOnWebApplication
@ConditionalOnClass(Controller.class)
@ConditionalOnMissingBean(Docket.class)
@EnableSwagger2
@EnableConfigurationProperties(SwaggerProperties.class)
@Configuration
@ConditionalOnProperty(name = "swagger.enabled", havingValue = "true", matchIfMissing = true)
public class Swagger2AutoConfiguration {

    @Autowired
    private SwaggerProperties swaggerProp;

    @Bean
    public Docket swaggerRestApi() {
        return new Docket(DocumentationType.SWAGGER_2)
                .apiInfo(apiInfo())
                .select()
                .apis(RequestHandlerSelectors.basePackage(swaggerProp.getBasePackage()))
                .paths(PathSelectors.any())
                .build();
    }

    private ApiInfo apiInfo() {
        ApiInfo apiInfo = new ApiInfoBuilder().title(getTitle())
                .description(getDescription())
                .termsOfServiceUrl(getServiceUrl())
                .contact(getContact())
                .version(getVersion())
                .build();

        return apiInfo;
    }



    private String getTitle() {
        String title = CompanyCopyRight.DEFAULT_COMPANY.getName();
        CompanyCopyRight company = swaggerProp.getCompany();
        if(company!=null && !StringUtils.isEmpty(company.getName())) {
            title = company.getName();
        }
        return title;
    }

    private String getDescription() {
        String description = CompanyCopyRight.DEFAULT_COMPANY.getDescription();
        CompanyCopyRight company = swaggerProp.getCompany();
        if(company!=null && !StringUtils.isEmpty(company.getDescription())) {
            description = company.getDescription();
        }
        return description;
    }


    private String getServiceUrl() {
        String serviceUrl = CompanyCopyRight.DEFAULT_COMPANY.getSiteUrl();
        CompanyCopyRight company = swaggerProp.getCompany();
        if(company!=null && !StringUtils.isEmpty(company.getSiteUrl())) {
            serviceUrl = company.getSiteUrl();
        }
        return serviceUrl;
    }


    private String getVersion() {
        String version = CompanyCopyRight.DEFAULT_COMPANY.getVersion();
        CompanyCopyRight company = swaggerProp.getCompany();
        if(company!=null && !StringUtils.isEmpty(company.getVersion())) {
            version = company.getVersion();
        }
        return version;
    }


    private Contact getContact() {
        String contactName = CompanyCopyRight.DEFAULT_COMPANY.getContactName();
        String contactEmail = CompanyCopyRight.DEFAULT_COMPANY.getContactEmail();

        CompanyCopyRight company = swaggerProp.getCompany();
        if(company!=null) {

            if (!StringUtils.isEmpty(company.getContactName())) {
                contactName = company.getContactName();
            }

            if(!StringUtils.isEmpty(company.getContactEmail())) {
                contactEmail = company.getContactEmail();
            }
        }

        return new Contact(contactName,getServiceUrl(),contactEmail);
    }


}
