package com.rocogz.syy.operation.common;

import cn.hutool.core.exceptions.ExceptionUtil;
import com.rocogz.syy.common.response.Response;
import com.rocogz.util.DateUtil;
import io.seata.common.util.CollectionUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.propertyeditors.CustomDateEditor;
import org.springframework.beans.propertyeditors.StringTrimmerEditor;
import org.springframework.util.StringUtils;
import org.springframework.validation.BindException;
import org.springframework.validation.FieldError;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.ServletRequestDataBinder;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ResponseBody;

import javax.servlet.http.HttpServletRequest;
import javax.validation.ValidationException;
import java.beans.PropertyEditorSupport;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Date;
import java.util.List;

/**
 * <dl>
 * <dd>描述:统一异常处理</dd>
 * <dd>公司: 大城若谷信息技术有限公司</dd>
 * <dd>创建人：zhangmin</dd>
 * </dl>
 */
@Slf4j
@ResponseBody
@ControllerAdvice
public class GlobalExceptionHandler {

    @InitBinder
    protected void initBinder(HttpServletRequest request, ServletRequestDataBinder binder) throws Exception {
        binder.registerCustomEditor(Date.class, new CustomDateEditor(new SimpleDateFormat(DateUtil.YYYY_MM_DD_HH_MM_SS), true));
        binder.registerCustomEditor(Date.class, new CustomDateEditor(new SimpleDateFormat(DateUtil.YYYY_MM_DD), true));
        binder.registerCustomEditor(LocalDate.class, new CustomLocalDateEditor());
        binder.registerCustomEditor(LocalDateTime.class, new CustomLocalDateTimeEditor());
        binder.registerCustomEditor(String.class,new StringTrimmerEditor(true));
    }

    /**
     * 使用jsr-303校验，如果校验不通过会抛出此异常，在此处捕获，进行处理
     * @param ex
     * @return
     */
    @ExceptionHandler({BindException.class})
    public Object validationException(BindException ex) {
        List<FieldError> errors = ex.getFieldErrors();
        String msg = this.getMessageFrom(errors);
        log.error(msg);
        return Response.failure(msg);
    }
    /**
     * 解决使用@ReqeustBody使用@Validated失效
     * @param ex
     * @return
     */
    @ExceptionHandler(MethodArgumentNotValidException.class)
    public Object methodArgumentNotValidException(MethodArgumentNotValidException ex) {
        List<FieldError> errors = ex.getBindingResult().getFieldErrors();
        String msg = this.getMessageFrom(errors);
        log.error(msg);
        return Response.failure(msg);
    }

    /**
     * 参数不合法异常
     * @return
     */
    @ExceptionHandler(IllegalArgumentException.class)
    public Object illegalArgumentException(IllegalArgumentException ex) {
        log.error(ExceptionUtil.stacktraceToString(ex));
        return Response.failure(ex.getMessage());
    }

    /**
     * 参数校验失败异常
     * @param ex
     * @return
     */
    @ExceptionHandler(ValidationException.class)
    public Object validationException(ValidationException ex) {
        log.error(ExceptionUtil.stacktraceToString(ex));
        return Response.failure(ex.getMessage());
    }

    /**
     * 统一异常处理
     *
     * @param ex
     * @return
     */
    @ExceptionHandler({Exception.class})
    public Object exception(Exception ex) {
        log.error(ExceptionUtil.stacktraceToString(ex));
        return Response.failure("操作失败！" +ex.getMessage());
    }

    public static class CustomLocalDateEditor extends PropertyEditorSupport {
        /**
         * Parse the Date from the given text, using the specified DateFormat.
         */
        @Override
        public void setAsText(String text) throws IllegalArgumentException {
            if (!StringUtils.hasText(text)) {
                setValue(null);
            } else {
                setValue(LocalDate.parse(text));
            }
        }

        /**
         * Format the LocalDate as String
         */
        @Override
        public String getAsText() {
            LocalDate value = (LocalDate) getValue();
            return (value != null ? value.toString() : "");
        }
    }

    public static class CustomLocalDateTimeEditor extends PropertyEditorSupport {
        /**
         * Parse the Date from the given text, using the specified DateFormat.
         */
        @Override
        public void setAsText(String text) throws IllegalArgumentException {
            if (!StringUtils.hasText(text)) {
                setValue(null);
            } else {
                setValue(LocalDateTime.parse(text, DateTimeFormatter.ofPattern(DateUtil.YYYY_MM_DD_HH_MM_SS)));
            }
        }
    }

    private String getMessageFrom(List<FieldError> errors) {
        StringBuilder sb = new StringBuilder();
        if (CollectionUtils.isNotEmpty(errors)) {
            errors.forEach((o) -> sb.append(o.getDefaultMessage()).append(","));
        }
        // 删除最后一个逗号
        sb.delete(sb.lastIndexOf(","), sb.length());
        return sb.toString();
    }
}
