package com.rocogz.syy.operation.entity.quotapply;

import com.baomidou.mybatisplus.annotation.TableField;
import com.rocogz.syy.common.annotation.UniqueField;
import com.rocogz.syy.common.entity.UserTimeEntity;
import com.rocogz.syy.operation.constants.OperationConstant;
import com.rocogz.syy.operation.enums.*;
import lombok.Getter;
import lombok.Setter;
import lombok.experimental.Accessors;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.List;

/**
 * <p>
 * 客户权益管理-> 额度申请表
 * </p>
 *
 * @author zhangmin
 * @since 2021-06-23
 */
@Setter
@Getter
@Accessors(chain = true)
public class OperateQuotaApply extends UserTimeEntity {

    /**
     * 申请单编码
     */
    @UniqueField
    private String code;

    /**
     * 申请单名称
     */
    private String name;

    /**
     * 额度申请的交易类型
     *
     * @see OperationConstant.DictQuotaTradeType#QUOTA_APPLY
     */
    private String tradeType;

    /**
     * 申请单 额度调整类型
     *
     * @see OperationConstant.DictQuotaAdjustType#ADD
     */
    private String adjustType;

    /**
     * 申请方式:整体或分团队或 分主体
     * 目前 applyWay都是 = ALL
     *
     * @see OperationConstant.DictQuotaApplyWay
     */
    private String applyWay;


    /**
     * 额度费用类型
     *
     * @see OperationConstant.DictQuotaApplyFeeType#UNIVERSAL_USE
     * @see settle_account_info#fee_type
     */
    private String feeType;

    /***
     * 申请端入口
     * @see OperationConstant.DictQuotaApplyPortal#B_WEAPP
     */
    private String portal;

    /**
     * 申请模式:运营代申请/主体自己申请  【额度申请 专有字段】
     *
     * @see OperationConstant.DictQuotaApplyMode#TYPE_CODE
     * @see OperateQuotaConfigAccount#applyMode
     */
    private QuotaApplyModeEnum applyMode;

    /**
     * 额度管理模式  【额度申请 专有字段】
     *
     * @see 备付金电子账户的 字典类型是： ELECTRONIC_ACCT_QUOTA_MODE
     * @see com.rocogz.syy.settlement.entity.electronicaccount.SettleElectronicAccountInfo#quotaMode
     * @see OperateQuotaConfigAccount#quotaManageMode
     */
    private QuotaManageModeEnum quotaManageMode;

    /**
     * 额度申请单 审批通过后的额度划拨模式，即审批通过后 申请单后续处理操作模式
     * （1）额度申请，读取 额度划拨模式 配置
     * （2）收回/划拨申请单，自动划拨,不读配置
     *
     * @see OperationConstant.DictQuotaApplyHandleMode#MANUAL_HANDLE
     * @see OperateQuotaConfigAccount#handleMode
     */
    private String handleMode;

    /**
     * 备付金模式下：充值类型  【额度申请 专有字段】
     *
     * @see 备付金电子账户的 字典类型是： ELECTRONIC_ACCT_PROVISION_TYPE
     */
    private CoverDepositTypeEnum depositType;

    /**
     * 申请单状态
     *
     * @see OperationConstant.DictQuotaApplyStatus
     */
    private String status;

    /**
     * 额度冻结状态
     * @see OperationConstant.DictQuotaApplyFrozenStatus#PENDING_FROZEN
     */
    private String frozenStatus;

    /**
     * 申请人用户id: reference basic_system_admin_user.id
     */
    private Integer applierUserId;

    /**
     * 申请人用户名: reference basic_system_admin_user.username
     */
    private String applierUser;

    /**
     * 申请人手机号:reference basic_system_admin_user.mobile
     */
    private String applierMobile;

    /**
     * 申请人名字: reference basic_system_admin_user.name
     */
    private String applierName;

    /**
     * 申请人 员工号 【发放主体账号 才有值】
     */
    private String empNo;

    /**
     * 申请人 员工所在部门 【发放主体账号 才有值】
     */
    private String empDept;

    /**
     * 申请人 工作岗位【发放主体账号 才有值】
     */
    private String empJob;


    /**
     * 上级审批节点编号,即最后审批/审核 节点编号
     */
    private String lastApproveNodeCode;

    /**
     * 上级审批节点名称,即最后审批/审核 节点名称
     */
    private String lastApproveNodeName;

    /**
     * 上级审批节点类型,即最后审批/审核节点类型
     */
    private QuotaApproveNodeTypeEnum lastApproveNodeType;

    /**
     * 上级审批结果
     *
     * @see OperationConstant.DictApproveResult
     */
    private String lastApproveResult;

    /**
     * 最后审批时间
     */
    private LocalDateTime lastApproveTime;

    /**
     * 当前审批节点编号(还未审)
     */
    private String curNodeCode;

    /**
     * 当前审批节点名称 (还未审)
     */
    private String curNodeName;

    /**
     * 当前审批节点类型,即审批业务类型
     *
     * @see QuotaApproveNodeTypeEnum
     */
    private String curNodeType;

    /**
     * 申请人的 发放主体编号: reference equity_issuing_body.code
     * (1) 额度申请，申请人发放主体编号
     * （2）收回或划拨申请，也是登录账户申请人的主体，即收回/划拨的父主体编号
     * （3）如果是划拨申请, 额度来源主体，即从该主体的 主体额度账户划出
     * （4）如果是收回申请, 即把额度收回到该主体额度账户,即该主体的主体额度账户转入额度
     */
    private String issuingBodyCode;

    /**
     * 操作的交易主体编号（子主体编号）,  【划拨/收回 申请专用字段】
     * （1）如果划拨申请，该主体编号就是划入的目标主体,    即划入到该主体额度账户
     * （2）如果是收回申请，该主体编号就是额度来源主体编号，即划出的主体额度账户
     */
    private String tradeIssuingBodyCode;

    /**
     * 申请人发放主体的祖先主体编号
     */
    private String topIssuingBodyCode;

    /**
     * 申请额度
     */
    private BigDecimal quota;

    /**
     * 额度转换备付金 兑换比例  【额度申请 专有字段】
     */
    private BigDecimal quotaConversionRatio;

    /**
     * 备付金：金额  【额度申请 专有字段】
     * （1）如果是额度管理模式  = 额度 * 额度转换备付金的 兑换比例
     * （2）如果是备付金模式  coverAmount = 充值金额(充值类型是付款充值) 或 授信借款金额（充值类型是授信充值）
     */
    private BigDecimal coverAmount;

    /**
     * 收款到账日期  【额度申请 专有字段】
     */
    private LocalDate receiptDate;

    /**
     * 【额度申请 专有字段】
     * <p>
     * 代理商编号：如果发放主体是代理商
     * 如果发放主体的quity_issuing_body.whether_customer='AGENT'
     */
    private String agentCode;

    /**
     * 【额度申请 专有字段】
     * <p>
     * 合作客户编号：如果发放主体的 equity_issuing_body.whether_customer='CUSTOMER'
     */
    private String customerCode;

    /**
     * 团队/发放主体 数量【团队/发放主体 申请方式专属字段】
     */
    private Integer teamNum;

    /**
     * 最近的操作: 如 审核、审批、处理
     *
     * @see QuotaApplyActionTypeEnum
     */
    private String recentAction;

    /**
     * 最近操作时间：审核时间,完成时间
     */
    private LocalDateTime recentActionTime;

    /**
     * 最近操作人用户名
     */
    private String recentActionUser;

    /**
     * 划拨成功操作人用户名
     */
    private String allocateUser;

    /**
     * 划拨操作人名字
     */
    private String allocateUserFullName;

    /**
     * 划拨操作时间
     */
    private LocalDateTime allocateTime;

    /**
     * 额度账号 的交易号logNo
     *
     * @see com.rocogz.syy.settlement.dto.TradeResp#logNo
     * @see settle_account_trade_log#log_no
     */
    private String quotaAcctLogNo;

    /**
     * 【额度申请 专有字段】
     * 划拨操作,备付金帐户充值交易业务单据号
     */
    private String coverAcctTradeNo;

    /**
     * 申请备注
     */
    private String remark;


    //是否被逻辑删除
    private Boolean deleted;


    /**
     * 是否允许修改
     *
     * @return
     */
    public boolean isModifyAllowed() {
        return OperationConstant.DictQuotaApplyStatus.AUDIT_FAILED.equals(status) ||
                OperationConstant.DictQuotaApplyStatus.APPROVE_FAILED.equals(status) ||
                OperationConstant.DictQuotaApplyStatus.DRAFT.equals(status);
    }


    /**
     * 申请是否可以审批
     */
    public boolean isApproveble() {
        return OperationConstant.DictQuotaApplyStatus.PENDING_AUDIT.equals(status) ||
                OperationConstant.DictQuotaApplyStatus.PENDING_APPROVE.equals(status);
    }


    /**
     * 需要充值的备付金:  coverQuota = quota *（合作客户或代理商）配置的额度转换系数
     */
    public BigDecimal calcCoverQuota() {
        if (quotaConversionRatio == null) {
            return null;
        }

        BigDecimal converQuota = quotaConversionRatio.multiply(quota.setScale(3, BigDecimal.ROUND_HALF_UP));
        return converQuota;
    }

    /**
     * 是否是最顶级发放主体
     */
    public boolean isTopLevelIssuingBody() {
        if (issuingBodyCode == null) {
            return false;
        }

        return this.issuingBodyCode.equals(topIssuingBodyCode);
    }

    //包含的团队/发放主体Item
    @TableField(exist = false)
    private List<OperateQuotaApplyItem> teamItemList;

    /**
     * 我最近的审批或审核记录 （小程序端 我已审批申请单列表 使用此字段）
     */
    @TableField(exist = false)
    private OperateQuotaApplyApproveRecord myRecentApproveRecord;


    //申请详情中要展示的 进度时间线
    @TableField(exist = false)
    private List<OperateQuotaApplyProgress> timelineList;


    //附件列表
    @TableField(exist = false)
    private List<OperateQuotaApplyAttach> attachList;


    //===== 以下是 非持久化属性 =====

    /**
     * 审批进度：待审核/待审批的 能 审批/审核的 用户名列表
     */
    private transient List<String> curApproverUserList;

    /**
     * 是否是新增/编辑申请时,是否是保存草稿,如果是保存草稿，就不启动工作流,申请单状态是DRAFT
     */
    private transient Boolean saveDraft;

    /**
     * 操作人：操作人信息用来设置到时间线上
     */
    @TableField(exist = false)
    private OperateQuotaApplyOperateUser operateUser;


    /**
     * 充值类型：名字
     */
    private transient String depositTypeLabel;

    //发放主体名称,页面显示使用
    private transient String issuingBodyName;

    /**
     * 主体额度账户编号,例如：NUMA00000001
     */
    private transient String issueBodyAcctNo;


    //交易的主体简称,页面显示使用【划拨/收回 专用字段】
    private transient String tradeIssuingBodyName;

    /**
     * 交易的主体 主体额度账户编号,例如：NUMA00000002，即下级主体的主体额度账户
     */
    private transient String tradeIssueBodyAcctNo;


    //合作客户名称,页面显示
    private transient String customerName;


    //是否能被 当前登录用户执行划拨操作 【申请单列表,判定登录用户是否有"划拨额度"操作权限】
    private transient Boolean hasAppropriatePerm;

    /**
     * 是否能撤回：申请列表页面 控制是否能看见 撤回按钮
     */
    private transient Boolean canRecall;


    /**
     * 登录账号是否 是该申请单的拥有者,即是申请创建者
     */
    private transient Boolean owner;

    /**
     * 当前审批节点 审批人账号类型
     */
    private transient QuotaApplyNodeApproverTypeEnum approverType;


    /**
     * 是否能被当前登录人撤回 ,只有刚申请 未被审批过的申请单,才可以撤回
     *
     * @return
     */
    public boolean isAllowRecallByLoginUser() {
        if (canRecall == null) {
            return false;
        }

        if (OperationConstant.DictQuotaTradeType.QUOTA_APPLY.equalsIgnoreCase(tradeType)) {
            //额度申请
            if (owner == null) {
                return false;
            }
            return canRecall && owner;

        } else {
            //划拨收回申请
            return canRecall;
        }
    }


    /**
     * 是否能被当前登录人编辑申请单
     *
     * @return
     */
    public boolean isAllowEditByLoginUser() {
        if (OperationConstant.DictQuotaTradeType.QUOTA_APPLY.equalsIgnoreCase(tradeType)) {
            //额度申请
            if (owner == null) {
                return false;
            }

            return isModifyAllowed() && owner;

        } else {
            //划拨收回申请
            return isModifyAllowed();
        }
    }


    /**
     * 是否能被当前登录人 作废 申请单
     *
     * @return
     */
    public boolean isAllowInvalidByLoginUser() {
        if (OperationConstant.DictQuotaTradeType.QUOTA_APPLY.equalsIgnoreCase(tradeType)) {
            //额度申请
            if (owner == null) {
                return false;
            }

            return OperationConstant.DictQuotaApplyStatus.DRAFT.equals(status) && owner;

        } else {
            //划拨收回申请
            return OperationConstant.DictQuotaApplyStatus.DRAFT.equals(status);
        }
    }


    /**
     * 是否能被当前登录人 删除 申请单
     *
     * @return
     */
    public boolean isAllowDeleteByLoginUser() {
        boolean statusAllowRemoved =
                OperationConstant.DictQuotaApplyStatus.INVALID.equalsIgnoreCase(status)
                        || OperationConstant.DictQuotaApplyStatus.AUDIT_FAILED.equalsIgnoreCase(status)
                        || OperationConstant.DictQuotaApplyStatus.APPROVE_FAILED.equalsIgnoreCase(status);

        if (OperationConstant.DictQuotaTradeType.QUOTA_APPLY.equalsIgnoreCase(tradeType)) {
            //额度申请
            if (owner == null) {
                return false;
            }

            return statusAllowRemoved && owner;

        } else {
            //划拨收回申请
            return statusAllowRemoved;
        }

    }

}
