package com.rocogz.util;


import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.*;

public class DateUtil {
    /**
     * 定义常量
     **/
    public static final String YYYY_MM = "yyyyMM";
    public static final String YYYY__MM = "yyyy-MM";
    public static final String YYYY_MM_DD_HH_MM_SS = "yyyy-MM-dd HH:mm:ss";
    public static final String YYYY_MM_DD = "yyyy-MM-dd";
    public static final String YYYYMMDD = "yyyyMMdd";
    public static final String DATE_KEY_STR = "yyMMddHHmmss";
    public static final String YYYY_MM_DD_HH_MM = "yyyy-MM-dd HH:mm";
    public static final String TIME_ZONE = "GMT+08:00";
    public static final String DEFAULT_DATE = "2099-12-31"; //默认日期
    public static final String START_TIME = " 00:00:00";//开始时间
    public static final String END_TIME = " 23:59:59"; //结束时间

    /**
     * 使用预设格式提取字符串日期
     *
     * @param strDate 日期字符串
     * @return
     */
    public static Date parseToDateTime(String strDate) {
        return parse(strDate, YYYY_MM_DD_HH_MM_SS);
    }

    /**
     * 使用预设格式提取字符串日期
     *
     * @param strDate 日期字符串
     * @return
     */
    public static Date parseToDate(String strDate) {
        return parse(strDate, YYYY_MM_DD);
    }

    /**
     * 使用用户格式提取字符串日期
     *
     * @param strDate 日期字符串
     * @param pattern 日期格式
     * @return
     */
    public static Date parse(String strDate, String pattern) {
        SimpleDateFormat df = new SimpleDateFormat(pattern);
        try {
            return df.parse(strDate);
        } catch (ParseException e) {
            e.printStackTrace();
            return null;
        }
    }

    /**
     * 比较指定时间与当前时间的大小
     *
     * @param date 指定时间
     * @return 指定时间大于当前时间返回正数 相等返回0  小于返回负数
     */
    public static int compareDateWithNow(Date date) {
        return compareDate(date, new Date());
    }

    /**
     * 两个date类型比较
     *
     * @param date1 date1
     * @param date2 date2
     * @return date1大于date2返回正数，dateValidate ，小于返回负数
     */
    public static int compareDate(Date date1, Date date2) {
        int rnum = date1.compareTo(date2);
        return rnum;
    }

    /**
     * 将指定的日期转换成Unix时间戳
     *
     * @param date 需要转换的日期 yyyy-MM-dd HH:mm:ss
     * @return long 时间戳
     */
    public static long format(String date) {
        long timestamp = 0;
        try {
            timestamp = new SimpleDateFormat(YYYY_MM_DD_HH_MM_SS).parse(date).getTime();
        } catch (ParseException e) {
            e.printStackTrace();
        }
        return timestamp;
    }

    /**
     * 将指定的日期转换成Unix时间戳
     *
     * @param date    需要转换的日期字符串
     * @param pattern 传入的指定日期字符串的格式
     * @return long 时间戳
     */
    public static long format(String date, String pattern) {
        long timestamp = 0;
        try {
            timestamp = new SimpleDateFormat(pattern).parse(date).getTime();
        } catch (ParseException e) {
            e.printStackTrace();
        }
        return timestamp;
    }

    /**
     * 将指定的日期转换成指定格式的时间字符串
     *
     * @param date    需要转换的日期
     * @param pattern 指定日期字符串格式
     * @return String 时间戳
     */
    public static String format(Date date, String pattern) {
        String timestamp = "0";
        try {
            timestamp = new SimpleDateFormat(pattern).format(date);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return timestamp;
    }

    /**
     * 将LocalDate类型转换成所需要格式的字符串
     * @param date 需要转换的日期 LocalDate
     * @param pattern 指定日期字符串格式
     * @return 日期字符串
     */
    public static String format(LocalDate date, String pattern){

        String timestamp = "0";
        try{
            timestamp = DateTimeFormatter.ofPattern(pattern).format(date);
        }catch (Exception e){
            e.printStackTrace();
        }
        return timestamp;
    }

    /**
     * 将当前日期转换成Unix时间戳
     *
     * @return long 时间戳
     */
    public static long dateToUnixTimestamp() {
        long timestamp = System.currentTimeMillis();
        return timestamp;
    }

    /**
     * 根据字符串格式,日期字符串
     *
     * @param formatStr 格式化参数
     * @param dateStr   日期
     * @return
     */
    public static String formatDate(String dateStr, String formatStr) {
        return new SimpleDateFormat(formatStr).format(parseToDate(dateStr));
    }

    public static LocalDateTime formatDateStrToLocalDateTime(String dateStr){
        DateTimeFormatter df = DateTimeFormatter.ofPattern(DateUtil.YYYY_MM_DD_HH_MM_SS);
        return LocalDateTime.parse(dateStr, df);
    }

    /**
     * 格式化localDateTime为string
     * @param localDateTime
     * @return
     */
    public static String formatLocalDateTimeToString(LocalDateTime localDateTime){
        DateTimeFormatter df = DateTimeFormatter.ofPattern(DateUtil.YYYY_MM_DD_HH_MM_SS);
        return df.format(localDateTime);
    }


    /**
     * 根据传的字符串返回所传日期是这个月的第几周
     * @param date
     * @return
     */
    public static int getWeekCntByDate(String date){
        return getWeekCntByDate(LocalDate.parse(date));
    }

    /**
     * 获取指定日期所属周
     * @param localDate
     * @return
     */
    public static int getWeekCntByDate(LocalDate localDate){
        String dateStr = localDate.format(DateTimeFormatter.ofPattern("yyyy-MM-dd"));
        int currentDay = localDate.getDayOfMonth();
        int days = localDate.withMonth(localDate.getMonth().getValue()).lengthOfMonth();
        String firstDate = dateStr.substring(0,7)+"-01";//获取每月的第一天
        LocalDate firstLocalDate = LocalDate.parse(firstDate);
        int firstDayWeek = firstLocalDate.getDayOfWeek().getValue();
        String endDate = dateStr.substring(0,7)+"-"+days;//获取每月的最后一天
        LocalDate endLocalDate = LocalDate.parse(endDate);
        int endDayWeek = endLocalDate.getDayOfWeek().getValue();
        int weekCnt = (days-(7-firstDayWeek+1)-endDayWeek)/7+2;//每个月减去第一周和最后一周计算中间日期周数，加2周为当月周数
        Map<Integer,Integer> weekDayMap = new HashMap<>();
        weekDayMap.put(1,7-firstDayWeek+1);//1周的周数，截止天数
        for (int i = 2; i <weekCnt; i++) {
            weekDayMap.put(i,(7-firstDayWeek+1)+(i-1)*7);//中间周的周数，截止天数
        }
        weekDayMap.put(weekCnt,days);//最后的周数，截止天数
        Iterator<Integer> iterator = weekDayMap.keySet().iterator();
        while (iterator.hasNext()){
            Integer week = iterator.next();
            int weekDayRange = weekDayMap.get(week);
            if(currentDay<=weekDayRange){
                return week;
            }
        }

        return -1;
    }

    /**
     * 根据所传日期,返回该日期所属第几周的字符串
     *     如: 2018年8月3日,是该月的第一周,则返回20180801
     * @param localDate
     * @return
     */
    public static String getWeekNumIdByDate(LocalDate localDate){
        String dateStr = localDate.format(DateTimeFormatter.ofPattern("yyyyMM"));
        return dateStr + "0" + String.valueOf(getWeekCntByDate(localDate));
    }

    /**
     * LocalDate -> Date
     *
     * @param date LocalDate
     * @return
     */
    public static Date localDateToDate(LocalDate date) {
        if (date == null) {
            return null;
        }
        return Date.from(date.atStartOfDay().atZone(ZoneId.systemDefault()).toInstant());
    }

    /**
     * 日期是否有效
     * @param startDate 开始日期
     * @param endDate   结束日期
     * @return true-有效（在保或有效）， false-（不在保或无效）
     */
    public static boolean dateValidate(String startDate, String endDate){
        Integer startInt = compareDate(localDateToDate(LocalDate.now()),  parseToDate(startDate));
        Integer endInt = compareDate(localDateToDate(LocalDate.now()),  parseToDate(endDate));
        if(startInt != -1 && (endInt == -1 || endInt == 0)){
            return true;
        }
        return false;
    }

    /**
     * 日期是否大于系统当前日期
     * @param startDate 开始日期
     * @param endDate   结束日期
     * @return 0-未生效，1-已生效， 2-已失效
     */
    public static int dateStatus(String startDate, String endDate){
        Integer startInt = compareDate(localDateToDate(LocalDate.now()),  parseToDate(startDate));
        Integer endInt = compareDate(localDateToDate(LocalDate.now()),  parseToDate(endDate));
        if(startInt == -1 && endInt == -1){
            return 0;
        }else if(startInt != -1 && (endInt == -1 || endInt == 0)){
            return 1;
        }else{
            return 2;
        }
    }
    /**
     * 根据字符串格式化时间
     * @param date 时间
     * @param pattern 格式化字符串
     * @return
     */
    public static LocalDate formatLocalDateByPattern(String date, String pattern){
        if (date == null) {
            return null;
        }
        return LocalDate.parse(date, DateTimeFormatter.ofPattern(pattern));
    }


    /**
     * 获取指定日期所在周的周一
     *
     * @param date 指定日期 为null 则但会当前日期所在周的周一
     * @return 返回周一
     */
    public static String getFirstDayOfWeek(Date date) {
        Calendar calendar = Calendar.getInstance();
        if (date != null) {
            calendar.setTime(date);
        }
        int offset;
        int dayOfWeek = calendar.get(Calendar.DAY_OF_WEEK);
        if (dayOfWeek == 1) {
            offset = -6;
        } else {
            offset = 2 - dayOfWeek;
        }
        calendar.add(Calendar.DAY_OF_WEEK, offset);
        return format(calendar.getTime(),YYYY_MM_DD);
    }

    /**
     * 获取指定日期所在周的周日
     * @param date 指定日期，为null则返回当前日期所在周日
     * @return 返回周日
     */
    public static String getLastDayOfWeek(Date date) {
        Calendar calendar = Calendar.getInstance();
        if (date != null) {
            calendar.setTime(date);
        }
        int offset;
        int dayOfWeek = calendar.get(Calendar.DAY_OF_WEEK);
        if (dayOfWeek == 1) {
            offset = 0;
        } else {
            offset = 7 - dayOfWeek + 1;
        }
        calendar.add(Calendar.DAY_OF_WEEK, offset);
        return format(calendar.getTime(),YYYY_MM_DD);
    }

    /**
     * 获取某个月份 最后一天 例如传入 2014-03 则返回 2014-03-01
     * @param yearMonth 月份 yyyy-MM
     * @return
     */
    public static String getFirstDayOfMonth(String yearMonth) {
        return yearMonth + "-01";
    }

    /**
     * 获取某个月份 最后一天 例如传入 2014-03 则返回 2014-03-31
     *
     * @param yearMonth 月份 yyyy-MM
     * @return
     */
    public static String getLastDayOfMonth(String yearMonth) {
        Date date = parse(yearMonth, YYYY__MM);
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        cal.add(Calendar.MONTH, 1);
        cal.set(Calendar.DAY_OF_MONTH, 1);
        cal.add(Calendar.DAY_OF_MONTH, -1);
        Date lastDayOfMonth = cal.getTime();
        return format(lastDayOfMonth,YYYY_MM_DD);
    }

}