package com.rocogz.syy.weixin.config;

import cn.binarywang.wx.miniapp.config.impl.WxMaDefaultConfigImpl;
import com.rocogz.redis.RedisDistributedLock;
import com.rocogz.redis.RedisService;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.locks.Lock;

/**
 * @author
 */
public class WxMaRedisConfig extends WxMaDefaultConfigImpl {

  private RedisService redisService;

  private static final String ACCESS_TOKEN = "accessToken";
  private static final String JSAPI_TICKET = "jsapiTicket";
  private static final String CARD_API_TICKET = "cardApiTicket";

  private static final String HASH_VALUE_FIELD = "value";
  private static final String HASH_EXPIRE_FIELD = "expire";

  /**
   * Redis Key 的前缀，默认为 maConfig
   */
  private String redisKeyPrefix = "maConfig";

  /**
   * 微信小程序唯一id，用于拼接存储到redis时的key，防止key重复.
   */
  private String maId;

  private Lock accessTokenLock;
  private Lock jsapiTicketLock;
  private Lock cardApiTicketLock;

  public WxMaRedisConfig(RedisService redisService) {
    this.redisService = redisService;
  }

  private String getRedisKey(String key) {
    StringBuilder redisKey = new StringBuilder(redisKeyPrefix).append(":");
    if (maId == null) {
      return redisKey.append(key).toString();
    } else {
      return redisKey.append(maId).append(":").append(key).toString();
    }
  }

  /**
   * 获取Redis缓存的值
   * @param key
   * @return
   */
  private String getValueFromRedis(String key) {
    return this.redisService.hget(getRedisKey(key), HASH_VALUE_FIELD);
  }

  private void setValueToRedis(String key, long expiresTime, String value) {
    Map<String, String> hash = new HashMap<String, String>();
    hash.put(HASH_VALUE_FIELD, value);
    hash.put(HASH_EXPIRE_FIELD, String.valueOf(expiresTime));

    this.redisService.hsetAll(getRedisKey(key), hash);
  }

  private long getExpireFromRedis(String key) {
    String expire = this.redisService.hget(getRedisKey(key), HASH_EXPIRE_FIELD);
    return expire == null ? 0 : Long.parseLong(expire);
  }

  private void setExpire(String key, long expiresTime) {
    this.redisService.hset(getRedisKey(key), HASH_EXPIRE_FIELD, String.valueOf(expiresTime));
  }

  public void setRedisKeyPrefix(String redisKeyPrefix) {
    this.redisKeyPrefix = redisKeyPrefix;
  }

  public void setMaId(String maId) {
    this.maId = maId;
  }

  @Override
  public String getAccessToken() {
    return getValueFromRedis(ACCESS_TOKEN);
  }

  @Override
  public Lock getAccessTokenLock() {
    if (accessTokenLock == null) {
      synchronized (this) {
        if (accessTokenLock == null) {
          accessTokenLock = new RedisDistributedLock(getRedisKey("accessTokenLock"), redisService);
        }
      }
    }
    return accessTokenLock;
  }

  @Override
  public boolean isAccessTokenExpired() {
    return isExpired(getExpireFromRedis(ACCESS_TOKEN));
  }

  @Override
  public synchronized void updateAccessToken(String accessToken, int expiresInSeconds) {
    setValueToRedis(ACCESS_TOKEN, expiresAheadInMillis(expiresInSeconds), accessToken);
  }

  @Override
  public String getJsapiTicket() {
    return getValueFromRedis(JSAPI_TICKET);
  }

  @Override
  public Lock getJsapiTicketLock() {
    if (jsapiTicketLock == null) {
      synchronized (this) {
        if (jsapiTicketLock == null) {
          jsapiTicketLock = new RedisDistributedLock(getRedisKey("jsapiTicketLock"), redisService);
        }
      }
    }
    return jsapiTicketLock;
  }

  @Override
  public boolean isJsapiTicketExpired() {
    return isExpired(getExpireFromRedis(JSAPI_TICKET));
  }

  @Override
  public void expireJsapiTicket() {
    setExpire(JSAPI_TICKET, 0);
  }

  @Override
  public void updateJsapiTicket(String jsapiTicket, int expiresInSeconds) {
    setValueToRedis(JSAPI_TICKET, expiresAheadInMillis(expiresInSeconds), jsapiTicket);
  }


  @Override
  public String getCardApiTicket() {
    return getValueFromRedis(CARD_API_TICKET);
  }

  @Override
  public Lock getCardApiTicketLock() {
    if (cardApiTicketLock == null) {
      synchronized (this) {
        if (cardApiTicketLock == null) {
          cardApiTicketLock = new RedisDistributedLock(getRedisKey("cardApiTicketLock"), redisService);
        }
      }
    }
    return cardApiTicketLock;
  }

  @Override
  public boolean isCardApiTicketExpired() {
    return isExpired(getExpireFromRedis(CARD_API_TICKET));
  }

  @Override
  public void expireCardApiTicket() {
    setExpire(CARD_API_TICKET, 0);
  }

  @Override
  public void updateCardApiTicket(String cardApiTicket, int expiresInSeconds) {
    setValueToRedis(CARD_API_TICKET, expiresAheadInMillis(expiresInSeconds), cardApiTicket);
  }

  @Override
  public void expireAccessToken() {
    setExpiresTime(0);
  }

  @Override
  public long getExpiresTime() {
    return getExpireFromRedis(ACCESS_TOKEN);
  }

  @Override
  public void setExpiresTime(long expiresTime) {
    setExpire(ACCESS_TOKEN, expiresTime);
  }
}
