package com.rocoinfo.aop.logger;

import com.rocoinfo.aop.logger.entity.LoggerEntity;
import com.rocoinfo.aop.logger.entity.Principal;
import com.rocoinfo.aop.logger.principal.PrincipalHandler;
import com.rocoinfo.aop.logger.utils.HttpUtils;
import com.rocoinfo.aop.logger.utils.LoggerJsonUtils;
import com.rocoinfo.aop.logger.utils.ServerUtils;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.annotation.PostConstruct;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * <dl>
 * <dd>Description: 日志切面</dd>
 * <dd>Company: 大城若谷信息技术有限公司</dd>
 * <dd>@date：2017/5/17 上午11:48</dd>
 * <dd>@author：Aaron</dd>
 * </dl>
 */
@Aspect
@Component
public class LoggerAspect {

    /**
     * 获取当前登录用户的工具类
     */
    @Autowired(required = false)
    private PrincipalHandler principalHandler;

    /**
     * 应用信息
     */
    @Value("${aspect.logger.app:}")
    private String app;

    /**
     * 日志类的名称
     */
    @Value("${aspect.logger.name:com.rocoinfo.api}")
    private String loggerName;

    /**
     * 实际记录日志的logger
     */
    private org.slf4j.Logger slf4jLogger;

    /**
     * 默认的日志类名称
     */
    private static final String DEFALUT_LOGGER_NAME = "com.rocoinfo.api";

    @PostConstruct
    public void init() {
        // 创建slf4jLogger对象
        if (loggerName == null || "".equals(loggerName.trim())) {
            loggerName = DEFALUT_LOGGER_NAME;
        }
        slf4jLogger = LoggerFactory.getLogger(loggerName);
    }

    /**
     * 定义切点
     */
    @Pointcut("within(com.rocoinfo..*) && @annotation(com.rocoinfo.aop.logger.Logger)")
    public void logger() {
    }

    @Around("logger()")
    public Object around(ProceedingJoinPoint jp) throws Throwable {
        // 开始时间
        Long startTime = System.currentTimeMillis();
        LoggerEntity loggerEntity = this.buildBaseLoggerEntity(jp);
        // 执行方法
        Object result = null;
        // 请求是否成功
        boolean success = true;
        try {
            result = jp.proceed();
            return result;
        } catch (Throwable e) {
            loggerEntity.setType(LoggerType.ERROR_LOG)
                    .setSuccess(Boolean.FALSE)
                    .setError(e.getMessage())
                    .setTrace(this.getErrorTrace(e));
            success = false;
            throw e;
        } finally {
            // 方法执行的结束时间
            Long endTime = System.currentTimeMillis();
            loggerEntity.setResptime(endTime - startTime);

            // 如果方法执行失败，记录异常trace信息
            if (!success) {
                slf4jLogger.error(LoggerJsonUtils.toJson(loggerEntity));
            } else { // 发放执行成功 则记录方法执行结果
                loggerEntity.setSuccess(Boolean.TRUE).setResult(result);
            }

            // 通过slf4j记录日志
            // 之所以将记录日志的步骤放到finally中，是因为防止logger.info()方法抛异常，导致方法执行的结果不能返回
            if (slf4jLogger.isInfoEnabled()) {
                slf4jLogger.info(LoggerJsonUtils.toJson(loggerEntity));
            }
        }
    }

    /**
     * 获取异常的跟踪信息
     *
     * @param e 异常
     * @return
     */
    private String getErrorTrace(Throwable e) {
        if (e == null) {
            return "";
        }
        StringWriter sw = new StringWriter();
        PrintWriter writer = new PrintWriter(sw);
        e.printStackTrace(writer);
        return sw.toString();
    }

    /**
     * 构建日志实体对象
     *
     * @return 返回日志实体类
     */
    private LoggerEntity buildBaseLoggerEntity(ProceedingJoinPoint jp) {
        // 方法传入的参数
        Object[] args = jp.getArgs();
        // 获取方法的签名
        MethodSignature signature = (MethodSignature) jp.getSignature();
        // 获取方法参数名
        String[] paramNames = signature.getParameterNames();
        Method method = signature.getMethod();
        // 获取注解对象
        Logger logger = method.getAnnotation(Logger.class);
        String[] excludeParams = logger.excludeParams();
        // 构建方法请求的参数  key->value形式
        Map<String, Object> params = this.buildMethodParams(paramNames, args, excludeParams);

        //获取代理方法签名 格式： 包名.类名.方法名
        Class clazz = method.getDeclaringClass();
        String methodSignature = clazz.getName() + "." + method.getName();

        HttpServletRequest req = this.getHttpServletRequest();
        //获取当前登录用户名
        String username = this.getUsername();

        LoggerEntity loggerEntity = new LoggerEntity()
                .setType(LoggerType.API_LOG) // 日志类型
                .setValue(logger.value())
                .setApp(app) // 请求应用
                .setModule(logger.module()) // 请求模块
                .setIp(HttpUtils.getRequestIp(req)) // 请求ip
                .setUrl(HttpUtils.getRequestUrl(req)) // 请求url
                .setMethod(HttpUtils.getRequestMethod(req)) // 请求方式
                .setBrowser(HttpUtils.getRequestBrowser(req)) // 请求浏览器
                .setTimestamp(System.currentTimeMillis()) // 请求时间戳
                .setSignature(methodSignature) // 请求方法签名
                .setParams(params) // 请求参数
                .setUsername(username) // 请求用户
                .setServer(ServerUtils.getServerIp());

        return loggerEntity;
    }

    /**
     * 构建方法参数
     *
     * @param paramNames    参数名
     * @param paramValues   参数值
     * @param excludeParams 排除的参数名
     * @return 返回方法参数名->参数值的map
     */
    private Map<String, Object> buildMethodParams(String[] paramNames, Object[] paramValues, String[] excludeParams) {
        if (paramNames != null && paramValues != null) {
            int len = paramNames.length > paramValues.length ? paramValues.length : paramNames.length;
            Map<String, Object> params = new HashMap<>(len);
            for (int i = 0; i < len; i++) {
                if (!excludeParam(paramNames[i], paramValues[i], Arrays.asList(excludeParams))) {
                    params.put(paramNames[i], paramValues[i]);
                }
            }
            return params;
        }
        return new HashMap<>(0);
    }

    /**
     * 判断是否需要在日志中排除参数,需要排除返回true，否则返回false
     * eg:默认排除HttpServletRequest和HttpServletReponse
     *
     * @param paramName     参数名称
     * @param paramValue    参数值
     * @param excludeParams 排除的参数名称列表
     */
    private boolean excludeParam(String paramName, Object paramValue, List<String> excludeParams) {
        if (paramName == null || "".equals(paramName) || excludeParams.contains(paramName)
                || paramValue instanceof ServletRequest || paramValue instanceof ServletResponse) {
            return true;
        }
        return false;
    }

    /**
     * 获取Request
     *
     * @return
     */
    private HttpServletRequest getHttpServletRequest() {
        return ((ServletRequestAttributes) RequestContextHolder.currentRequestAttributes()).getRequest();
    }

    /**
     * 获取当前登录用户名
     *
     * @return
     */
    private String getUsername() {
        if (principalHandler != null) {
            Principal principal = principalHandler.getPrincipal();
            return principal == null ? "" : principal.getUsername();
        }
        return "";
    }
}
