package com.rocoinfo.rocomall;

import com.fasterxml.jackson.annotation.JsonFormat;
import com.rocoinfo.rocomall.common.service.ServiceException;
import com.rocoinfo.rocomall.service.impl.UploadService;
import com.rocoinfo.rocomall.utils.WebUtils;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.multipart.MultipartFile;

import java.io.File;
import java.util.HashMap;
import java.util.Map;

public class MyUploadService extends UploadService {

	private Logger logger = LoggerFactory.getLogger(MyUploadService.class);

	/**
	 * 生成缩略图初始化数据
	 * 
	 * @author： 张文山
	 * @创建时间：2015-9-1 下午6:03:42
	 */
	@Override
	protected Map<String, ThumbnailProperty[]> initThumbnailPropertyMap() {
		thumbnailPropertyMap = new HashMap<String, ThumbnailProperty[]>();
		// 商品缩略图属性
		ThumbnailProperty[] productThumbnailProperty = { new ThumbnailProperty(1200, 1200), new ThumbnailProperty(500, 400),
				new ThumbnailProperty(200, 150) };

		ThumbnailProperty[] brandThumbnailProperty = { new ThumbnailProperty(1200, 1200), new ThumbnailProperty(500, 400),
				new ThumbnailProperty(200, 150) };

		ThumbnailProperty[] catalogThumbnailProperty = { new ThumbnailProperty(1200, 1200), new ThumbnailProperty(500, 400),
				new ThumbnailProperty(200, 150) };

		ThumbnailProperty[] userLevelThumbnailProperty = { new ThumbnailProperty(1200, 1200), new ThumbnailProperty(500, 400),
				new ThumbnailProperty(200, 150) };
		thumbnailPropertyMap.put(UploadCategory.PRODUCT.getPath(), productThumbnailProperty);
		thumbnailPropertyMap.put(UploadCategory.BRAND.getPath(), brandThumbnailProperty);
		thumbnailPropertyMap.put(UploadCategory.CATALOG.getPath(), catalogThumbnailProperty);
		thumbnailPropertyMap.put(UploadCategory.USER_LEVEL.getPath(), userLevelThumbnailProperty);
		return thumbnailPropertyMap;
	}

	/**
	 * 上传图片校验
	 * 
	 * @author： 张文山
	 * @创建时间：2015-9-1 下午6:03:57
	 */
	@Override
	public void validateUploadFile(MultipartFile file, UploadCategory uploadCategory) {
		ImageTypeInfo imageTypeInfo = convertUploadCategoryToImageType(uploadCategory);
		if (imageTypeInfo == null) {
			logger.warn("图片类型【{}】没有配置属性,使用默认验证", uploadCategory.name());
			super.validateUploadFile(file, uploadCategory);
			return;
		}
		// 做自己的校验
		String ext = FilenameUtils.getExtension(file.getOriginalFilename());
		if (StringUtils.isBlank(ext)) {
			throw new ServiceException("上传文件扩展名不能为空");
		}

		if (file.getSize() > imageTypeInfo.getSize()) {
			throw new ServiceException("上传文件不能大于 " + FileUtils.byteCountToDisplaySize(imageTypeInfo.getSize()));
		}
	}

	private ImageTypeInfo convertUploadCategoryToImageType(UploadCategory uploadCategory) {
		try {
			return ImageTypeInfo.valueOf(uploadCategory.name());
		} catch (Exception e) {
			logger.warn("图片类型【{}】没有配置属性", uploadCategory.name());
			return null;
		}
	}

	@Override
	protected void moveFileAfter(File file, String tmpPath) {
		setRequestAndUserId(WebUtils.getRequest(), WebUtils.getLoggedUserId());
		super.moveFileAfter(file, tmpPath);
	}

	/**
	 * <dl>
	 * <dd>描述:图片类型信息</dd>
	 * <dd>公司: 大城若谷信息技术有限公司</dd>
	 * <dd>@创建时间：2015-9-1 下午5:44:08</dd>
	 * <dd>@author： 张文山</dd>
	 * </dl>
	 */
	@JsonFormat(shape = JsonFormat.Shape.OBJECT)
	public static enum ImageTypeInfo {
		// name 和UploadCategory 的一致
		PRODUCT(600, 500, 1024 * 1024 * 2, 6), BRAND(600, 500, 1024 * 1024 * 2, 6);
		private int width;// 图片宽度
		private int length;// 图片长度
		private int size;// 图片大小 字节
		private int maxCount;// 最多多少张

		ImageTypeInfo(int width, int length, int size, int maxCount) {
			this.width = width;
			this.length = length;
			this.size = size;
			this.maxCount = maxCount;
		}

		public int getWidth() {
			return width;
		}

		public int getLength() {
			return length;
		}

		public int getSize() {
			return size;
		}

		public int getMaxCount() {
			return maxCount;
		}

		public static Map<String, ImageTypeInfo> convertToMap() {
			Map<String, ImageTypeInfo> map = new HashMap<String, MyUploadService.ImageTypeInfo>();
			for (ImageTypeInfo type : ImageTypeInfo.values()) {
				map.put(type.name(), type);
			}
			return map;
		}
	}

}
