package com.rocoinfo.rocomall.remote;

import com.rocoinfo.rocomall.common.BaseController;
import com.rocoinfo.rocomall.dto.StatusDto;
import com.rocoinfo.rocomall.entity.ProdComment;
import com.rocoinfo.rocomall.entity.Product;
import com.rocoinfo.rocomall.entity.account.User;
import com.rocoinfo.rocomall.entity.order.OrderItem;
import com.rocoinfo.rocomall.entity.params.CommentRequestParams;
import com.rocoinfo.rocomall.service.order.IOrderItemService;
import com.rocoinfo.rocomall.service.product.IProdCommentService;
import com.rocoinfo.rocomall.service.product.ISkuService;
import com.rocoinfo.rocomall.utils.ObjectFormatUtils;
import com.rocoinfo.rocomall.utils.ObjectUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.Date;
import java.util.List;

/**
 * <dl>
 * <dd>Description: 商品评价相关</dd>
 * <dd>Company: 大城若谷信息技术有限公司</dd>
 * <dd>@date：2016/8/12 13:38</dd>
 * <dd>@author：Aaron</dd>
 * </dl>
 */
@RestController
@RequestMapping(value = "/remote")
public class CommentRemoteRestController extends BaseController {

    @Autowired
    private IProdCommentService prodCommentService;
    @Autowired
    private IOrderItemService orderItemService;
    @Autowired
    private ISkuService skuService;

    /**
     * 添加商品评价
     *
     * @param params 添加订单请求的参数
     * @return
     */
    @RequestMapping(value = "/addComment", method = RequestMethod.POST)
    public Object add(CommentRequestParams params) {
        if (params == null)
            return StatusDto.buildFailureStatusDto("传入的参数为空或者不符合接口要求");
        if (ObjectUtils.isAnyNull(params.getUserId(), params.getOrderId(), params.getContent(), params.getLevel()))
            return StatusDto.buildFailureStatusDto("缺失必要参数,请检查传入的参数是否符合要求");

        // 获取子订单信息(因为评论是关联子订单的)
        OrderItem item = orderItemService.getFirstByOrderId(params.getOrderId());
        if (item == null)
            return StatusDto.buildFailureStatusDto("查询不到此订单");

        if (!params.getUserId().equals(item.getUser().getId()))
            return StatusDto.buildFailureStatusDto("请确保该用户是此订单的下单者");

        Long skuId = item.getSku() == null ? 0l : item.getSku().getId();
        // 获取商品id
        Long productId = skuService.getById(skuId).getProduct().getId();

        ProdComment com = new ProdComment();
        com.setProduct(new Product(productId));
        com.setContent(params.getContent());
        com.setCreateTime(new Date());
        com.setLevel(params.getLevel());
        com.setOrderItem(item);
        com.setShow(ProdComment.Show.DISPLAY);
        com.setUser(new User(params.getUserId()));
        try {
            //修改订单状态，保存评价信息
            item.setStatus(OrderItem.Status.FINISH);
            orderItemService.update(item);
            prodCommentService.insert(com);
            return StatusDto.buildSuccessStatusDto("评论添加成功！");
        } catch (Exception e) {
            return StatusDto.buildDataFailureStatusDto("评论添加失败！");
        }
    }

    /**
     * 根据商品id查询商品评价(每页显示10条)
     *
     * @param productId 商品id
     * @param page      页数
     * @return
     */
    @RequestMapping(value = "/comment")
    public Object findByProductId(@RequestParam() Long productId,
                                  @RequestParam(required = false, defaultValue = "0") Long page) {
        if (productId == null)
            return StatusDto.buildFailureStatusDto("商品id不能为空");
        List<ProdComment> list = this.prodCommentService.findByProductIdScrollPage(productId, page);
        Object obj = ObjectFormatUtils.format2Map(list,
                "id->id", "name->user.name", "content->content", "level->level", "createTime->createTime");
        return StatusDto.buildDataSuccessStatusDto("操作成功", obj);
    }
}
