package com.rocoinfo.rocomall.remote;

import com.google.common.collect.Maps;
import com.rocoinfo.rocomall.common.BaseController;
import com.rocoinfo.rocomall.dto.StatusDto;
import com.rocoinfo.rocomall.entity.Description;
import com.rocoinfo.rocomall.entity.Product;
import com.rocoinfo.rocomall.entity.Product.ShowStatus;
import com.rocoinfo.rocomall.service.product.IDescriptionService;
import com.rocoinfo.rocomall.service.product.IProductService;
import com.rocoinfo.rocomall.utils.MapUtils;
import com.rocoinfo.rocomall.utils.ObjectFormatUtils;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * <dl>
 * <dd>Description: 商品相关接口</dd>
 * <dd>Company: 大城若谷信息技术有限公司</dd>
 * <dd>@date：2016/8/12 15:01</dd>
 * <dd>@author：Aaron</dd>
 * </dl>
 */
@RestController
@RequestMapping(value = "/remote")
public class ProductRemoteRestController extends BaseController {

    @Autowired
    private IProductService productService;
    @Autowired
    private IDescriptionService descriptionService;


    /**
     * 查询商品列表
     *
     * @param keyword 查询条件
     * @param client  显示平台
     * @param page    分页(默认每页10条)
     * @return
     */
    @RequestMapping(value = "/products")
    public Object list(@RequestParam(required = false) ShowStatus client,
                       @RequestParam(required = false) String keyword,
                       @RequestParam(required = false, defaultValue = "0") int page) {
        Map<String, Object> params = Maps.newHashMap();
        MapUtils.putNotNull(params, "showStatus", client);
        MapUtils.putNotNull(params, "keyword", keyword);

        // 查询商品信息(带主图信息)
        List<Product> list = this.productService.searchWithMainImg(params, page);

        Object obj = ObjectFormatUtils.format2Map(list,
                "id->id", "name->name", "promotionWord->promotionWord", "price->cashAmt", "image->mainImage.path");

        return StatusDto.buildDataSuccessStatusDto("操作成功", obj);
    }

    /**
     * 查询商品详情
     *
     * @param productId 商品id
     * @return
     */
    @RequestMapping(value = "/product")
    public Object get(Long productId) {
        if (productId == null)
            return StatusDto.buildFailureStatusDto("商品id不能为空");

        Product product = this.productService.getById(productId);
        if (product == null) {
            return StatusDto.buildFailureStatusDto("查询不到此商品");
        }
        this.productService.buildDetail(product);
        // 由于工具类功能不够强大 此处格式化三次
        Object prodObj = ObjectFormatUtils.format2Map(product, "id->id", "name->name", "promotionWord->promotionWord");
        // 处理sku和图片信息
        if (prodObj instanceof HashMap) {
            HashMap prodMap = (HashMap) prodObj;
            if (CollectionUtils.isNotEmpty(product.getSkus())) {
                Object skuObj = ObjectFormatUtils.format2Map(product.getSkus(),
                        "id->id", "code->code", "name->name", "attribute1->attribute1", "attribute2->attribute2", "attribute3->attribute2", "availableStockQuantity->availableStockQuantity", "cashAmt->cashAmt", "attributes->attributes");
                prodMap.put("skus", skuObj);
            }
            if (CollectionUtils.isNotEmpty(product.getProductImages())) {
                Object skuObj = ObjectFormatUtils.format2Map(product.getProductImages(),
                        "id->id", "imageType->imageType", "imageUrl->path");
                prodMap.put("productImages", skuObj);
            }
        }
        return StatusDto.buildDataSuccessStatusDto("操作成功", prodObj);
    }

    /**
     * 获取商品描述
     *
     * @param productId 商品id
     * @return
     */
    @RequestMapping(value = "/description")
    public Object description(Long productId) {
        if (productId == null)
            return StatusDto.buildFailureStatusDto("商品id不能为空");
        Description description = descriptionService.findByProductId(productId);
        Object obj = ObjectFormatUtils.format2Map(description, "introduction->introduction", "specification->specification");
        return StatusDto.buildDataSuccessStatusDto("操作成功", obj);
    }
}
