package com.rocoinfo.rocomall.remote;

import com.rocoinfo.rocomall.Constants;
import com.rocoinfo.rocomall.common.BaseController;
import com.rocoinfo.rocomall.dto.StatusDto;
import com.rocoinfo.rocomall.entity.account.User;
import com.rocoinfo.rocomall.service.IUserService;
import com.rocoinfo.rocomall.utils.ObjectFormatUtils;
import com.rocoinfo.rocomall.utils.PasswordUtil;
import com.rocoinfo.rocomall.utils.RandomTools;
import com.rocoinfo.rocomall.utils.WebUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.shiro.session.Session;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

/**
 * <dl>
 * <dd>Description: 用户相关接口</dd>
 * <dd>Company: 大城若谷信息技术有限公司</dd>
 * <dd>@date：2016/8/11 15:52</dd>
 * <dd>@author：Aaron</dd>
 * </dl>
 */
@RestController
@RequestMapping(value = "/remote")
public class UserRemoteRestController extends BaseController {

    @Autowired
    private IUserService userService;

    /**
     * 用户注册接口
     *
     * @param user       用户信息
     * @param verifyCode 短信验证码
     * @return
     */
    @RequestMapping(value = "/register", method = RequestMethod.POST)
    public Object register(User user, String verifyCode) {
        // 校验请求的信息
        StatusDto validation = this.validateRequestParam(user, verifyCode);
        // 校验不通过返回错误信息
        if ("0".equals(validation.getCode()))
            return validation;

        if (this.userService.getByUsername(user.getMobile()) != null) {
            return StatusDto.buildFailureStatusDto("手机号已经被注册");
        }

        user.setUsername(user.getMobile());
        user.setPlainPassword(user.getPassword());
        // 如果用户没有输入姓名 随机输入姓名
        if (StringUtils.isBlank(user.getName()))
            user.setName("_用户_" + RandomTools.getRandStr(6, 0));

        StatusDto res = this.userService.register(user);

        return res;
    }

    /**
     * 修改密码
     *
     * @param user       用户信息
     * @param verifyCode 短信验证码
     * @return
     */
    @RequestMapping(value = "/changePwd", method = RequestMethod.POST)
    public Object changePassword(User user, String verifyCode) {
        // 校验请求的信息
        StatusDto validation = this.validateRequestParam(user, verifyCode);
        // 校验不通过返回错误信息
        if ("0".equals(validation.getCode()))
            return validation;

        // 数据库中存的用户信息
        User original = this.userService.getByUsername(user.getMobile());
        original.setPlainPassword(user.getPassword());
        //密码加密
        PasswordUtil.entryptPassword(original);
        this.userService.update(original);

        return StatusDto.buildSuccessStatusDto("修改成功");
    }

    /**
     * 根据用户id查询用户信息
     *
     * @param userId 用户id
     * @return
     */
    @RequestMapping(value = "/userInfo")
    public Object getUserInfo(Long userId) {
        if (userId == null)
            return StatusDto.buildFailureStatusDto("用户id不能为空");
        User user = this.userService.getById(userId);
        if (user == null)
            return StatusDto.buildFailureStatusDto("系统无此用户信息");
        // format 返回的格式
        Object obj = ObjectFormatUtils.format2Map(user,
                "id->id", "name->name", "gender->gender", "idCard->idCard", "birthday->birthday", "mobile->mobilePhone", "imgUrl ->imgUrl");

        return StatusDto.buildDataSuccessStatusDto("登录成功", obj);
    }

    /**
     * 更新用户信息
     *
     * @param user 用户信息
     * @return
     */
    @RequestMapping(value = "/updateUser")
    public Object update(User user) {
        if (user == null || user.getId() == null)
            return StatusDto.buildFailureStatusDto("用户信息或者用户id为空");
        user.setUsername(null);
        user.setMobilePhone(null);
        user.setPassword(null);
        user.setSalt(null);
        this.userService.update(user);

        //查询最新的用户信息　返回给接口调用方
        User userInfo = this.userService.getById(user.getId());
        // format 返回的格式
        Object obj = ObjectFormatUtils.format2Map(userInfo,
                "id->id", "name->name", "gender->gender", "idCard->idCard", "birthday->birthday", "mobile->mobilePhone", "imgUrl ->imgUrl");

        return StatusDto.buildDataSuccessStatusDto("登录成功", obj);
    }

    /**
     * 用户提交的数据校验
     *
     * @param user       用户信息
     * @param verifyCode 短信验证码
     * @return
     */
    private StatusDto validateRequestParam(User user, String verifyCode) {
        if (user == null)
            return StatusDto.buildFailureStatusDto("用户信息为null");

        if (StringUtils.isBlank(user.getMobile())) {
            return StatusDto.buildFailureStatusDto("手机号不能为空");
        }

        String password = user.getPassword();
        String confirmPassword = user.getConfirmPassword();

        if (StringUtils.isAnyBlank(password, confirmPassword)) {
            return StatusDto.buildFailureStatusDto("密码不能为空");
        }

        if (!password.equals(confirmPassword)) {
            return StatusDto.buildFailureStatusDto("密码和重复密码不相同");
        }

        if (StringUtils.isBlank(verifyCode)) {
            return StatusDto.buildFailureStatusDto("短信验证码不能为空");
        }

        Session session = WebUtils.getSession();
        String sessionMobilePhone = (String) session.getAttribute(Constants.REGISTER_VCODE_MOBILE_PHONE);

        if (sessionMobilePhone == null) {
            return StatusDto.buildFailureStatusDto("短信验证码不正确或已过期");
        }

        if (!sessionMobilePhone.equals(user.getMobile())) {
            return StatusDto.buildFailureStatusDto("手机号必须是接收验证码的手机号");
        }

        if (!WebUtils.isSmsVCodeValid(verifyCode)) {
            return StatusDto.buildFailureStatusDto("短信验证码不正确或已过期");
        }
        return StatusDto.buildSuccessStatusDto();
    }

    /**
     * 绑定新手机号(绑定完之后需要修改用户的登录名)
     *
     * @param userId     用户id
     * @param mobile     新手机号
     * @param verifyCode 短信验证码
     * @return
     */
    @RequestMapping(value = "/bindMobile")
    public Object bindMobile(@RequestParam Long userId,
                             @RequestParam String mobile,
                             @RequestParam String verifyCode) {
        if (userId == null)
            return StatusDto.buildFailureStatusDto("用户id不为空");

        if (StringUtils.isBlank(mobile))
            return StatusDto.buildFailureStatusDto("手机号不能为空");

        if (StringUtils.isBlank(verifyCode))
            return StatusDto.buildFailureStatusDto("短信验证码不能为空");

        Session session = WebUtils.getSession();
        String sessionMobilePhone = (String) session.getAttribute(Constants.REGISTER_VCODE_MOBILE_PHONE);

        if (sessionMobilePhone == null)
            return StatusDto.buildFailureStatusDto("短信验证码不正确或已过期");

        if (!WebUtils.isSmsVCodeValid(verifyCode))
            return StatusDto.buildFailureStatusDto("短信验证码不正确或已过期");

        User user = new User(userId);
        user.setMobilePhone(mobile);
        user.setUsername(mobile);

        this.userService.update(user);

        return StatusDto.buildSuccessStatusDto("用户绑定手机成功");
    }

}
