package com.rocoinfo.rocomall.rest;

import com.rocoinfo.rocomall.common.service.ServiceException;
import com.rocoinfo.rocomall.utils.WebUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.dao.DuplicateKeyException;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.http.converter.HttpMessageNotReadableException;
import org.springframework.http.converter.HttpMessageNotWritableException;
import org.springframework.web.HttpRequestMethodNotSupportedException;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.context.request.WebRequest;
import org.springframework.web.servlet.mvc.method.annotation.ResponseEntityExceptionHandler;

import javax.validation.ConstraintViolationException;

/**
 * 处理Spring MVC捕捉到的异常，响应友好信息。
 * 
 * @author liuwei
 */
@SuppressWarnings("unchecked")
@ControllerAdvice
public class RestExceptionHandler extends ResponseEntityExceptionHandler {
	private Logger logger = LoggerFactory.getLogger(RestExceptionHandler.class);

	@ExceptionHandler(IllegalArgumentException.class)
	public ResponseEntity<?> handleIllegalArgumentException(IllegalArgumentException e) {
		return WebUtils.response400(e.getMessage());
	}

	@ExceptionHandler(ServiceException.class)
	public ResponseEntity<?> handleServiceException(ServiceException e) {
		return WebUtils.response400(e.getMessage());
	}

	@ExceptionHandler(DuplicateKeyException.class)
	public ResponseEntity<?> handle(DuplicateKeyException e) {
		logger.warn("DuplicateKeyException", e);
		return WebUtils.response400("DuplicateKeyException");
	}

	@ExceptionHandler(ConstraintViolationException.class)
	public ResponseEntity<?> handle(ConstraintViolationException ex, WebRequest request) {
		logger.warn("ConstraintViolationException", ex);
		return WebUtils.response400(ex.getLocalizedMessage());
	}

	@ExceptionHandler(Exception.class)
	public ResponseEntity<?> handle(Exception e) {
		logger.warn("Exception", e);
		return WebUtils.response500("服务器内部错误");
	}

	@Override
	protected ResponseEntity<Object> handleHttpMessageNotReadable(HttpMessageNotReadableException ex, HttpHeaders headers, HttpStatus status, WebRequest request) {
		logger.debug("查询参数或请求体内容不合法", ex);
		return WebUtils.response400("查询参数或请求体内容不合法");
	}

	@Override
	protected ResponseEntity<Object> handleHttpRequestMethodNotSupported(HttpRequestMethodNotSupportedException ex, HttpHeaders headers, HttpStatus status, WebRequest request) {
		logger.debug("不支持请求方法", ex);
		return WebUtils.response400("不支持请求方法");
	}

	@Override
	protected ResponseEntity<Object> handleHttpMessageNotWritable(HttpMessageNotWritableException ex, HttpHeaders headers, HttpStatus status, WebRequest request) {
		logger.warn("输出响应内容时发生错误", ex);
		return WebUtils.response500("输出响应内容时发生错误");
	}
}