package com.rocoinfo.rocomall.rest.admin.order;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.rocoinfo.rocomall.common.BaseController;
import com.rocoinfo.rocomall.dto.StatusDto;
import com.rocoinfo.rocomall.entity.Sku;
import com.rocoinfo.rocomall.entity.order.Order;
import com.rocoinfo.rocomall.entity.order.Order.CheckState;
import com.rocoinfo.rocomall.entity.order.OrderDelivery;
import com.rocoinfo.rocomall.entity.order.OrderItem;
import com.rocoinfo.rocomall.entity.order.OrderItem.AdmStatus;
import com.rocoinfo.rocomall.service.IUserService;
import com.rocoinfo.rocomall.service.impl.product.SkuService;
import com.rocoinfo.rocomall.service.order.IDistributionOrderService;
import com.rocoinfo.rocomall.service.order.IOrderDeliveryService;
import com.rocoinfo.rocomall.service.order.IOrderItemService;
import com.rocoinfo.rocomall.service.order.IOrderService;
import com.rocoinfo.rocomall.utils.WebUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;
import java.util.Map;

/**
 * <dl>
 * <dd>描述:异常订单 处理</dd>
 * <dd>公司: 大城若谷信息技术有限公司</dd>
 * <dd>创建时间：2015年11月12日 下午7:18:35</dd>
 * <dd>创建人： 张敏</dd>
 * </dl>
 */
@RestController
@RequestMapping("/api/errorOrder")
public class AbnormalOrderRestController extends BaseController {
	private static final int OCCUPYSTOCK_REALSE = 1;
	private static final int OCCUPYLOCK_OCCUPED = 2;

	@Autowired
	private IOrderService orderService;
	@Autowired
	private IOrderItemService orderItemService;
	@Autowired
	private IUserService userService;
	@Autowired
	private IOrderDeliveryService orderDeliverService;
	@Autowired
	private SkuService skuService;
	@Autowired
	private IDistributionOrderService distributeService;

	//锁定或解锁 异常总订单
	@RequestMapping(value = "/switchLock/{orderId}")
	public Object lockOrUnlockOrder(@PathVariable Long orderId) {
		Order order = orderService.getById(orderId);
		if (order == null) {
			return StatusDto.buildFailureStatusDto("锁定/解锁的订单不存在");
		} else if (CheckState.NORMAL.equals(order.getCheckState())) {
			return StatusDto.buildFailureStatusDto("正常订单不能做“锁定/解锁”操作");
		}

		//(1)如果从来没有点击过“锁定”按钮的订单， 由 "未锁定" =>“已锁定”，释放占用库存 <br/>
		//(2)锁定状态：由“已锁定” =>"未锁定",占用库存不变 <br/>
		if (order.getLocked()) {
			//“取消锁定”操作 
			Order unlockOrder = new Order();
			unlockOrder.setId(orderId);
			unlockOrder.setLocked(Boolean.FALSE);
			orderService.update(unlockOrder);

			OrderItem updateItemStatus = new OrderItem();
			updateItemStatus.setAdmStatus(AdmStatus.EXCEPTION);
			updateItemStatus.setOrder(unlockOrder);
			orderItemService.update(updateItemStatus);
			return StatusDto.buildSuccessStatusDto("解锁成功");
		} else {
			//"锁定操作"
			String tipMsg = "锁定成功";
			if (!order.getClickedLockBtn()) {
				//如果是第一次锁定：释放占用库存
				tipMsg += "并释放占用库存";
				orderService.releaseOrOccupyStockByOrderId(orderId, OCCUPYSTOCK_REALSE);
			}

			Order lockOrder = new Order();
			lockOrder.setId(orderId);
			lockOrder.setClickedLockBtn(Boolean.TRUE);
			lockOrder.setLocked(Boolean.TRUE);
			orderService.update(lockOrder);

			OrderItem updateItemStatus = new OrderItem();
			updateItemStatus.setAdmStatus(AdmStatus.LOCKED);
			updateItemStatus.setOrder(lockOrder);
			orderItemService.update(updateItemStatus);

			return StatusDto.buildSuccessStatusDto(tipMsg);
		}
	}

	@RequestMapping(value = "/pass/{orderId}")
	public Object auditPass(@PathVariable Long orderId) {
		return executePass(orderId);
	}

	@RequestMapping(value = "/cancel/{orderId}")
	public Object cancel(@PathVariable Long orderId) {
		//将该订单的状态设置为“已取消”，如果没有点击过“锁定按钮”，则 释放占用库存
		Order order = orderService.getById(orderId);
		if (order == null) {
			return StatusDto.buildFailureStatusDto("要取消的订单不存在");
		} else if (order.getLocked()) {
			return StatusDto.buildFailureStatusDto("订单被锁定，不能取消");
		}

		boolean isReleaseOccupedStock = false;
		String msg = "订单取消成功";
		if (!order.getClickedLockBtn()) {
			//执行释放占用库存
			isReleaseOccupedStock = true;
			msg += ",并释放占用库存";
		}
		orderItemService.cancelOrder(orderId, WebUtils.getLoggedUserId(), isReleaseOccupedStock);
		return StatusDto.buildSuccessStatusDto(msg);
	}

	@RequestMapping(value = "/{orderId}")
	public Object get(@PathVariable Long orderId) {
		Order order = orderService.getById(orderId);
		if (order == null) {
			return StatusDto.buildFailureStatusDto("订单不存在,orderId:" + orderId);
		}
		order.setUser(userService.getById(order.getUser().getId()));
		order.setDelivery(orderDeliverService.getByOrderId(orderId));
		return order;
	}

	@RequestMapping(value = "/editSave")
	public Object edit(OrderDelivery delivery) {
		if (delivery == null || delivery.getId() < 1) {
			return StatusDto.buildFailureStatusDto("收获地址参数不合法");
		}
		//只需要保存：收货人、收货人手机、收货人地址；
		orderDeliverService.update(delivery);
		return StatusDto.buildSuccessStatusDto("收获地址信息保存成功");
	}

	@RequestMapping(value = "/editPass")
	public Object editAndPass(OrderDelivery delivery) {
		if (delivery == null || delivery.getId() < 1 || delivery.getOrderId() < 1) {
			return StatusDto.buildFailureStatusDto("参数id不合法");
		}

		if (!orderDeliverService.isValid(delivery)) {
			return StatusDto.buildFailureStatusDto("收货人或收货地址包含非法字符,导致订单为异常订单");
		}

		orderDeliverService.update(delivery);
		return executePass(delivery.getOrderId());
	}

	private Object executePass(Long orderId) {
		Order order = orderService.getById(orderId);
		if (order == null) {
			return StatusDto.buildFailureStatusDto("订单不存在。orderId:" + orderId);
		} else if (order.getLocked()) {
			return StatusDto.buildFailureStatusDto("订单已被锁定,不能通过");
		}

		List<OrderItem> orderItems = orderItemService.findByOrderId(orderId);

		if (order.getClickedLockBtn()) {
			List<Long> skuIds = Lists.newArrayList();
			Map<Long, Integer> skuQtyMap = Maps.newHashMap();
			for (OrderItem item : orderItems) {
				skuQtyMap.put(item.getSku().getId(), item.getQuantity());
				skuIds.add(item.getSku().getId());
			}
			List<Sku> skuList = skuService.findByIdIn(skuIds);
			for (Sku sku : skuList) {
				if (sku.getAvailableStockQuantity() < skuQtyMap.get(sku.getId())) {
					String error = "商品：" + sku.getName() + "可售库存不足,不能通过! 库存剩余：" + sku.getAvailableStockQuantity() + "件";
					return StatusDto.buildFailureStatusDto(error);
				}
			}
			//通过时占用上库存
			orderService.releaseOrOccupyStockByOrderId(orderId, OCCUPYLOCK_OCCUPED);
		}

		Order passOrder = new Order();
		passOrder.setCheckState(CheckState.NORMAL);
		passOrder.setDistributed(Boolean.TRUE);
		passOrder.setId(order.getId());
		orderService.update(passOrder);
		if (!order.getDistributed()) {
			order.setItems(orderItems);
			distributeService.generateDistributeOrder(order);
		}
		return StatusDto.buildSuccessStatusDto("异常订单审核通过");
	}
}
