package com.rocoinfo.rocomall.rest.admin.order;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.rocoinfo.rocomall.Constants;
import com.rocoinfo.rocomall.common.BaseController;
import com.rocoinfo.rocomall.dto.PageTable;
import com.rocoinfo.rocomall.dto.StatusDto;
import com.rocoinfo.rocomall.dto.admin.DistributeOrderExportToExcelDto;
import com.rocoinfo.rocomall.dto.admin.LogisticInfo;
import com.rocoinfo.rocomall.entity.dict.DictWarehouse;
import com.rocoinfo.rocomall.entity.dict.express.DictExpress;
import com.rocoinfo.rocomall.entity.order.DistributionOrder;
import com.rocoinfo.rocomall.entity.order.MergeDistributeOrder;
import com.rocoinfo.rocomall.entity.order.OrderDelivery;
import com.rocoinfo.rocomall.entity.order.OrderItem;
import com.rocoinfo.rocomall.entity.order.OrderItem.AdmStatus;
import com.rocoinfo.rocomall.entity.order.OrderItem.Status;
import com.rocoinfo.rocomall.service.IUploadService;
import com.rocoinfo.rocomall.service.dict.IDictWarehouseService;
import com.rocoinfo.rocomall.service.dict.express.IDictExpressService;
import com.rocoinfo.rocomall.service.order.IDistributionOrderService;
import com.rocoinfo.rocomall.service.order.IOrderItemService;
import com.rocoinfo.rocomall.utils.ExcelMergeRegion;
import com.rocoinfo.rocomall.utils.ExcelUtil;
import com.rocoinfo.rocomall.utils.StringEscapeEditor;
import com.rocoinfo.rocomall.utils.WebUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.propertyeditors.CustomDateEditor;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Sort;
import org.springframework.web.bind.ServletRequestDataBinder;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.text.SimpleDateFormat;
import java.util.*;

/**
 * <dl>
 * <dd>描述:配送单</dd>
 * <dd>公司: 大城若谷信息技术有限公司</dd>
 * <dd>创建时间：2015年11月12日 下午7:18:35</dd>
 * <dd>创建人： zhangmin</dd>
 * </dl>
 */
@RestController
@RequestMapping("/api/distributeOrder")
public class DistributionOrderRestController extends BaseController {

	@Autowired
	private IDistributionOrderService distributeOrderService;
	@Autowired
	private IDictExpressService expressService;
	@Autowired
	private IUploadService uploadService;
	@Autowired
	private IDictWarehouseService wareHouseService;
	@Autowired
	private IOrderItemService orderItemService;

	/**
	 * 配送单/拆单 列表
	 */
	@RequestMapping(method = RequestMethod.GET)
	public Object list(@RequestParam(required = false) String draw, @RequestParam(defaultValue = "0") int start, @RequestParam(defaultValue = "20") int length, @RequestParam(defaultValue = "id") String orderColumn,
		@RequestParam(defaultValue = "DESC") String orderSort, @RequestParam(required = false) String keyword, @RequestParam(required = false) Date startDate, @RequestParam(required = false) Date endDate,
		@RequestParam(required = false) Long expressId, @RequestParam(required = false) Boolean exported, @RequestParam(required = false) Boolean merged) {

		Map<String, Object> searchParams = Maps.newHashMap();
		//keyword 配送单号/订单号
		if (StringUtils.isNotBlank(keyword)) {
			searchParams.put("keyword", keyword.trim());
		}
		if (startDate != null) {
			searchParams.put("startDate", startDate);
		}
		if (endDate != null) {
			searchParams.put("endDate", endDate);
		}

		if (expressId != null) {
			searchParams.put("expressId", expressId);
		}

		if (exported != null) {
			searchParams.put("exported", exported);
		}

		if (merged != null) {
			searchParams.put("merged", merged);
		}

		PageRequest pageable = new PageRequest(start, length, new Sort(Sort.Direction.valueOf(orderSort.toUpperCase()), orderColumn));
		Page<DistributionOrder> page = null;
		page = this.distributeOrderService.searchScrollPage(searchParams, pageable);
		this.buildDictData(page.getContent());
		return new PageTable<DistributionOrder>(page.getContent(), draw, Integer.valueOf(page.getTotalElements() + ""));
	}

	/**
	 * 合单列表
	 */
	@RequestMapping(value = "mergeList", method = RequestMethod.GET)
	public Object mergeList(@RequestParam(required = false) String draw, @RequestParam(defaultValue = "0") int start, @RequestParam(defaultValue = "20") int length, @RequestParam(required = false) String keyword,
		@RequestParam(required = false) Long expressId) {
		Map<String, Object> searchParams = Maps.newHashMap();
		//keyword 配送单号/订单号
		if (StringUtils.isNotBlank(keyword)) {
			searchParams.put("keyword", keyword.trim());
		}
		if (expressId != null) {
			searchParams.put("expressId", expressId);
		}

		PageRequest pageable = new PageRequest(start, length);
		Page<MergeDistributeOrder> page = null;
		page = this.distributeOrderService.searchMergeDistributeOrder(searchParams, pageable);
		this.buildMergeOrderDictData(page.getContent());
		return new PageTable<MergeDistributeOrder>(page.getContent(), draw, Integer.valueOf(page.getTotalElements() + ""));
	}

	//普通/配送导出
	@RequestMapping(value = "/export", method = RequestMethod.GET)
	public void export(@RequestParam(required = false) String keyword, @RequestParam(required = false) Date startDate, @RequestParam(required = false) Date endDate, @RequestParam(required = false) Long expressId,
		@RequestParam(required = false) Boolean isDistributeExport, HttpServletResponse response) {

		Map<String, Object> searchParams = Maps.newHashMap();
		//keyword 配送单号/订单号
		if (StringUtils.isNotBlank(keyword)) {
			searchParams.put("keyword", keyword.trim());
		}

		if (startDate != null) {
			searchParams.put("startDate", startDate);
		}

		if (endDate != null) {
			searchParams.put("endDate", endDate);
		}

		if (expressId != null) {
			searchParams.put("expressId", expressId);
		}

		//isDistributeExport 是否配送导出
		if (Boolean.TRUE.equals(isDistributeExport)) {
			searchParams.put("exported", Boolean.FALSE);
		}

		int start = 0;
		//每次 导出 最多导出 5000条记录
		int maxLength = 5000;
		final String orderColumn = "merge_code";
		PageRequest pageable = new PageRequest(start, maxLength, new Sort(Sort.Direction.DESC, orderColumn));
		Page<DistributionOrder> page = this.distributeOrderService.searchScrollPage(searchParams, pageable);
		List<DistributionOrder> distriOrderList = page.getContent();
		this.buildDictData(distriOrderList);

		List<DistributeOrderExportToExcelDto> exportDataList = Lists.newArrayList();
		ExcelMergeRegion[] mergeRanges = null;

		if (CollectionUtils.isNotEmpty(distriOrderList)) {
			Map<String, ExcelMergeRegion> mergeRangeMap = Maps.newLinkedHashMap();
			final int mergeColIndex = 2;

			for (int rowIdx = 1; rowIdx <= distriOrderList.size(); rowIdx++) {
				DistributionOrder distriOrder = distriOrderList.get(rowIdx - 1);
				DistributeOrderExportToExcelDto toExcel = new DistributeOrderExportToExcelDto();
				String mergeCode = StringUtils.trimToEmpty(distriOrder.getMergeCode());
				//处理单元格合并
				if (!mergeCode.isEmpty()) {
					ExcelMergeRegion region = mergeRangeMap.get(mergeCode);
					if (region == null) {
						region = new ExcelMergeRegion();
						region.setFirstCol(mergeColIndex);
						region.setLastCol(mergeColIndex);
						region.setFirstRow(rowIdx);
						region.setLastRow(rowIdx);
						mergeRangeMap.put(mergeCode, region);
					} else {
						int lastRowIndex = region.getLastRow();
						lastRowIndex++;
						region.setLastRow(lastRowIndex);
					}
				}

				toExcel.setMergeCode(mergeCode);
				OrderItem item = distriOrder.getOrderItem();

				OrderDelivery delivery = item.getOrder().getDelivery();
				String address = delivery.getAddrPrefix() + delivery.getAddrDetail();
				toExcel.setAddress(address);
				toExcel.setDistriOrderCode(distriOrder.getCode());
				toExcel.setPostcode(delivery.getPostcode());
				toExcel.setRecipient(delivery.getRecipient());
				toExcel.setRecipientPhone(delivery.getRecipientPhone());
				if (distriOrder.hasExpress()) {
					toExcel.setExpressName(distriOrder.getExpress().getName());
				}

				toExcel.setOrderCreateTime(DateFormatUtils.format(item.getCreateTime(), Constants.YYYYMMDDHHMMSS));
				toExcel.setOrderItemCode(item.getOrderCode());
				toExcel.setQuantity(String.valueOf(item.getQuantity()));
				toExcel.setSkuCode(item.getSku().getCode());
				toExcel.setSkuName(item.getProductName());
				toExcel.setWareHouseName(item.getWareHouse().getName());
				exportDataList.add(toExcel);
			}

			if (MapUtils.isNotEmpty(mergeRangeMap)) {
				mergeRanges = new ExcelMergeRegion[mergeRangeMap.size()];
				Iterator<ExcelMergeRegion> iter = mergeRangeMap.values().iterator();
				int idx = 0;
				while (iter.hasNext()) {
					mergeRanges[idx] = iter.next();
					idx++;
				}
			}
		}

		//把订单数据写入到Excel文件中
		try {
			response.setContentType("application/vnd.ms-excel");
			response.setHeader("content-disposition", "attachment;filename=distributeOrder.xls");
			ExcelUtil.getInstance().exportObj2Excel(response.getOutputStream(), exportDataList, DistributeOrderExportToExcelDto.class, false, mergeRanges);

			if (Boolean.TRUE.equals(isDistributeExport)) {
				this.distributeOrderService.generateOutstoreOrder(distriOrderList, WebUtils.getLoggedUserId());
			}
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	@RequestMapping(value = "/downloadLogistic", method = RequestMethod.GET)
	public void downloadLogisticTemplate(HttpServletResponse response) {
		final String templateFileName = "logistic_template.xls";
		response.setContentType("application/vnd.ms-excel");
		response.setHeader("Content-Disposition", "attachment; filename=" + templateFileName);
		String templateClassPath = "template/" + templateFileName;
		InputStream in = Thread.currentThread().getContextClassLoader().getResourceAsStream(templateClassPath);
		OutputStream out = null;
		try {
			out = response.getOutputStream();
			IOUtils.copy(in, out);
		} catch (IOException e) {
			e.printStackTrace();
		} finally {
			IOUtils.closeQuietly(in);
			IOUtils.closeQuietly(out);
		}
	}

	/**
	 * 导入物流单信息
	 */
	@RequestMapping(value = "importLogistic", method = RequestMethod.POST)
	public Object importLogisticInfo(@RequestParam(required = true) String filePath) {
		//导入的数据包括：配送单号、快递公司编号、运单号,配送单对应的订单状态更新为“配送中”
		File file = this.uploadService.submitPathAndGetfile(filePath);
		if (!isExcelFile(file)) {
			return StatusDto.buildFailureStatusDto("文件格式不合法，请选择Excel文件");
		}

		try {
			List<LogisticInfo> logisticList = ExcelUtil.getInstance().readExcel2ObjsByFile(file, LogisticInfo.class);
			if (CollectionUtils.isNotEmpty(logisticList)) {
				Set<String> expressCodeSet = Sets.newHashSet();
				Map<String, DictExpress> expressMap = Maps.newHashMap();

				for (LogisticInfo logis : logisticList) {
					if (StringUtils.isNotBlank(logis.getExpressCode())) {
						expressCodeSet.add(logis.getExpressCode().trim());
					}
				}

				if (!expressCodeSet.isEmpty()) {
					expressMap = expressService.findExpressMapByCodesIn(Lists.newArrayList(expressCodeSet.iterator()));
				}

				List<DistributionOrder> updateList = Lists.newArrayList();

				for (LogisticInfo logis : logisticList) {
					if (StringUtils.isBlank(logis.getDistributeOrderCode()))
						continue;
					String distributeOrderCode = logis.getDistributeOrderCode().trim();
					DistributionOrder update = new DistributionOrder();
					update.setCode(distributeOrderCode);
					update.setTransportNo(logis.getTransportNo().trim());
					DictExpress exp = expressMap.get(logis.getExpressCode().trim());
					update.setExpress(exp);
					updateList.add(update);
				}
				this.distributeOrderService.importLogistics(updateList);
			}
		} catch (Exception e) {
			e.printStackTrace();
			return StatusDto.buildFailureStatusDto("导入失败，请检查文件内容");
		}
		return StatusDto.buildFailureStatusDto("物流单导入成功");
	}
	
	/**
	 * 签收导入
	 */
	@RequestMapping(value = "importSignLogistic", method = RequestMethod.POST)
	public Object importSignLogistic(@RequestParam(required = true) String filePath) {
		
		File file = this.uploadService.submitPathAndGetfile(filePath);
		if (!isExcelFile(file)) {
			return StatusDto.buildFailureStatusDto("文件格式不合法，请选择Excel文件");
		}
		
		try {
			List<LogisticInfo> logisticList = ExcelUtil.getInstance().readExcel2ObjsByFile(file, LogisticInfo.class);
			if (CollectionUtils.isNotEmpty(logisticList)) {
				List<DistributionOrder> updateList = Lists.newArrayList();
				for (LogisticInfo logis : logisticList) {
					if (StringUtils.isBlank(logis.getDistributeOrderCode()))
						continue;
					String distributeOrderCode = logis.getDistributeOrderCode().trim();
					DistributionOrder update = new DistributionOrder();
					update.setCode(distributeOrderCode);
					update.setTransportNo(logis.getTransportNo().trim());
					updateList.add(update);
				}
				distributeOrderService.updateOrderItemStatus(updateList, Status.FINISH, AdmStatus.FINISH);
			}
		} catch (Exception e) {
			e.printStackTrace();
			return StatusDto.buildFailureStatusDto("导入失败，请检查文件内容");
		}
		return StatusDto.buildFailureStatusDto("物流签收单导入成功");
	}

	/**
	 * 合并配送单
	 */
	@RequestMapping(value = "merge", method = RequestMethod.GET)
	public Object mergeDistributeOrder(@RequestParam String mergeCode, @RequestParam List<Long> distriOrderIdList) {
		if (StringUtils.isEmpty(mergeCode)) {
			return StatusDto.buildFailureStatusDto("合单操作：mergeCode不能为空");
		}

		if (distriOrderIdList.size() < 2) {
			return StatusDto.buildFailureStatusDto("合单，参数distriOrderIdList的size不能小于2");
		}
		//收货人、收货人地址、仓库、快递公司都相同且 未被配送导出,才能合并
		List<DistributionOrder> distriOrderList = this.distributeOrderService.findDistributeOrderWithItemAddrByIdIn(distriOrderIdList);
		if (distriOrderList.size() < 2) {
			return StatusDto.buildFailureStatusDto("传递的参数distriOrderIdList有误，没有找到能够合并的配送单记录");
		}

		DistributionOrder firstDistriOrder = distriOrderList.get(0);
		for (int i = 0; i < distriOrderList.size(); i++) {
			DistributionOrder destOrder = distriOrderList.get(i);
			if (destOrder.getExported()) {
				return StatusDto.buildFailureStatusDto("配送单(" + destOrder.getCode() + ")状态是“已配送导出”,不能做合并操作。");
			} else if (!destOrder.hasExpress()) {
				return StatusDto.buildFailureStatusDto("配送单(" + destOrder.getCode() + ")还没有导入物流信息,不能做合并操作。");
			}

			if (i >= 1) {
				if (!destOrder.getExpress().getId().equals(firstDistriOrder.getExpress().getId())) {
					return StatusDto.buildFailureStatusDto("配送单(" + destOrder.getCode() + ")快递公司不同,不能做合并操作。");
				}

				if (!destOrder.getOrderItem().getWareHouse().getId().equals(firstDistriOrder.getOrderItem().getWareHouse().getId())) {
					return StatusDto.buildFailureStatusDto("配送单(" + destOrder.getCode() + ")关联订单Item的仓库不同,不能做合并操作。");
				}

				if (!isSameDelivery(firstDistriOrder, destOrder)) {
					return StatusDto.buildFailureStatusDto("配送单(" + destOrder.getCode() + ")的收获地址信息不同,不能做合并操作。");
				}
			}
		}
		this.distributeOrderService.mergeOrSplit(distriOrderIdList, mergeCode);
		return StatusDto.buildSuccessStatusDto("配送单合并成功");
	}

	/**
	 * 合单页
	 */
	@RequestMapping(value = "mergeUI", method = RequestMethod.GET)
	public Object mergeUIDistriOrders(@RequestParam List<Long> distriOrderIdList) {
		if (CollectionUtils.isEmpty(distriOrderIdList) || distriOrderIdList.size() < 2) {
			return StatusDto.buildFailureStatusDto("参数distriOrderIdList的size不能小于2");
		}

		StatusDto<List<DistributionOrder>> statusDto = StatusDto.buildDataSuccessStatusDto();
		List<DistributionOrder> distrOrderList = this.distributeOrderService.findDistributeOrderWithItemAddrByIdIn(distriOrderIdList);
		this.buildDictData(distrOrderList);
		statusDto.setData(distrOrderList);
		return statusDto;
	}

	/**
	 * 拆单页
	 */
	@RequestMapping(value = "splitUI/{mergeCode}", method = RequestMethod.GET)
	public Object splitUIDistriOrders(@PathVariable String mergeCode) {
		if (StringUtils.isEmpty(mergeCode)) {
			return StatusDto.buildFailureStatusDto("参数mergeCode不能为空");
		}

		StatusDto<List<DistributionOrder>> statusDto = StatusDto.buildDataSuccessStatusDto();
		List<DistributionOrder> mergedDistrOrderList = this.distributeOrderService.findDistributeOrderWithItemAddrByIdIn(distributeOrderService.findDistributeOrderIdsByMergeCode(mergeCode));
		this.buildDictData(mergedDistrOrderList);
		statusDto.setData(mergedDistrOrderList);
		return statusDto;
	}

	/**
	 * 拆单
	 */
	@RequestMapping(value = "split", method = RequestMethod.GET)
	public Object splitDistributeOrder(@RequestParam List<Long> distriOrderIdList) {
		if (CollectionUtils.isEmpty(distriOrderIdList) || distriOrderIdList.size() < 2) {
			return StatusDto.buildFailureStatusDto("拆单，参数distriOrderIdList的size不能小于2");
		}
		this.distributeOrderService.mergeOrSplit(distriOrderIdList, StringUtils.EMPTY);
		//更新子订单状态
		List<DistributionOrder> distributionOrders = this.distributeOrderService.findDistributeOrderWithItemAddrByIdIn(distriOrderIdList);
		if (distributionOrders != null && distributionOrders.size() > 0) {
			List<Long> ids = new ArrayList<Long>();
			for (DistributionOrder order : distributionOrders) {
				ids.add(order.getOrderItem().getId());
			}
			orderItemService.batchUpdateStatus(OrderItem.Status.PAID, OrderItem.AdmStatus.DISTRIBUTION_WAIT_OUT, ids);
		}
		return StatusDto.buildSuccessStatusDto("拆单成功");
	}

	private boolean isExcelFile(File file) {
		String extName = FilenameUtils.getExtension(file.getName());
		if ("xls".equals(extName) || "xlsx".equals(extName)) {
			return true;
		}
		return false;
	}

	private boolean isSameDelivery(DistributionOrder sourceDistrOrder, DistributionOrder targetDistrOrder) {
		OrderDelivery sourceDelivery = sourceDistrOrder.getOrderItem().getOrder().getDelivery();
		OrderDelivery targeDelivery = targetDistrOrder.getOrderItem().getOrder().getDelivery();

		if (sourceDelivery == null) {
			return targeDelivery == null;
		}

		if (sourceDelivery.getRecipient().equals(targeDelivery.getRecipient()) && sourceDelivery.getProvince().equals(targeDelivery.getProvince()) && sourceDelivery.getCity().equals(targeDelivery.getCity())
			&& sourceDelivery.getCounty().equals(targeDelivery.getCounty()) && sourceDelivery.getAddrDetail().equals(targeDelivery.getAddrDetail())) {
			return true;
		}
		return false;
	}

	private void buildMergeOrderDictData(List<MergeDistributeOrder> mergeOrderList) {
		if (CollectionUtils.isNotEmpty(mergeOrderList)) {
			Map<Long, DictWarehouse> wareHouseMap = wareHouseService.findAllWarehouseMap();
			Map<Long, DictExpress> expressMap = expressService.findAllExpressMap();

			for (MergeDistributeOrder mergeOrder : mergeOrderList) {
				if (mergeOrder.getExpress() != null && mergeOrder.getExpress().getId() > 0) {
					mergeOrder.setExpress(expressMap.get(mergeOrder.getExpress().getId()));
				}
				mergeOrder.setWareHouse(wareHouseMap.get(mergeOrder.getWareHouse().getId()));
			}
		}
	}

	//构建快递、仓库 字典数据
	private void buildDictData(List<DistributionOrder> distriOrderList) {
		if (CollectionUtils.isNotEmpty(distriOrderList)) {
			Map<Long, DictWarehouse> wareHouseMap = wareHouseService.findAllWarehouseMap();
			Map<Long, DictExpress> expressMap = expressService.findAllExpressMap();
			for (DistributionOrder distrOrder : distriOrderList) {
				if (distrOrder.hasExpress()) {
					distrOrder.setExpress(expressMap.get(distrOrder.getExpress().getId()));
				}
				distrOrder.getOrderItem().setWareHouse(wareHouseMap.get(distrOrder.getOrderItem().getWareHouse().getId()));
			}
		}
	}

	@InitBinder
	protected void initBinder(HttpServletRequest request, ServletRequestDataBinder binder) throws Exception {
		binder.registerCustomEditor(Date.class, new CustomDateEditor(new SimpleDateFormat(Constants.YYYYMMDD), true));
		binder.registerCustomEditor(String.class, new StringEscapeEditor());
	}
}
