package com.rocoinfo.rocomall.rest.admin.order;

import com.google.common.collect.Maps;
import com.rocoinfo.rocomall.common.BaseController;
import com.rocoinfo.rocomall.dto.PageTable;
import com.rocoinfo.rocomall.dto.StatusDto;
import com.rocoinfo.rocomall.entity.account.AdminUser;
import com.rocoinfo.rocomall.entity.dict.DictWarehouse;
import com.rocoinfo.rocomall.entity.order.InstoreOrder;
import com.rocoinfo.rocomall.entity.order.InstoreOrder.ReceiveStatus;
import com.rocoinfo.rocomall.service.IUserService;
import com.rocoinfo.rocomall.service.dict.IDictWarehouseService;
import com.rocoinfo.rocomall.service.impl.product.SkuService;
import com.rocoinfo.rocomall.service.order.IInstoreOrderService;
import com.rocoinfo.rocomall.service.order.IOrderItemService;
import com.rocoinfo.rocomall.service.order.IOrderService;
import com.rocoinfo.rocomall.utils.CodeGenerator;
import com.rocoinfo.rocomall.utils.WebUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Sort;
import org.springframework.web.bind.annotation.*;

import java.util.Date;
import java.util.List;
import java.util.Map;

/**
 * <dl>
 * <dd>描述:入库单</dd>
 * <dd>公司: 大城若谷信息技术有限公司</dd>
 * <dd>创建时间：2015年11月12日 下午7:18:35</dd>
 * <dd>创建人： 张敏</dd>
 * </dl>
 */
@RestController
@RequestMapping("/api/instoreOrder")
public class InstoreOrderRestController extends BaseController {

	@Autowired
	private IOrderService orderService;
	@Autowired
	private IOrderItemService orderItemService;
	@Autowired
	private IUserService userService;
	@Autowired
	private SkuService skuService;
	@Autowired
	private IInstoreOrderService instoreOrderService;
	@Autowired
	private CodeGenerator codeGenerator;
	@Autowired
	private IDictWarehouseService wareHouseService;

	@RequestMapping(value = "")
	public Object list(@RequestParam(required = false) String keyword, @RequestParam(required = false) Date startDate, @RequestParam(required = false) Date endDate, @RequestParam(required = false) InstoreOrder.InstoreType instoreType,
		@RequestParam(required = false) ReceiveStatus receiveState, @RequestParam(required = false) String draw, @RequestParam(defaultValue = "0") int start, @RequestParam(defaultValue = "20") int length,
		@RequestParam(defaultValue = "id") String orderColumn, @RequestParam(defaultValue = "DESC") String orderSort) {

		Map<String, Object> searchParams = Maps.newHashMap();

		if (StringUtils.isNotBlank(keyword)) {
			// 入库单号/子订单号/入库操作人
			searchParams.put("keyword", keyword);
		}

		if (instoreType != null) {
			searchParams.put("instoreType", instoreType);
		}

		if (startDate != null) {
			searchParams.put("startDate", startDate);
		}

		if (endDate != null) {
			searchParams.put("endDate", endDate);
		}

		if (receiveState != null) {
			searchParams.put("receiveState", receiveState);
		}

		PageRequest pageable = new PageRequest(start, length, new Sort(Sort.Direction.valueOf(orderSort.toUpperCase()), orderColumn));
		Page<InstoreOrder> page = this.instoreOrderService.searchScrollPage(searchParams, pageable);
		this.buildDictWareHouse(page.getContent());
		return new PageTable<InstoreOrder>(page.getContent(), draw, Integer.valueOf(page.getTotalElements() + ""));
	}

	/**
	 * 点击确认收货,退货完成
	 */
	@RequestMapping(value = "/{id}/finish")
	public Object completeRefund(@PathVariable Long id) {
		InstoreOrder instoreOrder = this.instoreOrderService.getById(id);
		if (instoreOrder == null) {
			return StatusDto.buildFailureStatusDto("入库单不存在,id:" + id);
		}

		if (ReceiveStatus.FINISHED.equals(instoreOrder.getReceiveState())) {
			return StatusDto.buildFailureStatusDto("已确认收货的入库单无需再次确认收货");
		}
		try {
			this.instoreOrderService.refundConfirmReceive(instoreOrder);
			return StatusDto.buildSuccessStatusDto("确认收货成功");
		} catch (Exception e) {
			return StatusDto.buildFailureStatusDto("系统异常,退货失败!");
		}
	}

	/**
	 * 采购入库
	 */
	@RequestMapping(value = "/purchase", method = RequestMethod.POST)
	public Object instore(InstoreOrder instoreOrder) {
		if (instoreOrder == null) {
			return StatusDto.buildFailureStatusDto("系统异常,入库失败!");
		}
		if (instoreOrder.getSku() == null) {
			return StatusDto.buildFailureStatusDto("入库商品不能为空");
		}
		if (instoreOrder.getQuantity() == null || instoreOrder.getQuantity() <= 0) {
			return StatusDto.buildFailureStatusDto("入库数量必须是大于0的整数!");
		}

		instoreOrder.setOperator(new AdminUser(WebUtils.getLoggedUserId()));
		instoreOrder.setCode(codeGenerator.generateInstoreOrderCode());
		instoreOrder.setInType(InstoreOrder.InstoreType.PURCHASE);
		instoreOrder.setReceiveState(InstoreOrder.ReceiveStatus.FINISHED);
		instoreOrder.setCreateTime(new Date());
		try {
			this.instoreOrderService.insert(instoreOrder);
			return StatusDto.buildSuccessStatusDto("商品入库成功!");
		} catch (Exception e) {
			return StatusDto.buildFailureStatusDto("入库时,服务器发生错误");
		}
	}

	//构建仓库 字典数据
	private void buildDictWareHouse(List<InstoreOrder> instoreOrderList) {
		if (CollectionUtils.isNotEmpty(instoreOrderList)) {
			Map<Long, DictWarehouse> wareHouseMap = wareHouseService.findAllWarehouseMap();
			for (InstoreOrder instore : instoreOrderList) {
				instore.setWarehouse(wareHouseMap.get(instore.getWarehouse().getId()));
			}
		}
	}
}
