package com.rocoinfo.rocomall.rest.admin.order;

import com.google.common.collect.Lists;
import com.rocoinfo.rocomall.Constants;
import com.rocoinfo.rocomall.common.BaseController;
import com.rocoinfo.rocomall.common.service.ServiceException;
import com.rocoinfo.rocomall.dto.PageTable;
import com.rocoinfo.rocomall.dto.StatusDto;
import com.rocoinfo.rocomall.dto.admin.OrderBitchSendImportExcelDto;
import com.rocoinfo.rocomall.dto.admin.OrderBitchSendImportExcelFailDto;
import com.rocoinfo.rocomall.dto.admin.OrderExportExcelDto;
import com.rocoinfo.rocomall.dto.admin.OrderItemExchageRefundParamDto;
import com.rocoinfo.rocomall.entity.OrderItemOperationLog;
import com.rocoinfo.rocomall.entity.Sku;
import com.rocoinfo.rocomall.entity.account.AdminUser;
import com.rocoinfo.rocomall.entity.order.Order;
import com.rocoinfo.rocomall.entity.order.Order.CheckState;
import com.rocoinfo.rocomall.entity.order.OrderDelivery;
import com.rocoinfo.rocomall.entity.order.OrderItem;
import com.rocoinfo.rocomall.entity.order.OrderItem.Group;
import com.rocoinfo.rocomall.service.IAddressService;
import com.rocoinfo.rocomall.service.IUploadService;
import com.rocoinfo.rocomall.service.dict.express.IDictExpressService;
import com.rocoinfo.rocomall.service.impl.UploadService;
import com.rocoinfo.rocomall.service.impl.order.NewOrderService;
import com.rocoinfo.rocomall.service.order.IOrderItemService;
import com.rocoinfo.rocomall.service.order.IOrderService;
import com.rocoinfo.rocomall.service.product.IProductService;
import com.rocoinfo.rocomall.service.product.ISkuService;
import com.rocoinfo.rocomall.utils.ExcelUtil;
import com.rocoinfo.rocomall.utils.StringEscapeEditor;
import com.rocoinfo.rocomall.utils.WebUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.propertyeditors.CustomDateEditor;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Sort;
import org.springframework.web.bind.ServletRequestDataBinder;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.File;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.*;

/**
 * <dl>
 * <dd>描述: 订单管理</dd>
 * <dd>公司: 大城若谷信息技术有限公司</dd>
 * <dd>创建时间：15/9/7 上午9:11</dd>
 * <dd>创建人： weiys</dd>
 * </dl>
 * 
 * @author weiys
 * @author huyt
 * @author guokl
 */
@SuppressWarnings("all")
@RestController
@RequestMapping("/api/order")
public class OrderRestController extends BaseController {

	@Autowired
	private IOrderItemService orderItemService;
	@Autowired
	private IOrderService orderService;
	@Autowired
	public IAddressService addressService;
	@Autowired
	public IDictExpressService expressService;
	@Autowired
	private ISkuService skuService;
	@Autowired
	private IProductService productService;
	@Autowired
	private IUploadService uploadService;
	@Autowired
	private NewOrderService newOrderService;

	/**
	 * 显示订单列表
	 * 
	 * @param keyword 会员名/订单号/会员手机号
	 * @param startDate 订单起始时间(下单时间)
	 * @param endDate 订单结束时间(下单时间)
	 * @param supplierId 供应商
	 * @param status 订单状态
	 * @param orderType 订单类型
	 * @return
	 */
	@RequestMapping(method = RequestMethod.GET)
	public Object seach(@RequestParam(required = false) String draw, @RequestParam(defaultValue = "0") int start, @RequestParam(defaultValue = "20") int length, @RequestParam(defaultValue = "id") String orderColumn,
		@RequestParam(defaultValue = "DESC") String orderSort, @RequestParam(required = false) String keyword, @RequestParam(required = false) Date startDate, @RequestParam(required = false) Date endDate,
		@RequestParam(required = false) Long supplierId, @RequestParam(required = false) OrderItem.Status status, @RequestParam(required = false) Boolean orderLocked, @RequestParam(required = false) Order.CheckState checkState,
		@RequestParam(required = false) OrderItem.Group group, @RequestParam(required = false) OrderItem.OrderType orderType, @RequestParam(required = false) OrderItem.AdmStatus admStatus) {

		PageRequest pageable = new PageRequest(start, length, new Sort(Sort.Direction.valueOf(orderSort.toUpperCase()), orderColumn));
		List<Long> searchSupplierIds = WebUtils.getWhereClauseSupplierIds(supplierId);
		Page<OrderItem> page = null;
		page = this.orderItemService.searchScrollPage(this.buildSearchParamMap(searchSupplierIds, startDate, endDate, status, orderLocked, checkState, group, keyword, orderType, admStatus), pageable);
		return new PageTable<OrderItem>(page.getContent(), draw, Integer.valueOf(page.getTotalElements() + ""));
	}

	//查询某个订单中,能够退货的订单Item
	@RequestMapping(value = "/byCode", method = RequestMethod.GET)
	public Object getCanRefundOrderItems(@RequestParam(required = false) String draw, @RequestParam String orderCode) {
		List<OrderItem> orderItems = Collections.emptyList();
		//orderCode 总订单号
		if (StringUtils.isNotBlank(orderCode)) {
			Order order = this.orderService.getByOrderNo(orderCode);
			if (order != null) {
				//已完成状态 才能退货
				orderItems = orderItemService.findByOrderId(order.getId());
				for (Iterator<OrderItem> it = orderItems.iterator(); it.hasNext();) {
					if (!OrderItem.Status.FINISH.equals(it.next().getStatus())) {
						it.remove();
					}
				}
			}
		}
		return new PageTable<OrderItem>(orderItems, draw, orderItems.size());
	}

	/**
	 * 构建差选条件的Map
	 */
	private Map<String, Object> buildSearchParamMap(List<Long> paramSupplierIdList, final Date startDate, final Date endDate, final OrderItem.Status status, Boolean orderLocked, CheckState checkState, Group group, String keyword,
		final OrderItem.OrderType orderType, final OrderItem.AdmStatus admStatus) {
		keyword = StringUtils.trimToNull(keyword);
		Map<String, Object> params = new HashMap<String, Object>();

		if (CollectionUtils.isNotEmpty(paramSupplierIdList)) {
			params.put("supplierIds", paramSupplierIdList);
		}

		if (startDate != null) {
			params.put("startDate", startDate);
		}

		if (endDate != null) {
			params.put("endDate", endDate);
		}

		if (status != null) {
			params.put("status", status);
		}

		if (checkState != null) {
			params.put("checkState", checkState);
		}

		if (CheckState.EXCEPTION.equals(checkState) && orderLocked != null) {
			params.put("locked", orderLocked);
		}

		if (group != null) {
			params.put("group", group);
		}

		if (keyword != null) {
			params.put("keyword", Arrays.asList(StringUtils.splitByWholeSeparator(keyword, " ")));
		}
		if (orderType != null) {
			params.put("orderType", orderType);
		}
		if (admStatus != null) {
			params.put("admStatus", admStatus);
		}
		return params;
	}

	/**
	 * 导出订单(Excel)
	 */
	@RequestMapping(value = "/export", method = RequestMethod.GET)
	public void export(@RequestParam(required = false) String keyword, @RequestParam(required = false) Date startDate, @RequestParam(required = false) Date endDate, @RequestParam(required = false) Long supplierId,
		@RequestParam(required = false) OrderItem.Status status, @RequestParam(required = false) Boolean orderLocked, @RequestParam(required = false) Order.CheckState checkState, @RequestParam(required = false) OrderItem.Group group,
		@RequestParam(required = false) OrderItem.OrderType orderType, @RequestParam(required = false) OrderItem.AdmStatus admStatus, HttpServletResponse response) {

		final String orderTemplate = "template/order_template.xls";
		try {
			final String orderExcelFileName = java.net.URLEncoder.encode("order", Constants.DEFAULT_ENCODING);
			response.setContentType("application/vnd.ms-excel");
			response.setHeader("content-disposition", "attachment;filename=" + orderExcelFileName + ".xls");

			List<OrderExportExcelDto> orderExcelDataList = Lists.newArrayList();

			List<Long> whereClauseSupplierIds = WebUtils.getWhereClauseSupplierIds(supplierId);
			// TODO 由于暂时管理员没有关联供应商 所以暂时注释掉下面的代码,否则查不出数据
			// if (CollectionUtils.isNotEmpty(whereClauseSupplierIds)) {
			if (true) {
				Map<String, Object> params = this.buildSearchParamMap(whereClauseSupplierIds, startDate, endDate, status, orderLocked, checkState, group, keyword, orderType, admStatus);
				// EXCEL表 sheet最多65531条数据,所以限制limitSize为6万
				params.put("limitSize", 60000);
				List<OrderItem> orderItemList = this.orderItemService.adminExportSearch(params);

				// 导出完毕后，更新状态为支付的订单为已处理,并且清除订单的相关cache
				if (CollectionUtils.isNotEmpty(orderItemList)) {

					// 订单状态为“已支付” && 有收货地址的订单，更新为已处理(用于记录对应id)
					final List<Long> needToDealedOrderItemIds = Lists.newArrayList();

					// List<Address> addressLists = Lists.newArrayList();
					// for (final OrderItem orderItem : orderItemList) {
					// Address addr = orderItem.getOrder().getAddress();
					// if (addr != null) {
					// addressLists.add(addr);
					// }
					// }
					// 加载收货地址的省市县
					// this.addressService.buildProvCityCounty(addressLists);

					StringBuilder addrBuf = new StringBuilder();

					for (final OrderItem orderItem : orderItemList) {
						final OrderExportExcelDto orderDto = new OrderExportExcelDto();
						// 订单编号
						orderDto.setOrderNo(orderItem.getOrderCode());
						Sku sku = orderItem.getSku();
						if (sku != null) {
							// 商品编码
							orderDto.setProdCode(sku.getCode());
						}
						// 商品名称
						orderDto.setProdName(orderItem.getProductName());
						// 商品数量
						orderDto.setQuantity(orderItem.getQuantity());
						// 订单状态
						orderDto.setStatus(orderItem.getStatus().getLabel());
						// 下单时间
						orderDto.setPlaceOrderDateTime(DateFormatUtils.format(orderItem.getCreateTime(), Constants.YYYYMMDDHHMMSS));
						OrderDelivery addr = orderItem.getOrder().getDelivery();
						if (addr != null) {
							// 收货人姓名
							orderDto.setReceiverName(addr.getRecipient());
							// 收货人电话
							orderDto.setReceiverMobile(addr.getRecipientPhone());
							// 收货人邮编
							orderDto.setPostCode(addr.getPostcode());
							// 收货人地址
							addrBuf.append(addr.getProvince()).append(",").append(addr.getCity()).append(",").append(addr.getCounty()).append(",");
							addrBuf.append(addr.getAddrDetail());
							orderDto.setAddress(addrBuf.toString());

							// 清空stringBuffer
							addrBuf.delete(0, addrBuf.length());
							// 记录状态为已支付&&有收货地址的订单,导出完毕之后更新为已处理
							if (OrderItem.Status.PAID == orderItem.getStatus()) {
								needToDealedOrderItemIds.add(orderItem.getId());
							}
						}
						orderExcelDataList.add(orderDto);
					}
					// 处理需要更新状态的订单
					this.orderItemService.dealWithOrderItems(true, new AdminUser(WebUtils.getLoggedUserId()), needToDealedOrderItemIds.toArray(new Long[]{}));
				}
			}
			ExcelUtil.getInstance().exportObj2ExcelByTemplate((Map<String, String>) null, orderTemplate, response.getOutputStream(), orderExcelDataList, OrderExportExcelDto.class, Boolean.TRUE);
		} catch (Exception ex) {
			ex.printStackTrace();
		}
	}

	/**
	 * 获取订单项详细信息
	 * 
	 * @param orderItemId 订单项id
	 * @return
	 */
	@RequestMapping(value = "/{orderItemId}", method = RequestMethod.GET)
	public Object get(@PathVariable Long orderItemId) {

		// 查询指定订单项的信息
		OrderItem orderItem = this.orderItemService.getById(orderItemId);

		// 构建订单项详情
		this.orderItemService.buildDetail(orderItem);

		// 查询订单项操作日志
		List<OrderItemOperationLog> operationLogs = this.orderItemService.findOperationLogByOrderItemId(orderItem.getId());

		Map map = new HashMap(2);
		map.put("orderItem", orderItem);
		map.put("operationLogs", operationLogs);

		StatusDto result = StatusDto.buildDataSuccessStatusDto();
		result.setData(map);

		return result;
	}

	/**
	 * 订单处理(订单)
	 */
	@RequestMapping(value = "/deal/{orderItemId}", method = RequestMethod.GET)
	public Object deal(@PathVariable Long orderItemId) {

		OrderItem orderItem = this.orderItemService.getById(orderItemId);
		this.orderItemService.buildDetail(orderItem);
		// Address address = orderItem.getOrder().getAddress();
		// if (null == address && !orderItem.getSku().getProduct().getVisual())
		// {
		// return StatusDto.buildFailureStatusDto("订单地址不能为空");
		// }
		// 状态为已支付的订单可以处理
		this.orderItemService.dealWithOrderItems(false, new AdminUser(WebUtils.getLoggedUserId()), orderItemId);
		return StatusDto.buildSuccessStatusDto("订单处理成功");
	}

	/**
	 * 批量发货导入
	 * 
	 * @param filePath 上传之后的文件路径
	 * @param type 导入类型
	 * @return
	 */
	@RequestMapping(value = "/import", method = RequestMethod.POST)
	public Object importToSend(@RequestParam(required = true) String filePath, @RequestParam(required = false) UploadService.UploadCategory type, HttpServletResponse response) {
		File file = this.uploadService.submitPathAndGetfile(filePath);
		List<OrderBitchSendImportExcelDto> sendOrderList = ExcelUtil.getInstance().readExcel2ObjsByFile(file, OrderBitchSendImportExcelDto.class);
		StatusDto result = this.orderItemService.importToSend(sendOrderList, new AdminUser(WebUtils.getLoggedUserId()));
		if (result.isSuccess())
			return result;
		else {
			Map<String, Object> resMap = (Map<String, Object>) result.getData();
			List<OrderBitchSendImportExcelFailDto> canNotUpdateDtos = (List<OrderBitchSendImportExcelFailDto>) resMap.get("canNotUpdateDtos");
			List<OrderBitchSendImportExcelDto> canUpdateDtos = (List<OrderBitchSendImportExcelDto>) resMap.get("canUpdateDtos");
			response.setContentType("application/vnd.ms-excel");
			try {
				ExcelUtil.getInstance().exportObj2Excel(response.getOutputStream(), canNotUpdateDtos, OrderBitchSendImportExcelFailDto.class, true);
			} catch (IOException e) {
				e.printStackTrace();
			} finally {
				return StatusDto.buildFailureStatusDto();
			}
		}
	}

	/**
	 * 在发货页点击“保存按钮” 发货
	 */
	@RequestMapping(value = "/send", method = RequestMethod.POST)
	public Object send(OrderItem orderItem) {
		// 校验收货地址(虚拟商品不需要收货地址)

		/*
		 * 发货逻辑： 订单状态为“DEALED(已处理)”，后台操作人员点击发货按钮，填写物流公司和快递单号，保存后订单变为配送中状态，<br/> 订单发货后 占用库存减少并且良品库存减少
		 */
		try {
			this.orderItemService.sendOrder(orderItem, WebUtils.getLoggedUserId());
		} catch (ServiceException ex) {
			return StatusDto.buildFailureStatusDto(ex.getMessage());
		}

		return StatusDto.buildSuccessStatusDto("发货成功");
	}

	/**
	 * 修改物流信息
	 */
	@RequestMapping(value = "/modifyExpress", method = RequestMethod.POST)
	public Object modifyExpress(OrderItem orderItem) throws Exception {
		try {
			this.orderItemService.updateExpressAndTransportNoById(orderItem.getExpress().getId(), orderItem.getTransportNo(), orderItem.getId());
			return StatusDto.buildSuccessStatusDto();
		} catch (Exception e) {
			return StatusDto.buildFailureStatusDto("保存失败!");
		}
	}

	/**
	 * 在发货页点击“保存按钮” 发货
	 */
	@RequestMapping(value = "/repeatSend/{orderItemId}", method = RequestMethod.GET)
	public Object repeatSend(@PathVariable Long orderItemId) {
		try {
			this.orderItemService.repeatSendOrder(orderItemId);
		} catch (ServiceException ex) {
			return StatusDto.buildFailureStatusDto(ex.getMessage());
		}
		return StatusDto.buildSuccessStatusDto("重新发货成功");
	}

	/**
	 * 取消订单
	 * 
	 * @param orderItemId
	 * @return
	 */
	@RequestMapping(value = "/cancel/{orderItemId}", method = RequestMethod.GET)
	public Object cancel(@PathVariable Long orderItemId) {
		final OrderItem orderItem = this.orderItemService.getById(orderItemId);
		/*
		 * 取消订单逻辑： 订单状态为“PAID”，后台操作人员点击“取消”按钮，订单变为已取消状态，<br/> 退单之后会员的积分返还,并且 减少占用库存
		 */
		this.orderItemService.cancelOrderItem(orderItemId, WebUtils.getLoggedUserId());
		return StatusDto.buildSuccessStatusDto("订单取消成功");
	}

	/**
	 * 换货
	 */
	@RequestMapping(value = "/exchange", method = RequestMethod.POST)
	@ResponseBody
	public Object doExchage(@RequestBody OrderItemExchageRefundParamDto exchageParamDto) {
		try {
			this.orderItemService.exchangeSalesOrderItems(exchageParamDto);
		} catch (Exception e) {
			this.logger.error(e.getMessage());
			return StatusDto.buildFailureStatusDto(e.getMessage());
		}
		return StatusDto.buildSuccessStatusDto("换货操作成功");
	}

	/**
	 * 退货
	 */
	@RequestMapping(value = "/refund", method = RequestMethod.POST)
	public Object doRefund(@RequestBody OrderItemExchageRefundParamDto refundParamDto) {
		try {
			this.orderItemService.doRefundSalesOrderItems(refundParamDto);
		} catch (Exception e) {
			return StatusDto.buildFailureStatusDto(e.getMessage());
		}
		return StatusDto.buildSuccessStatusDto("退货操作成功");
	}

	/**
	 * 更新订单 最终状态 （包括：确认收货，已退货，完成换货）
	 */
	@RequestMapping(value = "/updateState/{orderItemId}", method = RequestMethod.GET)
	@ResponseBody
	public Object updateOrderFinalStatus(@PathVariable Long orderItemId, @RequestParam OrderItem.Status finalState, @RequestParam OrderItem.AdmStatus admStatus) {
		this.orderItemService.updateOrderState(orderItemId, finalState, admStatus, WebUtils.getLoggedUserId());
		return StatusDto.buildSuccessStatusDto("操作成功");
	}

	/**
	 * 下载发货模板
	 */
	@RequestMapping(value = "/download", method = RequestMethod.GET)
	public void downloadSendTemplate(HttpServletResponse response) {
		response.setContentType("application/vnd.ms-excel");
		response.setHeader("Content-Disposition", "attachment; filename=sendTemplate.xls");
		try {
			List<OrderBitchSendImportExcelDto> emptyList = Collections.emptyList();
			ExcelUtil.getInstance().exportObj2Excel(response.getOutputStream(), emptyList, OrderBitchSendImportExcelDto.class, false);
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	@InitBinder
	protected void initBinder(HttpServletRequest request, ServletRequestDataBinder binder) throws Exception {
		binder.registerCustomEditor(Date.class, new CustomDateEditor(new SimpleDateFormat(Constants.YYYYMMDD), true));
		binder.registerCustomEditor(String.class, new StringEscapeEditor());
	}
}
