package com.rocoinfo.rocomall.rest.admin.product;

import com.google.common.base.Optional;
import com.google.common.collect.Maps;
import com.rocoinfo.rocomall.Constants;
import com.rocoinfo.rocomall.dto.PageTable;
import com.rocoinfo.rocomall.dto.StatusDto;
import com.rocoinfo.rocomall.entity.ProdStockApply;
import com.rocoinfo.rocomall.entity.ProdStockApplyItem;
import com.rocoinfo.rocomall.entity.StockApprove;
import com.rocoinfo.rocomall.entity.account.AdminUser;
import com.rocoinfo.rocomall.entity.account.Role;
import com.rocoinfo.rocomall.entity.cent.Apply;
import com.rocoinfo.rocomall.entity.cent.Approve;
import com.rocoinfo.rocomall.service.product.IProdStockApplyService;
import com.rocoinfo.rocomall.utils.ClassToMapUtil;
import com.rocoinfo.rocomall.utils.CodeGenerator;
import com.rocoinfo.rocomall.utils.StringEscapeEditor;
import com.rocoinfo.rocomall.utils.WebUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.Predicate;
import org.apache.commons.lang3.StringUtils;
import org.apache.shiro.SecurityUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.propertyeditors.CustomDateEditor;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Sort;
import org.springframework.ui.Model;
import org.springframework.web.bind.ServletRequestDataBinder;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.Map;

/**
 * Created by huyt on 2015/9/1.
 * 商品出入库管理
 */
@SuppressWarnings("all")
@RestController
@RequestMapping(value = "/api/prodstockapp")
public class ProdStockAppRestController {


    @Autowired
    private IProdStockApplyService prodStockApplyService;


    /**
     * 商品出入库申请列表
     *
     * @return
     */
    @RequestMapping(method = RequestMethod.GET)
    public Object list(@RequestParam(required = false) String applyCode,
                       @RequestParam(required = false) ProdStockApply.StockType type,
                       @RequestParam(required = false) ProdStockApply.WareHouse warehouse,
                       @RequestParam(required = false) ProdStockApply.State state,
                       @RequestParam(required = false) String draw,
                       @RequestParam(defaultValue = "0") int start,
                       @RequestParam(defaultValue = "id") String orderColumn,
                       @RequestParam(defaultValue = "20") int length,
                       @RequestParam(defaultValue = "DESC") String orderSort,
                       Model model) {
        // 构建查询条件
        Map<String, Object> params = Maps.newHashMap();

        if (StringUtils.isNotBlank(applyCode)) {
            params.put("applyCodes", StringUtils.split(applyCode, " "));
        }

        if (type != null) {
            params.put("type", type);
        }

        if (warehouse != null) {
            params.put("warehouse", warehouse);
        }

        if (state != null) {
            params.put("state", state);
        }

        // 管理员之外
        if (!SecurityUtils.getSubject().hasRole(Role.ADMIN)) {
            params.put("userId", WebUtils.getLoggedUserId());
        }

        // 构建分页条件
        PageRequest pageRequest = new PageRequest(start, length, new Sort(Sort.Direction.valueOf(orderSort.toUpperCase()), orderColumn));

        Page<ProdStockApply> page = prodStockApplyService.searchScrollPage(params, pageRequest);

        List data = (List) ClassToMapUtil.dealSimple(page.getContent(), "id", "applyCode", "type", "warehouse", "applyer.name",
                "state", "applyDate");

        return new PageTable<ProdStockApply>(data, draw, Integer.valueOf(page.getTotalElements() + ""));
    }

    /**
     * 根据id查询 某一条记录
     *
     * @param id
     * @return
     */
    @RequestMapping(value = "/{id}")
    public Object get(@PathVariable Long id) {

        if (id == null)
            return StatusDto.buildFailureStatusDto("you has choose a wrong record!");

        // 判断查询出的记录是否为null
        Optional<ProdStockApply> optional = Optional.fromNullable(this.prodStockApplyService.getById(id));
        if (!optional.isPresent())
            return StatusDto.buildFailureStatusDto("can't find the record!");

        StatusDto statusDto = StatusDto.buildDataSuccessStatusDto();
        statusDto.setData(optional.get());
        return statusDto;
    }

    /**
     * 创建出入库申请单
     *
     * @param prodStockApply
     * @return
     */
    @RequestMapping(value = "/create/{state}", method = RequestMethod.POST)
    public Object create(ProdStockApply prodStockApply, @PathVariable Apply.State state) {

        //剔除mustache中加减行所造成的空行
        CollectionUtils.filter(prodStockApply.getProdStockApplyItems(), new Predicate() {
            @Override
            public boolean evaluate(Object object) {
                ProdStockApplyItem item = (ProdStockApplyItem) object;
                return item != null && item.getProd() != null;
            }
        });

        // 如果状态为草稿则不需要校验库存,如果直接提交审批,需要检查库存
        boolean checkFlag = false;
        if (Apply.State.DRAFT.equals(state)) {
            prodStockApply.setState(Apply.State.DRAFT);
        } else {
            if (ProdStockApply.StockType.OUT.equals(prodStockApply.getType())) {
                checkFlag = true;
            }
            prodStockApply.setState(Apply.State.WAIT_AUDIT);
        }

        // 设置出入库申请编码,申请人,申请日期
        prodStockApply.setApplyCode(CodeGenerator.generateStockApplyCode(ProdStockApply.StockType.IN.equals(prodStockApply.getType())));
        prodStockApply.setApplyer(new AdminUser(WebUtils.getLoggedUserId()));
        prodStockApply.setApplyDate(new Date());

        return this.prodStockApplyService.addOrUpdateStockApply(prodStockApply, checkFlag);
    }

    /**
     * 更新出入库申请单
     *
     * @param prodStockApply
     * @return
     */
    @RequestMapping(value = "/update", method = RequestMethod.POST)
    public Object update(ProdStockApply prodStockApply) {

        //剔除mustache中加减行所造成的空行
        CollectionUtils.filter(prodStockApply.getProdStockApplyItems(), new Predicate() {
            @Override
            public boolean evaluate(Object object) {
                ProdStockApplyItem item = (ProdStockApplyItem) object;
                return item != null && item.getProd() != null;
            }
        });

        // 更新完之后将状态更改为待审批
        prodStockApply.setState(Apply.State.WAIT_AUDIT);

        // 出入库申请人
        prodStockApply.setApplyer(new AdminUser(WebUtils.getLoggedUserId()));

        // 如果是出库 需要检查库存
        boolean checkFlag = false;
        if (ProdStockApply.StockType.OUT.equals(prodStockApply.getType()))
            checkFlag = true;

        return this.prodStockApplyService.addOrUpdateStockApply(prodStockApply, checkFlag);
    }

    /**
     * 审批申请单
     *
     * @param id     申请单id
     * @param result 申请结果
     * @return
     */
    @RequestMapping(value = "/approve/{id}/{result}", method = RequestMethod.GET)
    public Object approve(@PathVariable Long id, @PathVariable Approve.ApproveResult result) {

        // 查询相关申请单
        ProdStockApply prodStockApply = this.prodStockApplyService.getById(id);

        // 构建审批的JavaBean
        StockApprove approve = new StockApprove();
        approve.setApply(prodStockApply);
        approve.setApprover(new AdminUser(WebUtils.getLoggedUserId()));
        approve.setAuditDate(new Date());
        approve.setResult(result);

        return this.prodStockApplyService.auditStockAppoly(approve, result);
    }


    /**
     * 绑定日期转换的Converter
     *
     * @param request
     * @param binder
     * @throws Exception
     */
    @InitBinder
    protected void initBinder(HttpServletRequest request, ServletRequestDataBinder binder) throws Exception {
        binder.registerCustomEditor(Date.class, new CustomDateEditor(new SimpleDateFormat(Constants.YYYYMMDD), true));
        binder.registerCustomEditor(String.class, new StringEscapeEditor());
    }

}
