package com.rocoinfo.rocomall.rest.admin.product;

import com.google.common.base.Optional;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.rocoinfo.rocomall.Constants;
import com.rocoinfo.rocomall.dto.PageTable;
import com.rocoinfo.rocomall.dto.StatusDto;
import com.rocoinfo.rocomall.dto.admin.VirtualProductExcelDto;
import com.rocoinfo.rocomall.entity.Sku;
import com.rocoinfo.rocomall.entity.VirtualProduct;
import com.rocoinfo.rocomall.entity.VirtualProduct.Type;
import com.rocoinfo.rocomall.service.IUploadService;
import com.rocoinfo.rocomall.service.product.IProductService;
import com.rocoinfo.rocomall.service.product.ISkuService;
import com.rocoinfo.rocomall.service.product.IVirtualProductService;
import com.rocoinfo.rocomall.utils.ExcelUtil;
import com.rocoinfo.rocomall.utils.StringEscapeEditor;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.propertyeditors.CustomDateEditor;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Sort;
import org.springframework.web.bind.ServletRequestDataBinder;
import org.springframework.web.bind.annotation.*;
import org.springside.modules.utils.Collections3;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.File;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.*;

/**
 * <dl>
 * <dd>描述: 虚拟商品卡密管理Controller</dd>
 * <dd>公司: 大城若谷信息技术有限公司</dd>
 * <dd>创建时间：2015/9/5 17:35</dd>
 * <dd>创建人：huyt</dd>
 * </dl>
 */
@SuppressWarnings("all")
@RestController
@RequestMapping(value = "/api/virtualprod")
public class VirtualProductRestController {
	@Autowired
	private IVirtualProductService virtualProductService;
	@Autowired
	private ISkuService skuService;
	@Autowired
	private IProductService productService;
	@Autowired
	private IUploadService uploadService;

	/**
	 * 展示虚拟商品卡密列表
	 */
	@RequestMapping(method = RequestMethod.GET)
	public Object list(@RequestParam(required = false) String keyword, @RequestParam(required = false) VirtualProduct.Status status, @RequestParam(required = false) String draw, @RequestParam(defaultValue = "0") int start,
		@RequestParam(defaultValue = "id") String orderColumn, @RequestParam(defaultValue = "20") int length, @RequestParam(defaultValue = "DESC") String orderSort) {
		// 构建查询条件
		Map<String, Object> params = Maps.newHashMap();
		if (StringUtils.isNotBlank(keyword)) {
			params.put("keywords", StringUtils.split(keyword, " "));
		}

		if (status != null) {
			params.put("status", status);
		}
		PageRequest pageRequest = new PageRequest(start, length, new Sort(Sort.Direction.valueOf(orderSort.toUpperCase()), orderColumn));
		Page<VirtualProduct> page = this.virtualProductService.searchScrollPage(params, pageRequest);
		return new PageTable<VirtualProduct>(page.getContent(), draw, (int) page.getTotalElements());
	}

	/**
	 * 获取单条记录
	 */
	@RequestMapping(value = "/{id}")
	public Object get(@PathVariable Long id) {
		if (id == null || id < 1)
			return StatusDto.buildFailureStatusDto("参数Id不合法");

		// 判断查询出的记录是否为null
		Optional<VirtualProduct> optional = Optional.fromNullable(this.virtualProductService.getById(id));
		if (!optional.isPresent())
			return StatusDto.buildFailureStatusDto("can't find the VirtualProduct by id:" + id);

		StatusDto statusDto = StatusDto.buildDataSuccessStatusDto();
		statusDto.setData(optional.get());
		return statusDto;
	}

	/**
	 * 添加虚拟商品卡密
	 */
	@RequestMapping(value = "/create", method = RequestMethod.POST)
	public Object create(VirtualProduct virtualProduct) {
		// 需要先检查sku编码是否正确
		String skuCode = virtualProduct.getSkuCode();
		Sku sku = this.skuService.getByCode(skuCode);
		if (sku == null || !productService.getById(sku.getProduct().getId()).getVisual())
			return StatusDto.buildFailureStatusDto(String.format("code为%s的商品不存在或者不是虚拟商品", skuCode));

		if (Type.EXCHANGE_CODE.equals(virtualProduct.getType())) {
			//兑换码  没有卡号和序列号
			virtualProduct.setCardno(StringUtils.EMPTY);
			virtualProduct.setSerialNo(StringUtils.EMPTY);
		}

		// 商品序列号、卡号、密码不能同时重复
		if (isDuplicate(virtualProduct)) {
			return StatusDto.buildFailureStatusDto(String.format("code为%s的商品序列号、卡号、密码不能同时重复", skuCode));
		}
		virtualProduct.setStatus(VirtualProduct.Status.UNUSED);
		virtualProductService.insert(virtualProduct);
		skuService.increaseStockOnRefundIfCanUse(sku.getId(), 1);

		return StatusDto.buildSuccessStatusDto("虚拟商品添加成功");
	}

	/**
	 * 更新虚拟商品卡密信息
	 */
	@RequestMapping(value = "/update", method = RequestMethod.POST)
	public Object update(VirtualProduct virtualProduct) {
		// 需要先检查sku编码是否正确
		String skuCode = virtualProduct.getSkuCode();
		Sku sku = this.skuService.getByCode(skuCode);
		if (sku == null || !productService.getById(sku.getProduct().getId()).getVisual())
			return StatusDto.buildFailureStatusDto(String.format("code为%s的商品不存在或者不是虚拟商品", skuCode));

		VirtualProduct tempVirProd = virtualProductService.getById(virtualProduct.getId());

		//虚拟商品的类型不能修改
		virtualProduct.setType(tempVirProd.getType());

		if (isDuplicate(virtualProduct)) {
			return StatusDto.buildFailureStatusDto(String.format("code为%s的商品序列号、卡号、密码不能同时重复", skuCode));
		}

		if (Type.EXCHANGE_CODE.equals(virtualProduct.getType())) {
			virtualProduct.setCardno(null);
			virtualProduct.setSerialNo(null);
		}

		virtualProductService.update(virtualProduct);
		return StatusDto.buildSuccessStatusDto("更新成功!");
	}

	/**
	 * 虚拟卡密作废
	 */
	@RequestMapping(value = "destroy/{id}", method = RequestMethod.GET)
	public Object abolish(@PathVariable Long id) {
		// 如果状态不是未使用 不允许作废
		VirtualProduct virProduct = virtualProductService.getById(id);
		if (virProduct == null) {
			return StatusDto.buildFailureStatusDto("虚拟商品[" + id + "]不存在!");
		} else if (!VirtualProduct.Status.UNUSED.equals(virProduct.getStatus())) {
			return StatusDto.buildFailureStatusDto("只有尚未使用的虚拟卡才可以作废!");
		}

		VirtualProduct update = new VirtualProduct();
		update.setId(id);
		update.setStatus(VirtualProduct.Status.CANCEL);
		this.virtualProductService.update(update);
		return StatusDto.buildSuccessStatusDto("作废操作成功");
	}

	/**
	 * 批量导入虚拟商品卡密
	 */
	@RequestMapping(value = "/import")
	public Object batchImp(@RequestParam(required = true) String filePath) {
		File file = this.uploadService.submitPathAndGetfile(filePath);
		List<VirtualProductExcelDto> cardList = ExcelUtil.getInstance().readExcel2ObjsByFile(file, VirtualProductExcelDto.class);
		return importData(cardList);
	}

	private StatusDto<String> importData(List<VirtualProductExcelDto> cardList) {
		if (CollectionUtils.isEmpty(cardList)) {
			return StatusDto.buildSuccessStatusDto("excel中没有虚拟商品,无需导入");
		}

		List<VirtualProductExcelDto> canUpdateDtos = Lists.newArrayList();
		List<VirtualProductExcelDto> canNotUpdateDtos = Lists.newArrayList();

		int lineNumber = 1;
		Map<String, String> errorMsg = Maps.newLinkedHashMap();
		String lineTemp = "第%s行";
		for (VirtualProductExcelDto dto : cardList) {
			lineNumber++;
			// 校验虚拟商品是否存在
			String skuCode = dto.getSkuCode();
			Sku sku = this.skuService.getByCode(skuCode);
			if (sku == null || !productService.getById(sku.getProduct().getId()).getVisual()) {
				canNotUpdateDtos.add(dto);
				errorMsg.put(String.format(lineTemp, lineNumber), "商品编码不存在或者不是虚拟商品");
				continue;
			}
			try {
				VirtualProduct virtualProduct = new VirtualProduct();
				virtualProduct.setSkuCode(skuCode);
				virtualProduct.setPasswd(dto.getPasswd());
				virtualProduct.setType(VirtualProduct.Type.valueOfName(dto.getType()));

				if (Type.CARD_PWD.equals(virtualProduct.getType())) {
					virtualProduct.setCardno(dto.getCardno());
					virtualProduct.setSerialNo(dto.getSerialNo());
				} else {
					virtualProduct.setCardno(StringUtils.EMPTY);
					virtualProduct.setSerialNo(StringUtils.EMPTY);
				}

				virtualProduct.setExpired(java.sql.Date.valueOf(dto.getExpired()));
				virtualProduct.setStatus(VirtualProduct.Status.UNUSED);

				if (null == virtualProduct.getType()) {
					canNotUpdateDtos.add(dto);
					errorMsg.put(String.format(lineTemp, lineNumber), "类型码不对");
					continue;
				}

				if (isDuplicate(virtualProduct)) {
					canNotUpdateDtos.add(dto);
					errorMsg.put(String.format(lineTemp, lineNumber), "序列号、卡号、密码不能同时重复");
					continue;
				}

				// 卡密类型,卡号不能为空
				if (Type.CARD_PWD.equals(virtualProduct.getType())) {
					if (StringUtils.isBlank(dto.getCardno()) || StringUtils.isBlank(dto.getPasswd())) {
						canNotUpdateDtos.add(dto);
						errorMsg.put(String.format(lineTemp, lineNumber), "卡密类型,卡号和密码不能为空");
						continue;
					}
				} else if (Type.EXCHANGE_CODE.equals(virtualProduct.getType())) {
					if (StringUtils.isBlank(dto.getPasswd())) {
						canNotUpdateDtos.add(dto);
						errorMsg.put(String.format(lineTemp, lineNumber), "兑换码类型,兑换码不能为空");
						continue;
					}
				}

				virtualProductService.insert(virtualProduct);
				// 添加库存
				skuService.increaseStockOnRefundIfCanUse(sku.getId(), 1);

				canUpdateDtos.add(dto);
			} catch (Exception e) {
				canNotUpdateDtos.add(dto);
				errorMsg.put(String.format(lineTemp, lineNumber), e.getMessage());
			}
		}

		// 有导入失败记录，导出excel
		if (Collections3.isNotEmpty(canNotUpdateDtos)) {
			StringBuffer error = new StringBuffer(100);
			Iterator<String> iterator = errorMsg.keySet().iterator();
			while (iterator.hasNext()) {
				String k = iterator.next();
				error.append(k).append(":").append(errorMsg.get(k)).append(";");
			}
			return StatusDto.buildFailureStatusDto(error.toString());
		}
		return StatusDto.buildSuccessStatusDto("导入成功");
	}

	/**
	 * 导出虚拟商品卡密
	 */
	@RequestMapping(value = "/export", method = RequestMethod.GET)
	public Object batchExport(@RequestParam(required = false) String keyword, @RequestParam(required = false) VirtualProduct.Status status, @RequestParam(defaultValue = "id") String orderColumn, @RequestParam(defaultValue = "DESC") String orderSort,
		HttpServletResponse response) {
		final String vpTemplate = "template/virtualproduct_template.xls";
		try {
			final String excelFileName = java.net.URLEncoder.encode("virtualproduct", Constants.DEFAULT_ENCODING);
			response.setContentType("application/vnd.ms-excel");
			response.setHeader("content-disposition", "attachment;filename=" + excelFileName + ".xls");
			PageRequest pageRequest = new PageRequest(0, 30000, new Sort(Sort.Direction.valueOf(orderSort.toUpperCase()), orderColumn));
			Map<String, Object> params = Maps.newHashMap();
			if (StringUtils.isNotBlank(keyword)) {
				params.put("keywords", StringUtils.split(keyword, " "));
			}

			if (status != null) {
				params.put("status", status);
			}

			Page<VirtualProduct> page = virtualProductService.searchScrollPage(params, pageRequest);
			List<VirtualProductExcelDto> exportDataList = Lists.newArrayList();
			for (VirtualProduct vp : page.getContent()) {
				VirtualProductExcelDto dto = new VirtualProductExcelDto();
				dto.setCardno(vp.getCardno());
				dto.setExpired(vp.getExpired().toString());
				dto.setPasswd(vp.getPasswd());
				dto.setSerialNo(vp.getSerialNo());
				dto.setSkuCode(vp.getSkuCode());
				dto.setStatus(vp.getStatus().getLabel());
				dto.setType(vp.getType().getLabel());
				exportDataList.add(dto);
			}
			ExcelUtil.getInstance().exportObj2ExcelByTemplate((Map<String, String>) null, vpTemplate, response.getOutputStream(), exportDataList, VirtualProductExcelDto.class, Boolean.TRUE);
		} catch (Exception ex) {
			ex.printStackTrace();
		}
		return StatusDto.buildSuccessStatusDto("导出成功");
	}

	/**
	 * 下载虚拟商品卡密导入模版
	 */
	@RequestMapping(value = "/download")
	public Object downloadTemplate(HttpServletResponse response) {
		response.setContentType("application/vnd.ms-excel");
		response.setHeader("Content-Disposition", "attachment; filename=cardtemplate.xls");
		try {
			List<VirtualProductExcelDto> emptyList = Collections.emptyList();
			ExcelUtil.getInstance().exportObj2Excel(response.getOutputStream(), emptyList, VirtualProductExcelDto.class, false);
		} catch (IOException e) {
			e.printStackTrace();
		}
		return StatusDto.buildSuccessStatusDto();
	}

	@InitBinder
	protected void initBinder(HttpServletRequest request, ServletRequestDataBinder binder) throws Exception {
		binder.registerCustomEditor(Date.class, new CustomDateEditor(new SimpleDateFormat(Constants.YYYYMMDD), true));
		binder.registerCustomEditor(String.class, new StringEscapeEditor());
	}

	/**
	 * 去重判断: 同一商品code:(1)兑换码不能重复 (2)卡号或序列号不能重复
	 */
	private boolean isDuplicate(VirtualProduct virProduct) {
		long count = 0;
		if (VirtualProduct.Type.CARD_PWD.equals(virProduct.getType())) {
			count = virtualProductService.countCardPwd(virProduct);
		} else {
			count = virtualProductService.countExchangeCode(virProduct);
		}
		if (count > 0) {
			return true;
		}
		return false;
	}
}
