package com.rocoinfo.rocomall.shiro.session;

import com.rocoinfo.rocomall.redis.JedisTemplate.JedisAction;
import com.rocoinfo.rocomall.shiro.JedisManager;
import com.rocoinfo.rocomall.shiro.SerializeObjectUtil;
import org.apache.commons.lang3.StringUtils;
import org.apache.shiro.session.Session;
import org.apache.shiro.session.mgt.SimpleSession;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import redis.clients.jedis.Jedis;

import java.io.Serializable;
import java.util.Collection;

/**
 * redis save shiro session class
 * 
 * @author zhangmin
 */
public class JedisShiroSessionToStringRepository implements ShiroSessionRepository {

	public static final String REDIS_SHIRO_SESSION_PREFIX = "shiro-session-str:";

	private static final long DEFAULT_SESSION_TIMEOUT = 1800000L;

	private JedisManager jedisManager;

	private long sessionTimeOut = DEFAULT_SESSION_TIMEOUT;

	private final Logger logger = LoggerFactory.getLogger(JedisShiroSessionToStringRepository.class);

	@Override
	public void saveSession(Session session) {
		if (session == null || session.getId() == null)
			throw new NullPointerException("session is empty");
		try {
			logger.debug("save session {}", session.getId());
			String key = buildRedisSessionKey(session.getId());
			String sessionString = SerializeObjectUtil.serializeToString((SimpleSession) session);
			session.setTimeout(sessionTimeOut);
			long expireTime = sessionTimeOut / 1000;
			getJedisManager().setex(key, sessionString, (int) expireTime);
		} catch (Exception e) {
			logger.error("save session error", e);
		}
	}

	@Override
	public void deleteSession(Serializable id) {
		if (id == null) {
			throw new NullPointerException("session id is empty");
		}
		try {
			logger.debug("delete session {}", id);
			getJedisManager().del(buildRedisSessionKey(id));
		} catch (Exception e) {
			logger.error("delete session error", e);
		}
	}

	@Override
	public Session getSession(final Serializable id) {
		if (id == null)
			throw new NullPointerException("session id is empty");
		Session session = null;
		try {
			logger.debug("get redis session {}", id);
			session = getJedisManager().execute(new JedisAction<Session>() {
				@Override
				public Session action(Jedis jedis) {
					String key = buildRedisSessionKey(id);
					String sessionString = jedis.get(key);
					Session sess = null;
					if (StringUtils.isNotBlank(sessionString)) {
						sess = (Session) SerializeObjectUtil.deserializeFromString(sessionString);
						long expire = sessionTimeOut / 1000;
						// 重置Redis中缓存过期时间
						jedis.expire(key, (int) expire);
					}
					return sess;
				}
			});

		} catch (Exception e) {
			e.printStackTrace();
			logger.error("get redis session error {}", id);
		}
		return session;
	}

	@Override
	public Session getSessionWithoutExpire(Serializable sessionId) {
		if (sessionId == null)
			throw new NullPointerException("session id is empty");
		String key = buildRedisSessionKey(sessionId);
		String sessionString = getJedisManager().get(key);
		Session session = null;
		if (StringUtils.isNotBlank(sessionString)) {
			session = (Session) SerializeObjectUtil.deserializeFromString(sessionString);
		}
		return session;
	}

	@Override
	public Collection<Session> getAllSessions() {
		logger.debug("get all sessions");
		return null;
	}

	private String buildRedisSessionKey(Serializable sessionId) {
		return REDIS_SHIRO_SESSION_PREFIX + sessionId;
	}

	public JedisManager getJedisManager() {
		return jedisManager;
	}

	public void setJedisManager(JedisManager jedisManager) {
		this.jedisManager = jedisManager;
	}

	public long getSessionTimeOut() {
		return sessionTimeOut;
	}

	public void setSessionTimeOut(long sessionTimeOut) {
		this.sessionTimeOut = sessionTimeOut;
	}
}
