package com.rocoinfo.rocomall.utils;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.joor.Reflect;
import org.joor.ReflectException;

import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Map;

/**
 * <dl>
 * <dd>Description: 格式化对象以用来向前台返回json时,能够格式化成理想的状态</dd>
 * <dd>Company: 大城若谷信息技术有限公司</dd>
 * <dd>@date：2016/8/11 11:04</dd>
 * <dd>@author：Aaron</dd>
 * </dl>
 */
public class ObjectFormatUtils {

    private ObjectFormatUtils() {
    }

    /**
     * 传入一个对象和序列化的模版规则,将对象转成map格式,如果传入的是集合,那么返回List<Map>,否则返回Map对象
     * 譬如传入一个Person对象p,patterns为age->age,那么返回的map将会是{"age":"18"}
     * 支持多层嵌套pattern,譬如: user.name -> user.username 格式化成json串时,name为user对象的一个property
     * eg:
     * Object obj = ObjectFormatUtils.format2Map(user,
     * "id->id", "name->name", "gender->gender", "idCard->idCard", "birthday->birthday", "mobile->mobilePhone", "imgUrl ->imgUrl");
     * 最后返回一个map,可以为指定的key,value为指定字段的值
     *
     * @param obj      对象
     * @param patterns 模版 key->value  key为格式化之后map中的key value为需要去对象哪个字段的值
     * @return
     */
    public static Object format2Map(Object obj, String... patterns) {
        if (obj == null || patterns == null)
            return null;
        // 如果是集合
        if (obj instanceof Collection) {
            Collection objs = (Collection) obj;
            List list = Lists.newArrayListWithCapacity(objs.size());
            for (Object o : objs) {
                list.add(formatSingleObj2Map(o, patterns));
            }
            return list;
        } else {
            return formatSingleObj2Map(obj, patterns);
        }
    }

    /**
     * 处理单个对象的object->map
     *
     * @param o        对象
     * @param patterns 模版 key->value  key为格式化之后map中的key value为需要去对象哪个字段的值
     * @return
     */
    private static Object formatSingleObj2Map(final Object o, final String... patterns) {
        if (o == null || patterns == null)
            return null;
        // 创建返回的结果对象
        Map<String, Object> res = Maps.newHashMapWithExpectedSize(patterns.length);
        for (String pattern : patterns) { // 循环处理每隔pattern
            pattern = pattern.trim(); // remove blank
            if (!pattern.contains("->")) //　判断pattern格式是否正确
                continue;

            String[] strs = pattern.split("->");
            if (strs == null || strs.length != 2)
                continue;

            injectAttribute(o, res, strs[0], strs[1]);
        }
        return res;
    }

    /**
     * 将对象中的指定字段的value放入到map集合中
     * filed可能会是多级属性 譬如 address.province,需要多层反射来获取province的值
     *
     * @param o     对象
     * @param map   map
     * @param key   放入map集合中时的key
     * @param field 指定的字段
     */
    private static void injectAttribute(final Object o, final Map<String, Object> map, final String key, final String field) {
        if (o == null || map == null || StringUtils.isAnyBlank(key, field))
            return;
        // 如果key是多层 譬如 user.name 那么需要map嵌套 以实现json转换时 name作为user的一个属性出现
        Map<String, Object> innerMap = map; // 声明一个map 指向最原始的map 为了一层一层put数据用,主要目的是不改变原来map的指向
        String innerKey = key; //用于获取key是多层级时最后一层的key
        if (key.contains(".")) {
            String[] ks = key.split("\\.");
            for (int i = 0, length = ks.length; i < length; i++) {
                String k = ks[i];
                if (i < (length - 1)) {
                    Map<String, Object> m = Maps.newHashMap();
                    innerMap.put(k, m);
                    innerMap = m;
                    innerKey = k;
                }
            }
        }
        innerMap.put(innerKey, getObjectAttribute(o, field));
    }

    /**
     * 获取对象指定字段的value,注意filed的多层嵌套结构
     *
     * @param o     对象
     * @param field 字段,可能是多层结构
     * @return
     */
    private static Object getObjectAttribute(final Object o, final String field) {
        if (o == null || StringUtils.isBlank(field))
            return null;
        Object inner = o; // 为了获取多层嵌套的属性值
        if (field.contains(".")) {
            String[] fs = field.split("\\.");
            // 一层一层的反射获取数据
            for (int i = 0, length = fs.length; i < length; i++) {
                if (inner != null) {
                    String f = fs[i];
                    try {
                        // warning:此处调用的是属性的getter方法,所以一定要确保属性具有getter方法 并且getter方法符合规范
                        inner = Reflect.on(inner).call(propertyGetter(f)).get(); //  如果反射异常直接返回null
                    } catch (ReflectException e) {
                        inner = null;
                    }
                } else {
                    break; // 如果inner为null循环直接break
                }
            }
        } else {
            try {
                // warning:此处调用的是属性的getter方法,所以一定要确保属性具有getter方法 并且getter方法符合规范
                inner = Reflect.on(inner).call(propertyGetter(field)).get();
            } catch (ReflectException e) {
                inner = null;
            }
        }
        return formatFeildValue(inner);
    }

    /**
     * 格式化字段的值
     *
     * @param obj 要格式化的值
     * @return
     */
    private static Object formatFeildValue(Object obj) {
        if (obj != null) {
            if (obj instanceof Date) {
                obj = DateFormatUtils.format((Date) obj, "yyyy-MM-dd HH:mm:ss");
            }
        }
        return obj;
    }

    /**
     * 拼接属性的getter方法 getXxx 要符合驼峰式写法的规范
     *
     * @param field 属性名
     * @return
     */
    private static String propertyGetter(String field) {
        if (StringUtils.isBlank(field))
            return "";
        return new StringBuilder("get").append(StringUtils.capitalize(field)).toString();
    }
}
