package com.rocoinfo.rocomall.utils;

import com.rocoinfo.rocomall.Constants;
import com.rocoinfo.rocomall.PropertyHolder;
import com.rocoinfo.rocomall.entity.IdEntity;
import com.rocoinfo.rocomall.entity.Supplier;
import com.rocoinfo.rocomall.shiro.ShiroUser;
import com.rocoinfo.rocomall.utils.UserAgent.ClientType;
import com.rocoinfo.rocomall.utils.UserAgent.OsType;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.apache.shiro.SecurityUtils;
import org.apache.shiro.UnavailableSecurityManagerException;
import org.apache.shiro.session.Session;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.util.CollectionUtils;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;
import org.springside.modules.utils.Collections3;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * @author zhangmin
 */
@SuppressWarnings("all")
public class WebUtils {

	private WebUtils() {
	}

	/**
	 * 返回站点访问Base路径
	 * 
	 * @return http://localhost:8080/ctxPath
	 */
	public static String getBaseSiteUrl(HttpServletRequest request) {
		final StringBuilder basePath = new StringBuilder();
		basePath.append(request.getScheme()).append("://").append(request.getServerName());
		if (request.getServerPort() != 80) {
			basePath.append(":").append(request.getServerPort());
		}
		basePath.append(request.getContextPath());
		return basePath.toString();
	}

	public static String getFullImageUrl(String imagePath) {
		return PropertyHolder.getImageBaseUrl() + imagePath;
	}

	public static List<Supplier> getManagedSuppliersOfLoginUser() {
		List<Supplier> suppliers = (List<Supplier>) getSession().getAttribute(Constants.SUPPLIERS);
		if (suppliers == null) {
			return Collections.emptyList();
		}
		return suppliers;
	}

	public static ShiroUser getLoggedUser() {
		try {
			return (ShiroUser) SecurityUtils.getSubject().getPrincipal();
		} catch (UnavailableSecurityManagerException ex) {
			return null;
		}
	}

	public static List<Long> getWhereClauseSupplierIds(Long supplierIdQueryParam) {
		ShiroUser shiroUser = getLoggedUser();
		List<Long> managedSupplierIds = Collections3.extractToList(WebUtils.getManagedSuppliersOfLoginUser(), IdEntity.ID_FIELD_NAME);
		if (CollectionUtils.isEmpty(managedSupplierIds)) {// 没有权限 直接返回
			return managedSupplierIds;
		}

		List<Long> paramSupplierIds = new ArrayList<Long>();
		if (supplierIdQueryParam != null && supplierIdQueryParam > 0 && managedSupplierIds.contains(supplierIdQueryParam)) {
			paramSupplierIds.add(supplierIdQueryParam);
		} else {
			paramSupplierIds.addAll(managedSupplierIds);
		}
		return paramSupplierIds;
	}

	/**
	 * 返回登录用户ID，没有登录返回 null
	 */
	public static Long getLoggedUserId() {
		ShiroUser user = getLoggedUser();
		if (user == null) {
			return null;
		} else {
			return user.getId();
		}
	}

	public static boolean isAjaxRequest(HttpServletRequest req) {
		String constant = "x-requested-with";
		String xRequestedWith = req.getHeader(constant);
		return StringUtils.isNotBlank(xRequestedWith) && StringUtils.equalsIgnoreCase(xRequestedWith, "XMLHttpRequest");
	}

	/**
	 * @return true 已登录, false 未登录
	 */
	public static boolean isLogged() {
		return getLoggedUserId() != null;
	}

	/**
	 * 返回存储在Redis中的Session对象。
	 * 
	 * @return org.apache.shiro.session.Session
	 */
	public static Session getSession() {
		return SecurityUtils.getSubject().getSession();
	}

	/**
	 * 短信验证码是否正确，或已经过期
	 * 
	 * @param vcode
	 *            用户输入的验证码
	 */
	public static boolean isSmsVCodeValid(final String vcode) {
		Session session = getSession();
		String sessionVcode = (String) session.getAttribute(Constants.REGISTER_VCODE);
		if (StringUtils.isEmpty(sessionVcode)) {
			return false;
		}

		Long sendVcodeTimestamp = (Long) session.getAttribute(Constants.REGISTER_VCODE_SMS_TIMESTAMP);
		int count = Integer.parseInt(session.getAttribute(Constants.REGISTER_WRONGCOUNT).toString());
		// 短信验证码最多只能错误5次
		if (count > 5) {
			session.removeAttribute(Constants.REGISTER_VCODE);
			return false;
		}

		// 验证码有效期是5分钟
		if ((System.currentTimeMillis() - sendVcodeTimestamp) > DateUtils.MILLIS_PER_MINUTE * 5) {
			return false;
		}
		if (!sessionVcode.equalsIgnoreCase(vcode)) {
			session.setAttribute(Constants.REGISTER_WRONGCOUNT, count + 1);
		}

		return sessionVcode.equalsIgnoreCase(vcode);
	}

	/** 获得客户端ip地址 */
	public static String getClientIpAddr(HttpServletRequest request) {
		String ip = request.getHeader("x-forwarded-for");

		if (StringUtils.isEmpty(ip) || "unknown".equalsIgnoreCase(ip)) {
			ip = request.getHeader("X-Real-IP");
		}

		if (StringUtils.isEmpty(ip) || "unknown".equalsIgnoreCase(ip)) {
			ip = request.getRemoteAddr();
		}

		List<String> ips = findMatchsContentsByRegex(ip, "[\\d\\.]{7,15}");

		String ipAddr = ips.size() > 0 ? ips.get(0) : "0.0.0.0";
		if (StringUtils.isNotBlank(ipAddr) && ipAddr.length() > 20) {
			ipAddr = ipAddr.substring(0, 20);
		}
		return ipAddr;
	}

	/**
	 * 先从request参数中获取openid，获取到以后，放入session,中如果没有获取到，则到session中获取,
	 * session没有则读取cookie
	 * 
	 * @param request
	 * @return
	 */
	public static String getOpenIdAndStoreCookieSession(final HttpServletRequest request, final HttpServletResponse resp) {
		String openId = request.getParameter("openId");
		final Session session = getSession();
		final String openIdInSession = "sessionOpenId";

		if (StringUtils.isBlank(openId)) {
			openId = (String) session.getAttribute(openIdInSession);
			if (StringUtils.isEmpty(openId)) {
				// 如果session获取到的openId 为空，则继续从Cookie中获取
				openId = CookieUtil.getCookieOpenId(request);
				if (StringUtils.isNotBlank(openId)) {
					session.setAttribute(openIdInSession, openId);
				}
			}
		} else {
			session.setAttribute(openIdInSession, openId);

			// 将请求参数中的openId 加入cookie
			CookieUtil.addOpenIdToCookie(request, resp, openId);
		}
		return StringUtils.trimToEmpty(openId);
	}

	public static UserAgent parseUserAgent(HttpServletRequest request) {
		final String ua = StringUtils.trimToEmpty(request.getHeader("User-Agent"));
		UserAgent userAgent = new UserAgent();

		if (ua.contains("Windows")) { // 平台是 windows
			/**
			 * ****************** 台式机 Windows 系列 ****************** Windows NT
			 * 6.2 - Windows 8 Windows NT 6.1 - Windows 7 Windows NT 6.0 -
			 * Windows Vista Windows NT 5.2 - Windows Server 2003; Windows XP
			 * x64 Edition Windows NT 5.1 - Windows XP Windows NT 5.01 - Windows
			 * 2000, Service Pack 1 (SP1) Windows NT 5.0 - Windows 2000 Windows
			 * NT 4.0 - Microsoft Windows NT 4.0 Windows 98; Win 9x 4.90 -
			 * Windows Millennium Edition (Windows Me) Windows 98 - Windows 98
			 * Windows 95 - Windows 95 Windows CE - Windows CE Mozilla/5.0
			 * (compatible; MSIE 9.0; Windows Phone OS 7.5; Trident/5.0;
			 * IEMobile/9.0
			 */

			userAgent.setOsType(OsType.WINDOWS);

			if (ua.contains("Windows Phone")) {
				userAgent.setClientType(ClientType.WINDOWS_PHONE);
			} else {
				userAgent.setClientType(ClientType.PC);
			}
		} else if (ua.contains("Mac OS X")) {
			/**
			 * 苹果系列 iPod - Mozilla/5.0 (iPod; U; CPU iPhone OS 4_3_1 like Mac OS
			 * X; zh-cn) AppleWebKit/533.17.9 (KHTML, like Gecko) Version/5.0.2
			 * Mobile/8G4 Safari/6533.18.5 iPad - Mozilla/5.0 (iPad; U; CPU OS
			 * 3_2 like Mac OS X; en-us) AppleWebKit/531.21.10 (KHTML, like
			 * Gecko) Version/4.0.4 Mobile/7B334b Safari/531.21.10 iPad2 -
			 * Mozilla/5.0 (iPad; CPU OS 5_1 like Mac OS X; en-us)
			 * AppleWebKit/534.46 (KHTML, like Gecko) Version/5.1 Mobile/9B176
			 * Safari/7534.48.3 iPhone 4 - Mozilla/5.0 (iPhone; U; CPU iPhone OS
			 * 4_0 like Mac OS X; en-us) AppleWebKit/532.9 (KHTML, like Gecko)
			 * Version/4.0.5 Mobile/8A293 Safari/6531.22.7 iPhone 5 -
			 * Mozilla/5.0 (iPhone; CPU iPhone OS 5_0 like Mac OS X)
			 * AppleWebKit/534.46 (KHTML, like Gecko) Version/5.1 Mobile/9A334
			 * Safari/7534.48.3
			 */
			userAgent.setOsType(OsType.IOS);

			if (ua.contains("iPhone")) {
				userAgent.setClientType(ClientType.IPHONE);
			} else if (ua.contains("iPod")) {
				// iPod touch
				userAgent.setClientType(ClientType.ITOUCH);
			} else if (ua.contains("iPad")) {
				userAgent.setClientType(ClientType.IPAD);
			} else {
				// 苹果台式机或苹果笔记本
				userAgent.setClientType(ClientType.PC);
			}
		} else if (ua.contains("Linux")) {
			userAgent.setOsType(OsType.LINUX);

			if (ua.contains("Android")) {
				// andriod手机
				userAgent.setClientType(ClientType.ANDROID);
			} else {
				userAgent.setClientType(ClientType.PC);
			}
		}
		return userAgent;
	}

	/** 从sourceStr中 筛选出 匹配正在表达式regex的 内容 **/
	public static List<String> findMatchsContentsByRegex(String sourceStr, String regex) {
		List<String> strList = new ArrayList<String>(0);
		try {
			Pattern pattern = Pattern.compile(regex);
			Matcher matcher = pattern.matcher(sourceStr);
			while (matcher.find()) {
				strList.add(matcher.group());
			}
		} catch (Exception e) {
			e.printStackTrace();
		}
		return strList;
	}

	public static HttpServletRequest getRequest() {
		return ((ServletRequestAttributes) RequestContextHolder.getRequestAttributes()).getRequest();
	}

	/**
	 * 发生业务错误，或请求参数不符合条件，响应400状态码.
	 * 
	 * @param message
	 *            错误信息, 直接作为HTTP body输出
	 */
	@SuppressWarnings("unchecked")
	public static ResponseEntity response400(String message) {
		return new ResponseEntity(message, null, HttpStatus.BAD_REQUEST);
	}

	/**
	 * 发生服务器内部错误，响应500状态码.
	 * 
	 * @param message
	 *            错误信息, 直接作为HTTP body输出
	 */
	@SuppressWarnings("unchecked")
	public static ResponseEntity response500(String message) {
		return new ResponseEntity(message, null, HttpStatus.INTERNAL_SERVER_ERROR);
	}

	public static ResponseEntity RESPONSE_200 = new ResponseEntity(HttpStatus.OK);

	/**
	 * 分页查询map赋值
	 * 
	 */
	public static void setStringValue(Map<String, Object> map, String key, String value) {
		if (StringUtils.isNotBlank(value)) {
			map.put(key, value);
		}
	}

	public static void setObjectValue(Map<String, Object> map, String key, Object value) {
		if (value != null) {
			map.put(key, value);
		}
	}
}