package com.rocoinfo.rocomall.utils;

import org.apache.commons.io.IOUtils;

import java.io.*;
import java.util.Enumeration;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipOutputStream;




/**
 * 压缩/解压zip
 * @author zhangmin
 */
public final class ZipUtils {

	private static final int DEFAULT_BUFFER_SIZE = 1024 * 4;
	
	/**
	 * @param srcfile 源目录或文件
	 * @param destfile 目标文件路径，zip文件
	 */
	public void zip(String srcfile, String destfile) {
		ZipOutputStream out = null;
		try {
			out = new ZipOutputStream(new BufferedOutputStream(new FileOutputStream(destfile)));
			addZip(srcfile, out, "");
		} catch (IOException e) {
			throw new RuntimeException(e);
		} finally {
			IOUtils.closeQuietly(out);
		}
	}

	private void addZip(String srcFilePath, ZipOutputStream zipOut, String parentPath) {
		try {
			File srcFile = new File(srcFilePath);
			/** 忽略一些常见的不需要压缩的目录或文件
			  if (context.isPackageExclude(srcFile)) {
				return;
			 }*/

			// 压缩文件
			if (srcFile.isFile()) {
				byte data[] = new byte[DEFAULT_BUFFER_SIZE];
				BufferedInputStream buffIn = null;
				try {
					FileInputStream in = new FileInputStream(srcFile);
					buffIn = new BufferedInputStream(in, DEFAULT_BUFFER_SIZE);
					ZipEntry entry = new ZipEntry(parentPath + srcFile.getName());
					entry.setTime(srcFile.lastModified());
					zipOut.putNextEntry(entry);
					int count;
					while ((count = buffIn.read(data, 0, DEFAULT_BUFFER_SIZE)) != -1) {
						zipOut.write(data, 0, count);
					}
				} finally {
					IOUtils.closeQuietly(buffIn);
				}
			}

			// 压缩目录
			if (srcFile.isDirectory()) {
				File files[] = srcFile.listFiles();
				for (int i = 0; i < files.length; i++) {
					addZip(files[i].getAbsolutePath(), zipOut, parentPath + srcFile.getName() + "/");
				}
			}

		} catch (IOException e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * 解压缩 zip 文件
	 */
	public static void unzip(String zipfile, String unzipToDir) {
		ZipFile zf = null;
		try {
			zf = new ZipFile(zipfile);
			Enumeration<? extends ZipEntry> emu = zf.entries();
			while (emu.hasMoreElements()) {
				ZipEntry entry = emu.nextElement();
				//会把目录作为一个file读出一次，所以只建立目录就可以，之下的文件还会被迭代到。
				if (entry.isDirectory()) {
					new File(unzipToDir + "/" + entry.getName()).mkdirs();
					continue;
				}
				BufferedInputStream bis = new BufferedInputStream(zf.getInputStream(entry));
				File file = new File(unzipToDir + "/" + entry.getName());
				//加入这个的原因是zipfile读取文件是随机读取的，这就造成可能先读取一个文件
				//而这个文件所在的目录还没有出现过，所以要建出目录来。
				File parent = file.getParentFile();
				if (parent != null && (!parent.exists())) {
					parent.mkdirs();
				}
				FileOutputStream fos = new FileOutputStream(file);
				BufferedOutputStream bos = new BufferedOutputStream(fos, DEFAULT_BUFFER_SIZE);

				int count;
				byte data[] = new byte[DEFAULT_BUFFER_SIZE];
				while ((count = bis.read(data, 0, DEFAULT_BUFFER_SIZE)) != -1) {
					bos.write(data, 0, count);
				}
				bos.flush();
				bos.close();
				bis.close();
			}

		} catch (IOException e) {
			throw new RuntimeException(e);
		} finally {
			try {
				if (zf != null) {
					zf.close();
				}
			} catch (IOException e) {
			}
		}
	}
}