package com.rocoinfo.rocomall.service.impl.cent;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.rocoinfo.rocomall.Constants;
import com.rocoinfo.rocomall.common.service.CrudService;
import com.rocoinfo.rocomall.dto.DeductParam;
import com.rocoinfo.rocomall.dto.RefundParam;
import com.rocoinfo.rocomall.dto.StatusDto;
import com.rocoinfo.rocomall.entity.account.User;
import com.rocoinfo.rocomall.entity.cent.CentConsumeDetail;
import com.rocoinfo.rocomall.entity.cent.CentConsumeRecord;
import com.rocoinfo.rocomall.entity.cent.Cents;
import com.rocoinfo.rocomall.entity.dict.DictCentConsumeType;
import com.rocoinfo.rocomall.entity.dict.DictPlatform;
import com.rocoinfo.rocomall.redis.RedissonTemplate;
import com.rocoinfo.rocomall.repository.cent.CentApplyDetailDao;
import com.rocoinfo.rocomall.repository.cent.CentConsumeDetailDao;
import com.rocoinfo.rocomall.repository.cent.CentConsumeRecordDao;
import com.rocoinfo.rocomall.repository.cent.CentsDao;
import com.rocoinfo.rocomall.service.cent.ICentsService;
import com.rocoinfo.rocomall.utils.CodeGenerator;
import org.apache.commons.collections.CollectionUtils;
import org.redisson.core.RLock;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Isolation;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;


/**
 * @author zhangmin
 */
@SuppressWarnings("all")
@Service

public class CentsService extends CrudService<CentsDao, Cents> implements ICentsService {
    private Logger log = LoggerFactory.getLogger(CentsService.class);


    @Autowired
    private CentConsumeRecordService consumeRecordService;

    @Autowired
    private CentConsumeRecordDao centConsumeRecordDao;

    @Autowired
    private CentConsumeDetailDao consumeDetailDao;

    @Autowired
    private RedissonTemplate redissonTemplate;

    @Autowired
    private CentApplyDetailDao centApplyDetailDao;

    @Autowired
    private CodeGenerator codeGenerator;


    public Cents getCentByCode(String code) {
        return entityDao.getByCode(code);
    }

    public List<String> findExistCentCodes(List<String> codeList) {
        return entityDao.findExistCentCodes(codeList);
    }

    public void scheduleExpireCent(Date limitTime) {
        entityDao.scheduleExpireCent(limitTime);
    }


    @Override
    public void refundCent(RefundParam refundParam) {
        Date dateline = new Date();

        int refundCent = refundParam.getCent();
        User user = new User(refundParam.getUserId());
        CentConsumeRecord refundConsumeRecord = new CentConsumeRecord();
        refundConsumeRecord.setConsumeType(new DictCentConsumeType(refundParam.getConsumeTypeCode()));
        refundConsumeRecord.setUser(user);
        refundConsumeRecord.setEntityId(refundParam.getEntityId());

        CentConsumeRecord consumedRecord = this.centConsumeRecordDao.getSingle(refundConsumeRecord);

        //可以返分的积分消耗记录明细,积分过期期间时间越大，则先返回
        List<CentConsumeDetail> consumeDetails = this.consumeDetailDao.findCanRefundConsumeDetailByRcrdIdOrderByExpireDateDesc(consumedRecord.getId());

        Map<Long, Integer> refundedCentValueMap = Maps.newHashMap();
        Map<Long, Integer> centIncrMap = Maps.newHashMap();

        //返分明细
        List<CentConsumeDetail> refundDetailList = Lists.newArrayList();

        if (CollectionUtils.isNotEmpty(consumeDetails) && refundCent > 0) {
            for (int i = 0, j = consumeDetails.size(); i < j && refundCent > 0; i++) {
                CentConsumeDetail detail = consumeDetails.get(i);
                int itemRefundCent = Math.min(refundCent, detail.getAvaiableRefundCent());

                refundedCentValueMap.put(detail.getId(), itemRefundCent);
                centIncrMap.put(detail.getCent().getId(), itemRefundCent);

                refundCent -= itemRefundCent;
                CentConsumeDetail refundDetail = new CentConsumeDetail();
                refundDetail.setCent(detail.getCent());
                refundDetail.setCentAmt(-itemRefundCent);
                refundDetail.setRefundAmt(0);
                refundDetail.setRecord(refundConsumeRecord);
                refundDetailList.add(refundDetail);
            }


            if (CollectionUtils.isNotEmpty(refundDetailList)) {
                refundConsumeRecord.setSnNum(this.codeGenerator.generateCentConsumeSnNum(dateline, refundParam.getConsumeTypeCode()));
                refundConsumeRecord.setCentDetails(refundDetailList);
                refundConsumeRecord.setDateline(dateline);
                refundConsumeRecord.setPlatform(DictPlatform.CENT_PLATFORM);
                refundConsumeRecord.setStatus(CentConsumeRecord.Status.REFUND);
                refundConsumeRecord.setConsumeCent(-refundParam.getCent());
                refundConsumeRecord.setConsumeType(DictCentConsumeType.REFUND_CONSUME_TYPE);
                this.consumeRecordService.save(refundConsumeRecord);

                //返回账户积分
                Iterator<Map.Entry<Long, Integer>> iter = centIncrMap.entrySet().iterator();
                while (iter.hasNext()) {
                    Map.Entry<Long, Integer> entry = iter.next();
                    this.entityDao.incrBalanceAndUpdateToCanUse(entry.getKey(), entry.getValue());
                }

                //增加积分已返分额
                Iterator<Map.Entry<Long, Integer>> refundAmtIter = refundedCentValueMap.entrySet().iterator();
                while (refundAmtIter.hasNext()) {
                    Map.Entry<Long, Integer> entry = refundAmtIter.next();
                    this.consumeDetailDao.incrRefundAmt(entry.getKey(), entry.getValue());
                }
            }
        }
    }

    /**
     * 查询历史积分管理列表
     */
    @Transactional(readOnly = true)
    public Page<Cents> searchHistCents(Map<String, Object> parameters,
                                       Pageable pageable) {
        long total = this.entityDao.searchTotal(parameters);
        List<Cents> content = Collections.emptyList();
        if (total > pageable.getOffset()) {
            parameters.put(Constants.PAGE_OFFSET, pageable.getOffset());
            parameters.put(Constants.PAGE_SIZE, pageable.getPageSize());
            parameters.put(Constants.PAGE_SORT, pageable.getSort());
            content = this.entityDao.search(parameters);
        }
        return new PageImpl<Cents>(content, pageable, total);
    }

    /**
     * 扣分
     */
    @Transactional(isolation = Isolation.READ_COMMITTED, propagation = Propagation.REQUIRED)
    public StatusDto<String> deductCent(final DeductParam param, Long loggedUserId) {
        RLock rLock = redissonTemplate.getDistributeLock(loggedUserId + "");
        String msg = "";
        try {
            rLock.lock();

            if (param == null)
                return StatusDto.buildFailureStatusDto("扣积分参数不能为空!");
            List<Cents> centList = this.entityDao.findCanUseCentOrderByExpireDateOfUser(param.getUserId());
            if (CollectionUtils.isEmpty(centList)) {
                return StatusDto.buildFailureStatusDto("您账户没有任何可用积分!");
            }
            log.info("积分扣减参数：{}", param.toString());

            int deductCent = param.getCent();

            int availableTotalCent = 0;

            Map<Cents, Integer> centUsedMap = Maps.newHashMap();

            for (Cents centCard : centList) {
                // 当前积分卡，应该扣减的积分数
                int curCardDeductCent = Math.min(centCard.getBalance(), deductCent);
                availableTotalCent += curCardDeductCent;
                deductCent -= curCardDeductCent;
                centUsedMap.put(centCard, curCardDeductCent);
                if (deductCent == 0) {
                    break;
                }
            }

            if (availableTotalCent != param.getCent()) {
                String error = "剩余积分不足，剩余："
                        + availableTotalCent;
                log.debug("用户[" + param.getUserId() + "]" + error);
                return StatusDto.buildFailureStatusDto(error);
            }

            // 执行扣分逻辑
            List<CentConsumeDetail> consumeDetailList = Lists
                    .newArrayListWithExpectedSize(centUsedMap.size());

            // 保存积分消耗记录
            Date dateline = new Date();
            CentConsumeRecord consumeRecord = new CentConsumeRecord();
            consumeRecord.setDateline(dateline);
            consumeRecord.setSnNum(codeGenerator.generateCentConsumeSnNum(dateline,
                    param.getConsumeTypeCode()));
            consumeRecord.setConsumeType(new DictCentConsumeType(param
                    .getConsumeTypeCode()));
            consumeRecord.setPlatform(new DictPlatform(param.getPlatformCode()));
            consumeRecord.setConsumeCent(param.getCent());
            consumeRecord.setStatus(CentConsumeRecord.Status.CONSUME);
            consumeRecord.setUser(new User(param.getUserId()));
            consumeRecord.setEntityId(param.getEntityId());

            Iterator<Map.Entry<Cents, Integer>> iter = centUsedMap.entrySet().iterator();
            while (iter.hasNext()) {
                Map.Entry<Cents, Integer> entry = iter.next();
                Cents cent = entry.getKey();

                Cents updateCent = new Cents();
                updateCent.setId(cent.getId());
                updateCent.setBalance(cent.getBalance() - entry.getValue());

                if (updateCent.getBalance() == 0) {
                    updateCent.setStatus(Cents.Status.USED);
                }

                CentConsumeDetail detail = new CentConsumeDetail();
                detail.setCent(updateCent);
                detail.setCentAmt(entry.getValue());
                detail.setRecord(consumeRecord);
                consumeDetailList.add(detail);

                this.entityDao.update(updateCent);
            }

            consumeRecord.setCentDetails(consumeDetailList);

            this.consumeRecordService.save(consumeRecord);

            msg = "成功扣减积分：" + param.getCent();
            log.info("用户[" + param.getUserId() + "]" + msg);
        } finally {
            rLock.unlock();
        }
        return StatusDto.buildSuccessStatusDto(msg);
    }

}
