package com.rocoinfo.rocomall.service.impl.dict.express;

import com.rocoinfo.rocomall.common.service.CrudService;
import com.rocoinfo.rocomall.common.service.ServiceException;
import com.rocoinfo.rocomall.entity.dict.express.ExpressFeeItem;
import com.rocoinfo.rocomall.entity.dict.express.ExpressFeeTemplate;
import com.rocoinfo.rocomall.entity.dict.express.QuantityAmt;
import com.rocoinfo.rocomall.repository.dict.express.ExpressFeeTemplateDao;
import com.rocoinfo.rocomall.service.dict.express.IExpressFeeItemService;
import com.rocoinfo.rocomall.service.dict.express.IExpressFeeTemplateService;
import com.rocoinfo.rocomall.utils.ArithUtils;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Isolation;
import org.springframework.transaction.annotation.Transactional;

import java.util.Date;

/**
 * <dl>
 * <dd>描述:快递费用计算模板Service</dd>
 * <dd>公司: 大城若谷信息技术有限公司</dd>
 * <dd>创建时间：2015年11月26日 下午12:27:28</dd>
 * <dd>创建人： 张敏</dd>
 * </dl>
 */
@Service
public class ExpressFeeTemplateService extends CrudService<ExpressFeeTemplateDao, ExpressFeeTemplate> implements IExpressFeeTemplateService {

	@Autowired
	private IExpressFeeItemService feeItemService;

	@Transactional(isolation = Isolation.READ_COMMITTED)
	@Override
	public void insert(ExpressFeeTemplate feeTemplate) {
		if (feeTemplate != null) {
			super.insert(feeTemplate);
			saveTempateItems(feeTemplate);
		}
	}

	@Transactional(isolation = Isolation.READ_COMMITTED)
	@Override
	public void update(ExpressFeeTemplate feeTemplate) {
		if (feeTemplate != null) {
			super.update(feeTemplate);
			//删除掉原来的Item和城市
			this.feeItemService.deleteFeeItemsByTemplateId(feeTemplate.getId());
			saveTempateItems(feeTemplate);
		}
	}

	private void saveTempateItems(ExpressFeeTemplate feeTemplate) {
		if (CollectionUtils.isNotEmpty(feeTemplate.getItems())) {
			for (ExpressFeeItem item : feeTemplate.getItems()) {
				item.setFeeTemplate(feeTemplate);
				feeItemService.insert(item);
			}
		}
	}

	/**
	 * 根据模板id,查询此模板详细信息，包括该模板的子项，及模板子项绑定的城市
	 * 
	 * @param id模板id
	 */
	@Override
	public ExpressFeeTemplate getTemplateDetailById(Long id) {
		if (id == null)
			throw new ServiceException("模板ID为空！");
		return entityDao.getTemplateDetailById(id);
	}

	/**
	 * @param orderWeight 购买的订单的 （件数|重量|体积）
	 */
	public double calculateDeliverFee(final Long cityId, final Long expressId, int orderWeight) {
		QuantityAmt qAmt = null;
		ExpressFeeItem feeItem = this.feeItemService.getFeeItemByCityIdAndExpressId(cityId, expressId);
		if (feeItem != null) {
			qAmt = feeItem.getQamt();
		} else {
			ExpressFeeTemplate feeTemplate = this.entityDao.getByExpressId(expressId);
			if (feeTemplate != null) {
				qAmt = feeTemplate.getDef();
			}
		}

		if (qAmt == null)
			return 0;

		if (orderWeight <= qAmt.getInitQuantity()) {
			return qAmt.getInitAmt();
		}

		int orderOverWeight = orderWeight - qAmt.getInitQuantity();

		//需要支付的超重费用
		double overFee = (orderOverWeight * qAmt.getIncrAmt()) / qAmt.getIncrUnit();
		double totalFee = ArithUtils.round(overFee + qAmt.getInitAmt(), 2);
		return totalFee;
	}

	/**
	 * 通过快递公司Id,统计该快递公司的模板数量
	 * 
	 * @param expressTemplate
	 */
	public long countTemplateByExpress(ExpressFeeTemplate expressTemplate) {
		long count = 0;
		if (expressTemplate != null && expressTemplate.getExpress() != null && expressTemplate.getExpress().getId() > 0) {
			count = this.entityDao.countTemplateByExpress(expressTemplate);
		}
		return count;
	}

	/**
	 * 根据原始模板 id 复制新模板
	 * 
	 * @param id 2015年11月26日 下午3:37:23
	 */
	@Override
	public ExpressFeeTemplate cloneTemplate(Long id) {
		if (id == null)
			throw new ServiceException("原始模板ID为空！");
		//查询原始模板
		ExpressFeeTemplate template = this.getTemplateDetailById(id);
		if (template == null)
			throw new ServiceException("没有此模板！");
		//修改模板名称
		template.setName(creatColneName(template.getName()));
		template.setId(null);
		//克隆时设置快递公司为null,因为1个快递公司只能有1个模板
		template.setExpress(null);
		//插入新模板
		this.insert(template);
		return template;
	}
	
	private String creatColneName(String oldName){
		int index = oldName.lastIndexOf("复制(");
		String newName = "";
		if(index != -1){
			newName = oldName.substring(0,index)+"复制("+new Date().getTime() + ")";
		}else{
			newName = oldName+"复制(" + new Date().getTime() + ")";
		}
		return newName;
	}

}
