package com.rocoinfo.rocomall.service.impl.activity;

import com.google.common.collect.Lists;
import com.rocoinfo.rocomall.Constants;
import com.rocoinfo.rocomall.common.service.CrudService;
import com.rocoinfo.rocomall.common.service.ServiceException;
import com.rocoinfo.rocomall.entity.IdEntity;
import com.rocoinfo.rocomall.entity.Product;
import com.rocoinfo.rocomall.entity.Sku;
import com.rocoinfo.rocomall.entity.activity.Activity;
import com.rocoinfo.rocomall.entity.activity.Activity.Status;
import com.rocoinfo.rocomall.entity.activity.Activity.Type;
import com.rocoinfo.rocomall.entity.activity.ActivitySku;
import com.rocoinfo.rocomall.redis.CacheKeys;
import com.rocoinfo.rocomall.repository.activity.ActivityDao;
import com.rocoinfo.rocomall.repository.activity.ActivityRecordDao;
import com.rocoinfo.rocomall.service.activity.IActivityService;
import com.rocoinfo.rocomall.service.activity.IActivitySkuService;
import com.rocoinfo.rocomall.service.product.*;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.aop.framework.AopContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Isolation;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springside.modules.utils.Collections3;

import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;

@SuppressWarnings("all")
@Service
public class ActivityService extends CrudService<ActivityDao, Activity> implements IActivityService {
    private static Logger logger = LoggerFactory.getLogger(ActivityService.class);
    @Autowired
    private IActivitySkuService activitySkuService;
    @Autowired
    private ISkuService skuService;
    @Autowired
    private IProductImageService productImageService;
    @Autowired
    private IProductService productService;
    @Autowired
    private ISupplierService supplierService;
    @Autowired
    private IBrandService brandService;
    @Autowired
    private IDescriptionService descriptionService;
    @Autowired
    private IProductCatalogService productCatalogService;
    @Autowired
    private ActivityRecordDao activityRecordDao;

    @Transactional(isolation = Isolation.READ_COMMITTED, propagation = Propagation.REQUIRED)
    public void insert(Activity activity) {

        if (activity == null) {
            throw new ServiceException("活动不能为空");
        }

        if (entityDao.findByActivityName(activity.getName(), null) != null) {
            throw new ServiceException("活动名称已经存在");
        }

        if (!activity.getEndTime().after(activity.getStartTime())) {
            throw new ServiceException("结束时间应该大于开始时间");
        }

        if (!activity.getEndTime().after(new Date())) {
            throw new ServiceException("结束时间应该大于当前时间");
        }

        activity.setStatus(Activity.Status.DRAFT);
        entityDao.insert(activity);
        if (activity.getType() == Activity.Type.SCRAPE_CARD || activity.getType() == Activity.Type.TURN_TABLE) {
            activityRecordDao.initActivityTotalCent(activity.getId());
        }
        for (ActivitySku activitySku : activity.getActivitySkus()) {
            if (activitySku == null || activitySku.getSku() == null) {
                continue;
            }
            activitySku.setActivity(activity);
            activitySkuService.insert(activitySku);
        }
    }

    /**
     * 前台活动是否可以参与 如果 活动上架 && 活动下的商品还有库存，则可以参与
     */
    public boolean isActivityJoinable(Long id) {
        Activity activity = getProxy().getById(id);
        if (activity == null)
            return false;

        if (activity.getStatus() != Activity.Status.LIST)
            return false;

        List<ActivitySku> activitySkuList = activitySkuService.findByActivityId(id);
        return isStockEnoughOfActivity(activitySkuList);
    }

    public boolean isStockEnoughOfActivity(List<ActivitySku> activitySkuList) {
        if (CollectionUtils.isEmpty(activitySkuList)) {
            return false;
        }
        for (ActivitySku activitySku : activitySkuList) {
            Sku sku = skuService.getById(activitySku.getSku().getId());
            if (sku.getAvailableStockQuantity() < activitySku.getCount())
                return false;
        }
        return true;
    }

    @Cacheable(value = CacheKeys.DEFAULT_NAME, key = CacheKeys.ACTIVITY_KEY_PREV + "#id")
    public Activity getById(Long id) {
        return entityDao.getById(id);
    }

    public List<Activity> findByIdIn(List<Long> ids) {
        if (Collections3.isEmpty(ids)) {
            return Collections.emptyList();
        } else {
            return entityDao.findByIdIn(ids);
        }
    }

    public void buildDetail(Activity activity) {
        if (activity == null) {
            return;
        }
        // 组装 ActivitySku
        List<ActivitySku> activitySkus = activitySkuService.findByActivityId(activity.getId());
        activity.setActivitySkus(activitySkus);

        for (ActivitySku activitySku : activitySkus) {
            activitySku.setSku(skuService.getById(activitySku.getSku().getId()));
            Sku sku = activitySku.getSku();
            sku.setProductImages(productImageService.getBySkuId(sku.getId()));
            if (sku.getProduct() != null) {
                Product product = productService.getById(sku.getProduct().getId());
                sku.setProduct(product);
                product.setSupplier(supplierService.getById(product.getSupplier().getId()));
                product.setBrand(brandService.getById(product.getBrand().getId()));
                product.setCatalogs(productCatalogService.findCatalogByProductId(product.getId()));
                product.setDescription(descriptionService.findByProductId(product.getId()));
                product.setProductImages(productImageService.getProductPrimaryImages(product.getId()));
            }

        }

    }

    @Deprecated
    public void buildDetail(List<Activity> activities) {
        for (Activity activity : activities) {
            buildDetail(activity);
        }
    }

    @Deprecated
    public void buildCartActivityDetail(Activity activity) {
        List<ActivitySku> activitySkus = activitySkuService.findByActivityId(activity.getId());
        activity.setActivitySkus(activitySkus);
        for (ActivitySku activitySku : activitySkus) {
            activitySku.setActivity(activity);
            activitySku.setSku(skuService.getById(activitySku.getSku().getId()));
            Sku sku = activitySku.getSku();
            sku.setProductImages(productImageService.getBySkuId(sku.getId()));
            if (sku.getProduct() != null) {
                Product product = productService.getById(sku.getProduct().getId());
                sku.setProduct(product);
                product.setSupplier(supplierService.getById(product.getSupplier().getId()));
                product.setProductImages(productImageService.getProductPrimaryImages(product.getId()));
            }
        }
    }

    /**
     * 更新活动. 注意，要同时设置上架时间和下架时间，否则本方法不保证下架时间大于上架时间
     *
     * @param activity
     */
    @CacheEvict(value = CacheKeys.DEFAULT_NAME, key = CacheKeys.ACTIVITY_KEY_PREV + "#activity.id")
    @Transactional(isolation = Isolation.READ_COMMITTED, propagation = Propagation.REQUIRED)
    public void update(Activity activity) {
        Activity oldActivity = getById(activity.getId());

        if (entityDao.findByActivityName(activity.getName(), activity.getId()) != null) {
            throw new ServiceException("活动名称已经存在");
        }

        if (!(oldActivity.getStatus() == Status.DRAFT || oldActivity.getStatus() == Status.REFUSE)) {
            throw new ServiceException("当前状态为 " + oldActivity.getStatus().getLabel() + ",活动不允许编辑");
        }
        if (!activity.getEndTime().after(activity.getStartTime())) {
            throw new ServiceException("结束时间应该大于开始时间");
        }

        if (!activity.getEndTime().after(new Date())) {
            throw new ServiceException("结束时间应该大于当前时间");
        }
        activity.setStatus(Status.DRAFT);
        entityDao.update(activity);

        // 更新activitySkus
        if (activity.getActivitySkus() != null) {
            // 删除数据库中多余的
            List newActivitySkuIds = Collections3.extractToList(activity.getActivitySkus(), IdEntity.ID_FIELD_NAME);
            List<ActivitySku> oldActivitySkus = activitySkuService.findByActivityId(activity.getId());
            for (ActivitySku oldActivitySku : oldActivitySkus) {
                if (!newActivitySkuIds.contains(oldActivitySku.getId())) {
                    activitySkuService.deleteById(oldActivitySku.getId());
                }
            }

            // 更新或添加
            for (ActivitySku activitySku : activity.getActivitySkus()) {
                if (activitySku == null || activitySku.getSku() == null) {
                    continue;
                }
                activitySku.setActivity(activity);
                if (activitySku.getId() == null) {
                    activitySkuService.insert(activitySku);
                } else {
                    activitySkuService.update(activitySku);
                }
            }
        }
    }

    @CacheEvict(value = CacheKeys.DEFAULT_NAME, key = CacheKeys.ACTIVITY_KEY_PREV + "#id")
    public void updateStatus(Long id, Activity.Status status) {
        entityDao.updateStatus(id, status);
    }

    /**
     * 更新下架时间, 可将下架时间设为null
     *
     * @param id         活动ID
     * @param delistTime 下架时间，可为null
     */
    @CacheEvict(value = CacheKeys.DEFAULT_NAME, key = CacheKeys.ACTIVITY_KEY_PREV + "#id")
    public void updateDelistTime(Long id, Date delistTime) {
        entityDao.updateDelistTime(id, delistTime);
    }

    @CacheEvict(value = CacheKeys.DEFAULT_NAME, key = CacheKeys.ACTIVITY_KEY_PREV + "#id")
    public void delete(Long id) {
        activitySkuService.deleteByActivityId(id);
        entityDao.deleteById(id);
    }

    public Page<Activity> adminSearch(Map<String, Object> params, Pageable pageable) {

        // 查询
        List<Activity> activitys;
        Long total = entityDao.searchTotal(params);
        if (total > pageable.getOffset()) {
            params.put(Constants.PAGE_OFFSET, pageable.getOffset());
            params.put(Constants.PAGE_SIZE, pageable.getPageSize());
            params.put(Constants.PAGE_SORT, pageable.getSort());
            activitys = entityDao.search(params);
        } else {
            activitys = Collections.emptyList();
        }

        return new PageImpl<Activity>(activitys, pageable, total);
    }

    /**
     * 修改活动状态.
     *
     * @throws com.rocoinfo.exception.ServiceException 如果不可从当前状态改为目标状态
     */
    @CacheEvict(value = CacheKeys.DEFAULT_NAME, key = CacheKeys.ACTIVITY_KEY_PREV + "#id")
    private void changeStatus(long id, Activity.Status newStatus) {
        Activity activity = getById(id);

        if (activity.getStatus() != newStatus) {
            if (Status.isAllowChangeStatus(activity.getStatus(), newStatus)) {
                Activity activity1 = new Activity();
                activity1.setId(id);
                activity1.setStatus(newStatus);
                entityDao.update(activity1);
            } else {
                throw new ServiceException("活动状态不能从 " + activity.getStatus().getLabel() + " 改为 " + newStatus.getLabel());
            }
        }
    }

    /**
     * 弃用
     */
    @CacheEvict(value = CacheKeys.DEFAULT_NAME, key = CacheKeys.ACTIVITY_KEY_PREV + "#id")
    public void discard(Long id) {
        changeStatus(id, Status.DISCARD);
    }

    /**
     * 提交审核
     */
    @CacheEvict(value = CacheKeys.DEFAULT_NAME, key = CacheKeys.ACTIVITY_KEY_PREV + "#id")
    public void submitAudit(Long id) {
        changeStatus(id, Status.AUDIT);
    }

    /**
     * 审核拒绝
     */
    @CacheEvict(value = CacheKeys.DEFAULT_NAME, key = CacheKeys.ACTIVITY_KEY_PREV + "#id")
    public void auditRefuse(Long id) {
        changeStatus(id, Status.REFUSE);
    }

    /**
     * 审核通过
     */
    @CacheEvict(value = CacheKeys.DEFAULT_NAME, key = CacheKeys.ACTIVITY_KEY_PREV + "#id")
    public void auditOk(Long id) {
        changeStatus(id, Status.READY_LIST);
    }

    /**
     * 下架.
     */
    @CacheEvict(value = CacheKeys.DEFAULT_NAME, key = CacheKeys.ACTIVITY_KEY_PREV + "#id")
    public void delist(Long id) {
        List<ActivitySku> activitySkus = activitySkuService.findByActivityId(id);
        for (ActivitySku actSku : activitySkus) {
            List<Activity> exclusiveActivitys = activitySkuService.getExclusiveActivityCountByskuId(actSku.getSku().getId());
            List<Long> activityIds = null;
            boolean iscontain = false;
            if (!CollectionUtils.isEmpty(exclusiveActivitys)) {
                activityIds = Collections3.extractToList(exclusiveActivitys, "id");
                for (Long aid : activityIds) {
                    if (StringUtils.equals(String.valueOf(aid), String.valueOf(id))) {
                        iscontain = true;
                        break;
                    }
                }
            }
        }
        changeStatus(id, Status.DELIST);
    }

    /**
     * @throws com.rocoinfo.exception.ServiceException 如果当前状态不可上架
     */
    @CacheEvict(value = CacheKeys.DEFAULT_NAME, key = CacheKeys.ACTIVITY_KEY_PREV + "#id")
    public void list(Long id) {
        list(getById(id), null);
    }

    @CacheEvict(value = CacheKeys.DEFAULT_NAME, key = CacheKeys.ACTIVITY_KEY_PREV + "#activity.id")
    public void list(Activity activity, Long userId) {
        if (activity.getType() == Type.SCRAPE_CARD || activity.getType() == Type.TURN_TABLE) {
            if (findActivityByStatusAndType(Status.LIST, activity.getType()) != null) {
                throw new ServiceException("已经存在一个上架的" + activity.getType().getLabel() + "活动");
            }
        }
        if (!Status.isAllowChangeStatus(activity.getStatus(), Status.LIST)) {
            throw new ServiceException("活动状态不能从 " + activity.getStatus().getLabel() + " 改为 " + Status.LIST.getLabel());
        }
        buildActivityDetail(Lists.newArrayList(activity), false);
        // 检查商品是否都是已上架状态
        List<Long> productIds = activitySkuService.findProductIdByActivityId(activity.getId());
        for (Long productId : productIds) {
            Product product = productService.getById(productId);
            if (product.getStatus() != Product.Status.LIST) {
                throw new ServiceException("商品未上架(请先上架该商品). 商品名称=" + product.getName());
            }
            Date endTime = product.getDelistTime();
            if (endTime != null && activity.getEndTime().after(product.getDelistTime())) {
                throw new ServiceException("商品下架时间不能早于活动下架时间(请延长商品下架时间).商品名称=" + product.getName());
            }
        }

        // 一个SKU同一时间只能有一个活动，下架其他活动
        List<ActivitySku> activitySkus = activity.getActivitySkus();
        for (ActivitySku activitySku : activitySkus) {
            Sku sku = activitySku.getSku();
            if (activity.getType() != Type.AUCTION) {
                // 竞拍不需要验证库存
                if (sku.getAvailableStockQuantity() <= 0) {
                    throw new ServiceException("商品库存不足(请先添加商品库存).商品编码=" + activitySku.getSku().getCode());
                }
            }
            if (sku.getStatus() != com.rocoinfo.rocomall.entity.Sku.Status.ENABLED) {
                throw new ServiceException("商品未启用(请先启用该商品).商品编码=" + activitySku.getSku().getCode());
            }
        }
        // 更新活动状态
        Activity updateActivity = new Activity();
        updateActivity.setId(activity.getId());

        updateActivity.setStatus(Status.LIST);
        entityDao.update(updateActivity);

        // 在Sku表中记录活动信息
        /*
         * for (ActivitySku activitySku : activitySkus) { Sku sku = new Sku();
		 * sku.setId(activitySku.getSku().getId());
		 * sku.setListedActivityId(activity.getId());
		 * sku.setListedActivityType(activity.getType());
		 * skuService.update(sku); }
		 */
    }

    /***
     * 判断活动下的商品是否全上架
     *
     * @param activity
     * @return false 有未上架的
     */
    @Deprecated
    public boolean isProductAllList(Activity activity) {
        // 检查商品是否都是已上架状态, 如果存在未上架商品，返回false
        List<Long> productIds = activitySkuService.findProductIdByActivityId(activity.getId());
        for (Long productId : productIds) {
            Product product = productService.getById(productId);
            if (product.getStatus() != Product.Status.LIST) {
                return false;
            }
        }
        return true;
    }

    @CacheEvict(value = CacheKeys.DEFAULT_NAME, key = CacheKeys.ACTIVITY_KEY_PREV + "#activity.id")
    @Transactional(isolation = Isolation.SERIALIZABLE, propagation = Propagation.REQUIRED)
    public void delist(Activity activity, Long userId) {
        if (!Status.isAllowChangeStatus(activity.getStatus(), Status.DELIST)) {
            throw new ServiceException("活动状态不能从 " + activity.getStatus().getLabel() + " 改为 " + Status.LIST.getLabel());
        }

        // 更新活动状态
        Activity activity1 = new Activity();
        activity1.setId(activity.getId());
        activity1.setStatus(Status.DELIST);
        entityDao.update(activity1);

        // 将SKU中活动信息字段设为null
		/*
		 * List<ActivitySku> activitySkus =
		 * activitySkuService.findByActivityId(activity.getId()); for
		 * (ActivitySku activitySku : activitySkus) {
		 * skuService.setListedActivityColumnsNull
		 * (activitySku.getSku().getId()); }
		 */
    }

    /**
     * 获得首页最大排序值的活动
     */
    public List<Activity> findListableTopActivitysGroupByType() {
        List<Activity> topActivityList = this.entityDao.findListableTopActivitysGroupByType(new Date());

        if (!CollectionUtils.isEmpty(topActivityList)) {
            buildActivityDetail(topActivityList, true);
            return topActivityList;
        }
        return Collections.EMPTY_LIST;
    }

    /**
     * @param status 活动状态
     * @param type   活动类型
     * @return
     */
    public List<Activity> findActivitysByStatusAndType(Activity.Status status, Activity.Type type) {
        return entityDao.findActivityByStatusAndType(status, type);
    }

    /**
     * 返回单个活动
     *
     * @param status 活动状态
     * @param type   活动类型
     * @return
     */
    public Activity findActivityByStatusAndType(Activity.Status status, Activity.Type type) {
        return Collections3.getFirst(findActivitysByStatusAndType(status, type));
    }

    private ActivityService getProxy() {
        return (ActivityService) AopContext.currentProxy();
    }

    /**
     * 前台活动列表页面展现
     *
     * @param params
     * @param pageable
     * @return
     */
    public Page<Activity> search(Map<String, Object> params, Pageable pageable) {

        // 查询
        List<Activity> activitys = Collections.emptyList();
        Long total = entityDao.searchTotal(params);
        if (total > pageable.getOffset()) {
            params.put(Constants.PAGE_OFFSET, pageable.getOffset());
            params.put(Constants.PAGE_SIZE, pageable.getPageSize());
            params.put(Constants.PAGE_SORT, pageable.getSort());
            activitys = entityDao.search(params);
            buildActivityDetail(activitys, true);
        }
        return new PageImpl<Activity>(activitys, pageable, total);
    }

    public void buildActivityDetail(List<Activity> activitys, boolean isLoadImg) {
        if (!CollectionUtils.isEmpty(activitys)) {
            Map<Long, List<ActivitySku>> skuListMap = this.activitySkuService.findListMapByActivityIdIn(Collections3.extractToList(activitys, "id"));
            List<Sku> loadImgSkuList = Lists.newArrayList();
            for (Activity activity : activitys) {
                activity.setActivitySkus(skuListMap.get(activity.getId()));
                if (!CollectionUtils.isEmpty(activity.getActivitySkus())) {
                    for (ActivitySku actSku : activity.getActivitySkus()) {
                        loadImgSkuList.add(actSku.getSku());
                    }
                }
            }

            if (isLoadImg) {
                this.skuService.loadImages(loadImgSkuList);
            }
        }
    }

    /**
     * 查找需要生成退积分数据的竞拍活动
     *
     * @return
     */
    public List<Activity> findNeedCreateAuctionRefundRecodeActivity() {
        return entityDao.findNeedCreateAuctionRefundRecodeActivity();
    }

    @Override
    public void buildActivityDetail(Activity activity, boolean isLoadImg) {
        if (activity != null) {
            Map<Long, List<ActivitySku>> skuListMap = this.activitySkuService.findListMapByActivityIdIn(Collections3.extractToList(
                    Lists.newArrayList(activity), "id"));
            List<Sku> loadImgSkuList = Lists.newArrayList();
            activity.setActivitySkus(skuListMap.get(activity.getId()));
            if (!CollectionUtils.isEmpty(activity.getActivitySkus())) {
                for (ActivitySku actSku : activity.getActivitySkus()) {
                    loadImgSkuList.add(actSku.getSku());
                }
            }
            if (isLoadImg) {
                this.skuService.loadImages(loadImgSkuList);
            }
        }
    }
}
