package com.rocoinfo.rocomall.service.impl.dict;

import com.google.common.collect.Maps;
import com.rocoinfo.rocomall.common.service.CrudService;
import com.rocoinfo.rocomall.common.service.ServiceException;
import com.rocoinfo.rocomall.entity.dict.DictWarehouse;
import com.rocoinfo.rocomall.enumconst.Status;
import com.rocoinfo.rocomall.redis.CacheKeys;
import com.rocoinfo.rocomall.repository.dict.DictWarehouseDao;
import com.rocoinfo.rocomall.service.dict.IDictWarehouseService;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.aop.framework.AopContext;
import org.springframework.cache.Cache;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * <dl>
 * <dd>描述:仓库字典服务</dd>
 * <dd>公司: 大城若谷信息技术有限公司</dd>
 * <dd>创建时间：15/11/11 上午11:50</dd>
 * <dd>创建人： Johnny</dd>
 * </dl>
 */
@Service
public class DictWarehouseService extends CrudService<DictWarehouseDao, DictWarehouse> implements IDictWarehouseService {

	@Override
	protected void evictCacheEntitysHook() {
		String allCatalogCacheKey = StringUtils.substringBetween(CacheKeys.DICT_WAREHOUSE_ALL_KEY, "'");
		Cache cache = cacheManager.getCache(CacheKeys.DEFAULT_NAME);
		cache.evict(allCatalogCacheKey);
	}

	@Override
	@Cacheable(value = CacheKeys.DEFAULT_NAME, key = CacheKeys.DICT_WAREHOUSE_ALL_KEY, unless = "#result==null || #result.isEmpty()")
	public List<DictWarehouse> findAll() {
		return super.findAll();
	}

	public Map<Long, DictWarehouse> findAllWarehouseMap() {
		Map<Long, DictWarehouse> warehouseMap = Maps.newLinkedHashMap();

		List<DictWarehouse> allWarehouse = getProxy().findAll();
		if (CollectionUtils.isNotEmpty(allWarehouse)) {
			for (DictWarehouse exp : allWarehouse) {
				warehouseMap.put(exp.getId(), exp);
			}
		}
		return warehouseMap;
	}

	public boolean isExistCode(final String warehouseCode) {
		if (StringUtils.isBlank(warehouseCode)) {
			return false;
		}
		for (DictWarehouse exp : getProxy().findAll()) {
			if (exp != null && exp.getCode().equalsIgnoreCase(warehouseCode)) {
				return true;
			}
		}
		return false;
	}

	public boolean isExistName(final String warehouseName) {
		if (StringUtils.isBlank(warehouseName)) {
			return false;
		}
		for (DictWarehouse exp : getProxy().findAll()) {
			if (exp != null && exp.getName().equalsIgnoreCase(warehouseName)) {
				return true;
			}
		}
		return false;
	}

	// 根据编号查询快递信息
	public Map<String, DictWarehouse> findWarehouseMapByCodesIn(List<String> warehouseCodes) {
		Map<String, DictWarehouse> expressMap = Maps.newHashMap();
		if (CollectionUtils.isNotEmpty(warehouseCodes)) {
			for (DictWarehouse exp : getProxy().findAll()) {
				if (exp != null && warehouseCodes.contains(exp.getCode())) {
					expressMap.put(exp.getCode(), exp);
				}
			}
		}
		return expressMap;
	}

	private DictWarehouseService getProxy() {
		return (DictWarehouseService) AopContext.currentProxy();
	}

	@Override
	public void insert(DictWarehouse warehouse) {
		if (isExistCode(warehouse.getCode())) {
			throw new ServiceException("快递公司编号已存在");
		}
		if (isExistName(warehouse.getName())) {
			throw new ServiceException("快递公司名称已存在");
		}
		super.insert(warehouse);
	}

	@Override
	public void update(DictWarehouse warehouse) {
		DictWarehouse oldWarehouse = getById(warehouse.getId());
		if (!oldWarehouse.getName().equalsIgnoreCase(warehouse.getName()) && isExistName(warehouse.getName())) {
			throw new ServiceException("快递公司名称已存在,请换个名称");
		}
		super.update(warehouse);
	}

	public List<DictWarehouse> findAllWithStatus(Status state) {
		List<DictWarehouse> all = getProxy().findAll();
		if (state == null) {
			return all;
		}

		List<DictWarehouse> opens = new ArrayList<DictWarehouse>();
		List<DictWarehouse> locks = new ArrayList<DictWarehouse>();
		for (DictWarehouse advType : all) {
			if (advType.getState() == Status.OPEN) {
				opens.add(advType);
			} else {
				locks.add(advType);
			}
		}
		if (state == Status.OPEN) {
			return opens;
		} else {
			return locks;
		}
	}
}
