package com.rocoinfo.rocomall.service.impl.dict.express;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.rocoinfo.rocomall.common.service.CrudService;
import com.rocoinfo.rocomall.common.service.ServiceException;
import com.rocoinfo.rocomall.entity.dict.DictProviceCity;
import com.rocoinfo.rocomall.entity.dict.express.DictExpress;
import com.rocoinfo.rocomall.enumconst.Status;
import com.rocoinfo.rocomall.redis.CacheKeys;
import com.rocoinfo.rocomall.repository.dict.express.DictExpressDao;
import com.rocoinfo.rocomall.service.dict.express.IDictExpressService;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.aop.framework.AopContext;
import org.springframework.cache.Cache;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;

import java.util.*;

@Service
public class DictExpressService extends CrudService<DictExpressDao, DictExpress> implements IDictExpressService {

	public Collection<DictExpress> findAllOpenedExpress() {
		List<DictExpress> openedList = Lists.newArrayList();
		for (DictExpress express : findAllExpressMap().values()) {
			if (express.getStatus() == Status.OPEN) {
				openedList.add(express);
			}
		}

		return openedList;
	}

	@Override
	protected void evictCacheEntitysHook() {
		String allCatalogCacheKey = StringUtils.substringBetween(CacheKeys.DICT_EXPRESS_ALL_KEY, "'");
		Cache cache = cacheManager.getCache(CacheKeys.DEFAULT_NAME);
		cache.evict(allCatalogCacheKey);
	}

	@Override
	@Cacheable(value = CacheKeys.DEFAULT_NAME, key = CacheKeys.DICT_EXPRESS_ALL_KEY, unless = "#result==null || #result.isEmpty()")
	public List<DictExpress> findAll() {
		return super.findAll();
	}

	public Map<Long, DictExpress> findAllExpressMap() {
		Map<Long, DictExpress> expressMap = Maps.newLinkedHashMap();

		List<DictExpress> allExpress = getProxy().findAll();
		if (CollectionUtils.isNotEmpty(allExpress)) {
			for (DictExpress exp : allExpress) {
				expressMap.put(exp.getId(), exp);
			}
		}
		return expressMap;
	}

	public boolean isExistCode(final String expressCode) {
		if (StringUtils.isBlank(expressCode)) {
			return false;
		}
		for (DictExpress exp : getProxy().findAll()) {
			if (exp != null && exp.getCode().equalsIgnoreCase(expressCode)) {
				return true;
			}
		}
		return false;
	}

	public boolean isExistName(final String expressName) {
		if (StringUtils.isBlank(expressName)) {
			return false;
		}
		for (DictExpress exp : getProxy().findAll()) {
			if (exp != null && exp.getName().equalsIgnoreCase(expressName)) {
				return true;
			}
		}
		return false;
	}

	// 根据编号查询快递信息
	public Map<String, DictExpress> findExpressMapByCodesIn(List<String> expressCodes) {
		Map<String, DictExpress> expressMap = Maps.newHashMap();
		if (CollectionUtils.isNotEmpty(expressCodes)) {
			for (DictExpress exp : getProxy().findAll()) {
				if (exp != null && expressCodes.contains(exp.getCode())) {
					expressMap.put(exp.getCode(), exp);
				}
			}
		}
		return expressMap;
	}

	private DictExpressService getProxy() {
		return (DictExpressService) AopContext.currentProxy();
	}

	@Override
	public void insert(DictExpress express) {
		if (isExistCode(express.getCode())) {
			throw new ServiceException("快递公司编号已存在");
		}
		if (isExistName(express.getName())) {
			throw new ServiceException("快递公司名称已存在");
		}
		super.insert(express);
	}

	@Override
	public void update(DictExpress express) {
		DictExpress oldExpress = getById(express.getId());
		if (!oldExpress.getName().equalsIgnoreCase(express.getName()) && isExistName(express.getName())) {
			throw new ServiceException("快递公司名称已存在,请换个名称");
		}
		super.update(express);
	}

	public List<DictExpress> findAllWithStatus(Status status) {
		List<DictExpress> all = getProxy().findAll();
		if (status == null) {
			return all;
		}

		List<DictExpress> opens = new ArrayList<DictExpress>();
		List<DictExpress> locks = new ArrayList<DictExpress>();
		for (DictExpress advType : all) {
			if (advType.getStatus() == Status.OPEN) {
				opens.add(advType);
			} else {
				locks.add(advType);
			}
		}
		if (status == Status.OPEN) {
			return opens;
		} else {
			return locks;
		}
	}

	/**
	 * 根据快递公司ID 查询快递公司绑定的所有城市
	 * 
	 * @param id
	 * @return 2015年11月23日 下午5:48:26
	 */
	@Override
	@Deprecated
	public List<DictProviceCity> findAllCityById(Long id) {
		if (id == null)
			throw new ServiceException("快递公司Id不能为空");

		return entityDao.findAllCityById(id);
	}

	/**
	 * 根据快递公司ID 查询快递公司绑定的所有城市Id
	 * 
	 * @param id 快递Id
	 */
	public List<Long> findBoundedCityIdsById(Long id) {
		if (id == null || id < 1)
			return Collections.emptyList();

		return this.entityDao.findBoundedCityIdsById(id);
	}

	/**
	 * 通过城市Id，获得绑定的快递公司,如果城市尚未绑定快递，则返回null
	 */
	public DictExpress findBoundExpressByCityId(Long cityId) {
		List<DictProviceCity> expCityList = findCityExpressByBoundedCityIds(Lists.newArrayList(cityId));
		if (CollectionUtils.isNotEmpty(expCityList)) {
			return expCityList.get(0).getExpress();
		}
		return null;
	}

	/**
	 * 通过城市id查找 绑定的快递信息
	 * 
	 * @param cityIds 城市Ids
	 * @return 绑定的城市信息与快递信息
	 */
	public List<DictProviceCity> findCityExpressByBoundedCityIds(List<Long> cityIds) {
		if (CollectionUtils.isEmpty(cityIds)) {
			return Collections.emptyList();
		}
		return this.entityDao.findCityExpressByBoundedCityIds(cityIds);
	}

	@Override
	public void updateCitys(DictExpress express) {
		if (express == null || express.getId() == null)
			throw new ServiceException("快递公司Id不能为空");
		entityDao.deleteAllCitysById(express.getId());
		if (express.getCitys() != null && express.getCitys().size() > 0)
			entityDao.insertCitys(express);
	}
}
