package com.rocoinfo.rocomall.service.impl.order;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.rocoinfo.rocomall.Constants;
import com.rocoinfo.rocomall.common.service.CrudService;
import com.rocoinfo.rocomall.dto.StatusDto;
import com.rocoinfo.rocomall.dto.admin.OrderDetailsDto;
import com.rocoinfo.rocomall.dto.admin.OrderListDto;
import com.rocoinfo.rocomall.entity.Address;
import com.rocoinfo.rocomall.entity.ProductImage;
import com.rocoinfo.rocomall.entity.Sku;
import com.rocoinfo.rocomall.entity.account.User;
import com.rocoinfo.rocomall.entity.order.Order;
import com.rocoinfo.rocomall.entity.order.OrderItem;
import com.rocoinfo.rocomall.entity.order.OrderItem.OrderType;
import com.rocoinfo.rocomall.entity.order.OrderItem.Status;
import com.rocoinfo.rocomall.entity.params.OrderParams;
import com.rocoinfo.rocomall.repository.OrderItemOperationLogDao;
import com.rocoinfo.rocomall.repository.order.OrderDao;
import com.rocoinfo.rocomall.repository.order.OrderItemDao;
import com.rocoinfo.rocomall.service.IUserService;
import com.rocoinfo.rocomall.service.impl.AddressService;
import com.rocoinfo.rocomall.service.order.INewOrderService;
import com.rocoinfo.rocomall.service.order.IOrderItemService;
import com.rocoinfo.rocomall.service.order.IOrderService;
import com.rocoinfo.rocomall.service.product.IProductImageService;
import com.rocoinfo.rocomall.service.product.ISkuService;
import com.rocoinfo.rocomall.utils.CodeGenerator;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Isolation;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springside.modules.utils.Collections3;

import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Service
public class NewOrderService extends CrudService<OrderDao, Order> implements INewOrderService {

    @Autowired
    private AddressService addressService;
    @Autowired
    private ISkuService skuService;
    @Autowired
    private IUserService userService;
    @Autowired
    private CodeGenerator codeGenerator;
    @Autowired
    private OrderDao orderDao;
    @Autowired
    private OrderItemDao orderItemDao;
    @Autowired
    private IOrderItemService orderItemService;
    @Autowired
    private IOrderService orderService;
    @Autowired
    private IProductImageService productImageService;
    @Autowired
    private OrderItemOperationLogDao operLogDao;

    @Transactional(isolation = Isolation.READ_COMMITTED)
    public Object createOrder(OrderParams orderParams) {
        //获取用户信息
        User user = userService.getById(orderParams.getUserId());
        if (user == null) {
            return StatusDto.buildFailureStatusDto("用户不存在！");
        }

        //1.根据商品skuId,查询sku商品信息
        Sku sku = skuService.buildDetail(orderParams.getSkuId(), false);


        //2. 判断商品库存是否充足
        if (!isCanBuy(sku, orderParams.getQuantity())) {
            return StatusDto.buildFailureStatusDto("该商品库存不足！");
        }
        //3. 查询用户收货地址信息
        Address address = null;
        if (!orderParams.getIsInvite()) {
        	address = addressService.getById(orderParams.getAddressId());
        	if(address == null){
        		return StatusDto.buildFailureStatusDto("收货地址不能为空！");
        	}
        }
        //4.为用户创建订单
        Order order = buildOrder(user, sku, orderParams);

        //6.创建子订单
        OrderItem orderItem = buildOrderItem(sku, order, orderParams.getQuantity());

        order.setItems(Lists.newArrayList(orderItem));
        //7.保存订单、订单子项，并修改占用库存信息
        insert(order);
        
        Map<String, Object> orderInfo = new HashMap<String, Object>();
        orderInfo.put("orderId", order.getId());
        return StatusDto.buildDataSuccessStatusDto("操作成功！",orderInfo);
    }

    /**
     * 创建子订单
     *
     * @param order    父订单
     * @param quantity 商品对象
     * @return
     */
    private OrderItem buildOrderItem(Sku sku, Order order, int quantity) {
        OrderItem orderItem = new OrderItem();
        orderItem.setOrderCode(order.getOrderCode());
        orderItem.setOrder(order);
        orderItem.setSku(sku);

        orderItem.setUser(order.getUser());
        orderItem.setOrderType(OrderType.common);
        orderItem.setPrice(sku.getCashAmt());
        orderItem.setQuantity(quantity);
        orderItem.setAttributes(buildAttributes(sku));

        orderItem.setStatus(OrderItem.Status.NOT_PAID);//待支付
        orderItem.setAdmStatus(OrderItem.AdmStatus.NOT_PAID);//待支付，后台状态
        orderItem.setSupplier(sku.getProduct().getSupplier());
        orderItem.setProductName(sku.getProduct().getName());
        orderItem.setCreateTime(new Date());
        return orderItem;
    }


    /**
     * 构建购买商品的属性值
     *
     * @param sku 购买商品的sku
     * @return
     */
    private String buildAttributes(Sku sku) {
        StringBuffer attributes = new StringBuffer();
        String attribute = sku.getAttribute1();
        if (StringUtils.isNotEmpty(attribute)) {
            attributes.append(attribute).append(" ");
        }
        String attribute2 = sku.getAttribute2();
        if (StringUtils.isNotEmpty(attribute2)) {
            attributes.append(attribute2).append(" ");
        }
        String attribute3 = sku.getAttribute3();
        if (StringUtils.isNotEmpty(attribute3)) {
            attributes.append(attribute3).append(" ");
        }
        return attributes.toString();
    }

    /**
     * 创建订单信息
     *
     * @param user 用户信息
     * @param sku  sku
     * @return
     */
    private Order buildOrder(User user, Sku sku, OrderParams orderParams) {
        Order order = new Order();
        order.setIsInvite(orderParams.getIsInvite());
        order.setAddressId(orderParams.getAddressId());
        
        order.setUser(user);
        order.setCreateTime(new Date());
        order.setOrderCode(codeGenerator.generateOrderCode(new Date(), OrderType.common));
        order.setDeliverFee(orderParams.getCost());//设置运费
        if(orderParams.getCost() != null){
        	order.setCashAmt(sku.getCashAmt() * orderParams.getQuantity()+orderParams.getCost());
        }else{
        	order.setCashAmt(sku.getCashAmt() * orderParams.getQuantity());
        }
        order.setCreateTime(new Date());
        order.setOrderCode(codeGenerator.generateOrderCode(new Date(), OrderType.common));
        order.setDistributed(false);
        order.setLocked(false);
        order.setCashPayed(true);
        order.setCent(0);
        order.setCentPayed(false);
        order.setNote(orderParams.getNote());

        return order;
    }

    /**
     * 判断商品是否可以购买
     *
     * @param sku      所买商品sku
     * @param quantity 购买商品数量
     * @return
     */
    private boolean isCanBuy(Sku sku, Integer quantity) {
        if (sku == null) {
            return false;
        }

        if (sku.getAvailableStockQuantity() < quantity) {
            return false;
        }
        return true;
    }

    /**
     * 保存订单子项信息，并修改库存信息
     */
    @SuppressWarnings("all")
	public void insert(Order order) {
        super.insert(order);
        //设置订单id
        order.setId(orderDao.getByOrderNo(order.getOrderCode()).getId());
        String consumeTypeCode = OrderItem.OrderType.common.getCode();
        if (!CollectionUtils.isEmpty(order.getItems())) {
            consumeTypeCode = Collections3.getFirst(order.getItems()).getOrderType().getCode();
            //设置订单子项的订单号,保存子订单
            int i = 0;
            for (OrderItem item : order.getItems()) {
                i++;
                item.setOrderCode(order.getOrderCode() + "0" + i);
                orderItemService.insert(item);
            }
            List<Long> skuIdList = Lists.newArrayList();
            List<Integer> qtyList = Lists.newArrayList();

            for (OrderItem item : order.getItems()) {
                skuIdList.add(item.getSku().getId());
                qtyList.add(item.getQuantity());
            }
            skuService.batchIncreaseOccupiedStockAndSaleVolumeOnPayed(skuIdList, qtyList);

        }
    }

    @Transactional(isolation = Isolation.READ_COMMITTED)
    public Object payOrder(OrderParams orderParams) {
        //1. 根据订单id获取订单详细信息
        Order order = orderService.getById(orderParams.getOrderId());
        //2.构建订单详情，包括子订单信息
        order.setItems(orderItemService.findByOrderId(order.getId()));

        //3.验证订单是否是待支付状态
        for(OrderItem item : order.getItems()){
        	if(!OrderItem.Status.NOT_PAID.equals(item.getStatus())){
        		return StatusDto.buildFailureStatusDto("该订单不能支付！"); 
        	}
        }
        //4.修改订单子项状态，设置为付款成功，设置支付方式
        payOrder(order,orderParams);
        
        return StatusDto.buildDataSuccessStatusDto("操作成功！");
    }

    /**
     * 更新订单子项状态为付款成功，设置支付方式
     * @param order 订单详情
     */
    private void payOrder(Order order,OrderParams orderParams) {
		order.setCashPayType(orderParams.getPaymode());//设置支付方式
		order.setPayedTime(new Date());
		orderService.update(order);
		//修改订单状态，并更新到数据库
		for(OrderItem item : order.getItems()){
			item.setStatus(OrderItem.Status.PAID);
			item.setAdmStatus(OrderItem.AdmStatus.PAID);
			orderItemService.update(item);
		}
		
	}

    
	@Override
	public Object findOrderItemList(PageRequest pageable, Long userId, String status) {
		//1.判断所传订单状态是否在 用户可见状态中
		OrderItem.Status orderStatus= null;
		if(!StringUtils.isEmpty(status)){
			orderStatus = inStatus(status);
			if(orderStatus == null)
				return StatusDto.buildFailureStatusDto("您查询订单状态有误！"); 
		}
		
		Map<String,Object> params = Maps.newHashMap();
		params.put("status", orderStatus);
		params.put("userId", userId);
		params.put("statusList", OrderItem.userCanSeeStatus);
		//获取订单列表
		List<OrderItem> items = Lists.newArrayList();
		long total = orderItemDao.searchTotal(params);//获取订单总数
		if (total > pageable.getOffset()) {
            params.put(Constants.PAGE_OFFSET, pageable.getOffset());
            params.put(Constants.PAGE_SIZE, pageable.getPageSize());
            params.put(Constants.PAGE_SORT, pageable.getSort());
            items = orderItemDao.search(params);
        }
		//构建订单列表参数
		List<OrderListDto> orderListDto = Lists.newArrayList();
		for(OrderItem item : items){
			OrderListDto orderDto = buildOrderDtoList(item);
			orderListDto.add(orderDto);
		}
		
		return StatusDto.buildDataSuccessStatusDto("获取列表成功！",new PageImpl<OrderListDto>(orderListDto, pageable, total));
		
	}
	/**
	 * 由订单子项构建订单信息
	 * @param items 子订单
	 * @return
	 */
	private OrderListDto buildOrderDtoList(OrderItem item) {
		
		Long skuId = item.getSku().getId();
		Sku sku = skuService.getById(skuId);
		List<ProductImage> productImages = productImageService.getProductPrimaryImages(sku.getProduct().getId());
		String prodImage = "";
		if(productImages != null && productImages.size() > 0 ){
			prodImage = productImages.get(0).getImageUrl();
		}
		return  new OrderListDto.Builder(item.getOrder().getId(), item.getOrder().getCashAmt()).attributes(item.getAttributes())
		.prodName(item.getProductName()).prodImg(prodImage).quantity(item.getQuantity()).status(item.getStatus()).build();
	}


	/**
	 * 判断所传状态是否在用户可见的四种中
	 * @param status
	 * @return 在：四种状态中，返回状态枚举对象  不在：null
	 */
	private Status inStatus(String status) {
		for(OrderItem.Status orderItemStatus :OrderItem.userCanSeeStatus){
			if(orderItemStatus.toString().equals(status)){
				return orderItemStatus;
			}
		}
		return null;
	}

	@Override
	public Object details(Long orderId) {
		//1.获取订单信息
		Order order = orderService.getById(orderId);
		if(order == null)
			return StatusDto.buildDataFailureStatusDto("对不起，您要查看的订单不存在！");
		
		//2.获取订单子项信息
		order.setItems(orderItemService.findByOrderId(order.getId()));
		
		//3.构建订单详情
		orderService.buildOrderItems(Lists.newArrayList(order),true,null);
		
		//4.构建订单详情结构
		OrderDetailsDto orderDetails = buildOrderDetailDto(order);
		
		return StatusDto.buildDataSuccessStatusDto("查询订单详情成功！", orderDetails);
	}

	/**
	 * 构建订单详情信息
	 * @param order 订单信息
	 * @return 订单详情dto
	 */
	private OrderDetailsDto buildOrderDetailDto(Order order) {
		//1.构建收货地址信息
		Map<String,Object> address = Maps.newHashMap();
		if(order.getOrderAddress() != null){
			address.put("name", order.getOrderAddress().getConsignee());//姓名
			address.put("phone",order.getOrderAddress().getMobile());//手机号
			address.put("addressDetail", order.getOrderAddress().getProvince()+order.getOrderAddress().getAddress());
		}
		//2.构建订单子项列表
		OrderItem.Status status = null;
		List<Map<String,Object>> items = Lists.newArrayList();
		for(OrderItem item : order.getItems()){
			Map<String,Object> itemInfo = Maps.newHashMap();
			itemInfo.put("itemId", item.getId());
			itemInfo.put("prodImg", item.getSku().getPreviewImagePath());
			itemInfo.put("attributes", item.getAttributes());
			itemInfo.put("quantity", item.getQuantity());
			status = item.getStatus();
			items.add(itemInfo);
		}
		
		return new OrderDetailsDto.Builder(order.getId(),order.getOrderCode()).address(address)
		.items(items).amount(order.getCashAmt()).createTime(order.getCreateTime()).status(status).build();
	
	}

	@Override
	public Object cancel(Long orderId, String cancelReason) {
		//1.获取订单信息
		Order order = orderService.getById(orderId);
		if(order == null)
			return StatusDto.buildDataFailureStatusDto("对不起，您要取消的订单不存在！");
		
		//2.获取订单子项信息
		order.setItems(orderItemService.findByOrderId(order.getId()));
		
		//3.判断订单状态是不是待支付状态，如果不是则不能取消
		if(!isCanOperate(order, OrderItem.Status.NOT_PAID, OrderItem.AdmStatus.NOT_PAID)){
			return StatusDto.buildDataFailureStatusDto("该订单不能取消！");
		}
		
		//4.修改子订单状态、保存取消原因
		for(OrderItem item : order.getItems()){
			item.setStatus(OrderItem.Status.CANCELED);
			item.setAdmStatus(OrderItem.AdmStatus.CANCELED);
			orderItemService.update(item);
		}
		order.setCancelReason(cancelReason);
		orderService.update(order);
		
		//5.修改库存（释放占用库存）
		orderService.releaseOrOccupyStockByOrderId(orderId,1);
		
		
		return StatusDto.buildSuccessStatusDto("订单取消成功！");
	}

	/**
	 * 判断订单是否可以操作
	 * @param order 主订单
	 * @param status 订单前台状态 
	 * @param admStatus 订单后天状态
	 * @return 
	 */
	private boolean isCanOperate(Order order,OrderItem.Status status ,OrderItem.AdmStatus admStatus) {
		if(order != null && order.getItems() != null){
			for(OrderItem item : order.getItems()){
				if(status != null){
					if(!status.equals(item.getStatus())){
						return false;
					}
				}
				if(admStatus != null && !admStatus.equals(item.getAdmStatus())){
					return false;
				}
			}
		}else{
			return false;
		}
		return true;
	}

	@Override
	public Object delete(Long orderId) {
		//1.获取订单信息
		Order order = orderService.getById(orderId);
		if(order == null)
			return StatusDto.buildDataFailureStatusDto("对不起，您要删除的订单不存在！");
		
		//2.获取订单子项信息
		order.setItems(orderItemService.findByOrderId(order.getId()));
		
		//3.判断订单是否可以删除，只能删除支付完成的订单
		if(!isCanOperate(order,null,OrderItem.AdmStatus.FINISH)){
			return StatusDto.buildDataFailureStatusDto("该订单不能删除！");
		}
		
		//4. 修改订单状态
		for(OrderItem item : order.getItems()){
			item.setStatus(OrderItem.Status.DELETED);//已删除
			orderItemService.update(item);
		}
		return StatusDto.buildSuccessStatusDto("删除订单成功！");
	}

	@Override
	public Object remindShipments(Long orderId) {
		// TODO Auto-generated method stub
		return StatusDto.buildSuccessStatusDto("已经提醒卖家发货！");
	}

	/**
	 * 确认收货接口
	 * @param orderId 订单id
	 * @return
	 */
	@Override
	public Object confirmDelivery(Long orderId) {
		//1.获取订单信息
		Order order = orderService.getById(orderId);
		if(order == null)
			return StatusDto.buildDataFailureStatusDto("对不起，您要确认收货的订单不存在！");
		
		//2.获取订单子项信息
		order.setItems(orderItemService.findByOrderId(order.getId()));
		
		//3.判断订单是否可以完成确认收货
		if(!isCanOperate(order,OrderItem.Status.RECEIVING, null)){
			return StatusDto.buildDataFailureStatusDto("对不起，该订单不能确认收货！");
		}
		//4.修改子订单状态为并更新
		for(OrderItem item : order.getItems()){
			item.setStatus(OrderItem.Status.APPRAISING);//待评价
			item.setAdmStatus(OrderItem.AdmStatus.FINISH);//完成
			orderItemService.update(item);
		}
		return StatusDto.buildSuccessStatusDto("确认收货成功！");
	}

}
