package com.rocoinfo.rocomall.service.impl.order;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.rocoinfo.rocomall.Constants;
import com.rocoinfo.rocomall.common.service.CrudService;
import com.rocoinfo.rocomall.common.service.ServiceException;
import com.rocoinfo.rocomall.dto.RefundParam;
import com.rocoinfo.rocomall.dto.StatusDto;
import com.rocoinfo.rocomall.dto.admin.OrderBitchSendImportExcelDto;
import com.rocoinfo.rocomall.dto.admin.OrderBitchSendImportExcelFailDto;
import com.rocoinfo.rocomall.dto.admin.OrderItemExchageRefundParamDto;
import com.rocoinfo.rocomall.entity.*;
import com.rocoinfo.rocomall.entity.account.AdminUser;
import com.rocoinfo.rocomall.entity.account.User;
import com.rocoinfo.rocomall.entity.dict.express.DictExpress;
import com.rocoinfo.rocomall.entity.order.Order;
import com.rocoinfo.rocomall.entity.order.OrderItem;
import com.rocoinfo.rocomall.entity.order.OrderItem.AdmStatus;
import com.rocoinfo.rocomall.enumconst.Status;
import com.rocoinfo.rocomall.redis.CacheKeys;
import com.rocoinfo.rocomall.redis.JedisTemplate;
import com.rocoinfo.rocomall.repository.OrderItemOperationLogDao;
import com.rocoinfo.rocomall.repository.ProdReturnRecordDao;
import com.rocoinfo.rocomall.repository.order.OrderItemDao;
import com.rocoinfo.rocomall.service.IAddressService;
import com.rocoinfo.rocomall.service.IUserService;
import com.rocoinfo.rocomall.service.cent.ICentsService;
import com.rocoinfo.rocomall.service.dict.express.IDictExpressService;
import com.rocoinfo.rocomall.service.impl.ICardDeliverRecordService;
import com.rocoinfo.rocomall.service.order.IOrderItemService;
import com.rocoinfo.rocomall.service.order.IOrderService;
import com.rocoinfo.rocomall.service.product.IProductService;
import com.rocoinfo.rocomall.service.product.ISkuService;
import com.rocoinfo.rocomall.service.product.ISupplierService;
import com.rocoinfo.rocomall.service.product.IVirtualProductService;
import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Isolation;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import org.springside.modules.utils.Collections3;

import java.text.MessageFormat;
import java.text.SimpleDateFormat;
import java.util.*;

/**
 * <dl>
 * <dd>描述: 子订单service</dd>
 * <dd>公司: 大城若谷信息技术有限公司</dd>
 * <dd>创建时间：15/7/31 下午1:36</dd>
 * <dd>创建人： weiys</dd>
 * </dl>
 */
@Service
@Transactional
public class OrderItemService extends CrudService<OrderItemDao, OrderItem> implements IOrderItemService {
    @Autowired
    private OrderItemOperationLogDao operLogDao;
    @Autowired
    private ProdReturnRecordDao prodReturnRecordDao;
    @Autowired
    private ISkuService skuService;
    @Autowired
    private IOrderService orderService;
    @Autowired
    private JedisTemplate jedisTemplate;
    @Autowired
    private IAddressService addressService;
    @Autowired
    private IDictExpressService expressService;
    @Autowired
    private IProductService productService;
    @Autowired
    private ISupplierService supplierService;
    @Autowired
    private IUserService userService;
    @Autowired
    private ICardDeliverRecordService cardDeliverRecordService;
    @Autowired
    private IVirtualProductService virtualProductService;
    @Autowired
    private ICentsService centsService;

    // @Autowired
    // private SmsService smsService;

    @Cacheable(value = CacheKeys.DEFAULT_NAME, key = CacheKeys.SALE_ORDER_ITEMS_IN_ORDER + "#orderId")
    public List<OrderItem> findByOrderId(Long orderId) {
        return super.entityDao.findByOrderId(orderId);
    }

    public OrderItem getSaleItemWithExpressById(Long id) {
        return super.entityDao.getSaleItemWithExpressById(id);
    }

    @CacheEvict(value = CacheKeys.DEFAULT_NAME, key = CacheKeys.SALE_ORDER_ITEM_PREFIX + "#orderItemId")
    public void updateExpressAndTransportNoById(final long expressId, String transportNo, final long orderItemId) {
        if (expressId > 0 && StringUtils.isNotEmpty(transportNo) && orderItemId > 0) {
            OrderItem orderItem = new OrderItem();
            orderItem.setId(orderItemId);
            orderItem.setExpress(new DictExpress(expressId));
            orderItem.setTransportNo(transportNo);
            update(orderItem);
        }
    }

    public List<OrderItem> findWithSkuByOrderIdsIn(List<Long> orderIdList) {
        if (CollectionUtils.isNotEmpty(orderIdList)) {
            return super.entityDao.findWithSkuByOrderIdsIn(orderIdList);
        }
        return Collections.emptyList();
    }

    public Page<OrderItem> adminSearch(Map<String, Object> params, Pageable pageable) {
        Long total = super.entityDao.adminSearchTotal(params);
        List<OrderItem> orderitems = Collections.emptyList();
        if (total > pageable.getOffset()) {
            params.put(Constants.PAGE_OFFSET, pageable.getOffset());
            params.put(Constants.PAGE_SIZE, pageable.getPageSize());
            params.put(Constants.PAGE_SORT, pageable.getSort());
            orderitems = super.entityDao.adminSearch(params);
        }
        return new PageImpl<OrderItem>(orderitems, pageable, total);
    }

    public List<OrderItem> adminExportSearch(Map<String, Object> params) {
        return super.entityDao.adminExportSearch(params);
    }

    /**
     * 取消总订单
     *
     * @param orderId               总订单Id
     * @param loggedUserId          操作人
     * @param isReleaseOccupedStock 释放需要释放占用库存
     */
    public void cancelOrder(final long orderId, Long loggedUserId, boolean isReleaseOccupedStock) {
        Order order = orderService.getById(orderId);

        //根据总订单Id 更新订单状态
        OrderItem updateItem = new OrderItem();
        updateItem.setAdmStatus(OrderItem.AdmStatus.CANCELED);
        updateItem.setStatus(OrderItem.Status.CANCELED);
        updateItem.setOrder(order);
        this.update(updateItem);

        // 需要退的积分数量 TODO
        int refundCent = order.getCent();

        if (isReleaseOccupedStock) {
            orderService.releaseOrOccupyStockByOrderId(orderId, 1);
        }

        // TODO 记录操作日志
    }

    /**
     * 取消子订单
     */
    @Transactional(isolation = Isolation.READ_COMMITTED, propagation = Propagation.REQUIRED)
    @CacheEvict(value = CacheKeys.DEFAULT_NAME, key = CacheKeys.SALE_ORDER_ITEM_PREFIX + "#orderItemId")
    public void cancelOrderItem(final long orderItemId, Long loggedUserId) {

        final OrderItem orderItem = getById(orderItemId);
        if (orderItem != null && orderItem.getStatus() == OrderItem.Status.PAID) {
            OrderItem item = new OrderItem();
            item.setId(orderItemId);
            item.setStatus(OrderItem.Status.CANCELED);
            // 子订单表更新
            update(item);
            // 库存变更
            this.skuService.decreaseOccupiedStockAndSaleVolumeOnCancel(item.getSku().getId(), item.getQuantity());

            // 退分
            int refundCent = item.getCent() * item.getQuantity();
            RefundParam refundParam = new RefundParam();
            refundParam.setCent(refundCent);
            // 主订单Id
            refundParam.setEntityId(item.getOrder().getId());
            refundParam.setConsumeTypeCode(item.getOrderType().getCode());
            refundParam.setUserId(item.getUser().getId());

            // this.centsService.refundCent(refundParam);

            // 保存操作日志
            OrderItemOperationLog operLog = new OrderItemOperationLog();
            operLog.setOperator(new AdminUser(loggedUserId));
            operLog.setOperation(OrderItem.OrderAction.CANCEL.getName());
            operLog.setLogTime(new Date());
            operLog.setOrderItem(item);
            this.operLogDao.insert(operLog);
        }
    }

    /**
     * 发货
     */
    @Transactional(isolation = Isolation.READ_COMMITTED, propagation = Propagation.REQUIRED)
    public void sendOrder(OrderItem sendOrder, Long loggedUserId) {

        final long orderItemId = sendOrder.getId();
        if (sendOrder != null && (sendOrder.getStatus() == OrderItem.Status.DEALED)) {
            sendOrder.setStatus(OrderItem.Status.SENDING);
            update(sendOrder);

            OrderItem detail = this.entityDao.getById(orderItemId);

            long skuId = detail.getSku().getId();
            int quantity = detail.getQuantity();

            // 库存变更
            skuService.decreaseStockAndOccupiedOnSend(skuId, quantity);

            // 保存操作日志
            OrderItemOperationLog operLog = new OrderItemOperationLog();
            operLog.setOperator(new AdminUser(loggedUserId));
            operLog.setOperation(OrderItem.OrderAction.MANUAL_SEND.getName());
            operLog.setLogTime(new Date());
            operLog.setOrderItem(sendOrder);
            operLogDao.insert(operLog);
        }
    }

    /**
     * 重新发货
     */
    @Transactional(isolation = Isolation.READ_COMMITTED, propagation = Propagation.REQUIRED)
    public void repeatSendOrder(long orderItemId) {
        if (orderItemId > 0) {
            OrderItem orderItem = super.getById(orderItemId);
            List<CardDeliverRecord> cardDeliverRecords = cardDeliverRecordService.getRecordByOrderItemIdNoStatus(orderItemId);
            final Long userId = orderItem.getUser().getId();
            Sku sku = skuService.getById(orderItem.getSku().getId());
            // 发送短信
            new Thread(new SendSMS(cardDeliverRecords, sku.getName(), userId)).start();
        }
    }

    class SendSMS implements Runnable {

        private List<CardDeliverRecord> recordList;
        private String skuName;
        private Long userId;

        public SendSMS(List<CardDeliverRecord> records, String skuName, Long userId) {
            this.recordList = records;
            this.skuName = skuName;
            this.userId = userId;
        }

        @Override
        public void run() {
            try {
                if (this.recordList == null || this.recordList.size() == 0) {
                    logger.info("没有发货记录.");
                    return;
                }
                SimpleDateFormat sdf = new SimpleDateFormat(Constants.YYYYMMDD);
                User u = userService.getById(this.userId);
                String content = "您已成功兑换%s，请妥善保管，如有问题请联系客服4000108068。";
                StringBuffer smsInfo = new StringBuffer(100);
                smsInfo.append(this.skuName.trim()).append(":");
                for (int i = 0, j = this.recordList.size(); i < j; i++) {
                    CardDeliverRecord cdr = recordList.get(i);
                    long vpId = cdr.getVirtualProductId();
                    VirtualProduct virtualProduct = virtualProductService.getById(vpId);
                    if (virtualProduct.getType().equals(VirtualProduct.Type.CARD_PWD)) {
                        smsInfo.append("卡号").append(virtualProduct.getCardno()).append("，");
                        smsInfo.append("密码").append(virtualProduct.getPasswd()).append("，");
                        smsInfo.append("有效期至").append(sdf.format(virtualProduct.getExpired())).append("，");
                    } else if (virtualProduct.getType() == VirtualProduct.Type.EXCHANGE_CODE) {
                        smsInfo.append("兑换码").append(virtualProduct.getPasswd()).append("，");
                        smsInfo.append("有效期至").append(sdf.format(virtualProduct.getExpired())).append("，");
                    }

                    if (i != (j - 1)) {
                        smsInfo.deleteCharAt(smsInfo.length() - 1);
                        smsInfo.append(";");
                    }
                }

                boolean status = false;// smsService.sendSms(String.format(content,
                // smsInfo.toString()),
                // u.getMobilePhone());
                if (status) {
                    logger.info("短信发送成功");
                    for (CardDeliverRecord cdr : this.recordList) {
                        cdr.setDeliverStatus(CardDeliverRecord.DeliverStatus.SUCCESS);
                        cardDeliverRecordService.update(cdr);
                    }
                } else {
                    logger.info("短信发送失败");
                    for (CardDeliverRecord cdr : this.recordList) {
                        cdr.setDeliverStatus(CardDeliverRecord.DeliverStatus.FAILED);
                        cardDeliverRecordService.update(cdr);
                    }
                }

            } catch (Exception e) {
                logger.info("短信发送失败");
                for (CardDeliverRecord cdr : this.recordList) {
                    cdr.setDeliverStatus(CardDeliverRecord.DeliverStatus.FAILED);
                    cardDeliverRecordService.update(cdr);
                }
            }
        }

    }

    /**
     * @param isBatch 是否批量操作 <br/>
     *                支付成功的订单,更新为 “已处理”
     */
    public void dealWithOrderItems(boolean isBatch, AdminUser operator, Long... orderItemIds) {
        if (ArrayUtils.isNotEmpty(orderItemIds)) {
            super.entityDao.dealWithOrderItems(orderItemIds);

            // 更新完毕后，更新redis缓存
            this.batchClearCachedOrderItems(Arrays.asList(orderItemIds));

            // 保存操作日志
            OrderItemOperationLog operLog = new OrderItemOperationLog();
            operLog.setOperation(isBatch ? OrderItem.OrderAction.BATCH_DEAL.getName() : OrderItem.OrderAction.MANUAL_DEAL.getName());
            operLog.setLogTime(new Date());
            operLog.setOrderItem(new OrderItem(orderItemIds[0]));
            operLog.setOperator(operator);

            if (orderItemIds.length == 1) {
                operLogDao.insert(operLog);
            } else {
                List<OrderItemOperationLog> operLogList = Lists.newArrayListWithExpectedSize(orderItemIds.length);
                operLogList.add(operLog);

                for (int i = 1; i < orderItemIds.length; i++) {
                    OrderItemOperationLog orderOperLog = new OrderItemOperationLog();
                    try {
                        BeanUtils.copyProperties(orderOperLog, operLog);
                        orderOperLog.setOrderItem(new OrderItem(orderItemIds[i]));
                        operLogList.add(orderOperLog);
                    } catch (Exception e) {
                    }
                }
                operLogDao.batchInsert(operLogList);
            }
        }
    }

    private void batchClearSaleOrderItemCachesIdIn(List<Long> odrItemIdList) {
        String orderItemKeyPrefix = CacheKeys.SALE_ORDER_ITEM_PREFIX;
        orderItemKeyPrefix = StringUtils.substringBetween(orderItemKeyPrefix, "'");

        List<String> cacheKeyList = Lists.newArrayList();
        for (final Long id : odrItemIdList) {
            cacheKeyList.add(orderItemKeyPrefix + id);
        }
        jedisTemplate.del(cacheKeyList.toArray(new String[]{}));
    }

    /**
     * 订单确认收货|换货完成|退货完成 调用
     */
    @CacheEvict(value = CacheKeys.DEFAULT_NAME, key = CacheKeys.SALE_ORDER_ITEM_PREFIX + "#orderItemId")
    public void updateOrderState(long orderItemId, OrderItem.Status newStatus, OrderItem.AdmStatus admStatus, Long loggedUserId) {
        if (orderItemId > 0 && (newStatus == OrderItem.Status.FINISH || newStatus == OrderItem.Status.EXCHANGED || newStatus == OrderItem.Status.REFUNDED)) {

            OrderItem updateOrdItem = new OrderItem();
            updateOrdItem.setId(orderItemId);
            updateOrdItem.setStatus(newStatus);
            updateOrdItem.setAdmStatus(admStatus);
            update(updateOrdItem);

            OrderItemOperationLog operLog = new OrderItemOperationLog();
            operLog.setOperator(new AdminUser(loggedUserId));
            Date currDate = new Date();
            operLog.setLogTime(currDate);
            operLog.setOrderItem(new OrderItem(orderItemId));

            String operation = StringUtils.EMPTY;
            int status = 0;
            if (newStatus == OrderItem.Status.FINISH) {
                operation = OrderItem.OrderAction.RECEIVE.getName();
            } else if (newStatus == OrderItem.Status.EXCHANGED) {
                operation = OrderItem.OrderAction.COMPLETE_EXCHANGE.getName();
                // 换货 成功
                status = 2;
            } else if (newStatus == OrderItem.Status.REFUNDED) {
                operation = OrderItem.OrderAction.COMPLETE_REFUND.getName();
                // 退货 成功
                status = 4;
            }
            operLog.setOperation(operation);
            this.operLogDao.insert(operLog);
            if (status != 0) {
                ProdReturnRecord prodReturnRecord = new ProdReturnRecord();
                prodReturnRecord.setOrderItem(updateOrdItem);
                prodReturnRecord.setOrderFinishTime(currDate);
                prodReturnRecord.setStatus(status);
                this.prodReturnRecordDao.update(prodReturnRecord);
            }
        }
    }

    /**
     * 批量导入发货 根据orderCodes查询所有能够发货的SalesOrderItem列表
     *
     * @param orderCodes
     * @return
     */
    public List<OrderItem> findDealedOrderItemsByCodesIn(List<String> orderCodes) {
        if (CollectionUtils.isNotEmpty(orderCodes)) {
            return super.entityDao.findDealedOrderItemsByCodesIn(orderCodes);
        }
        return Collections.emptyList();
    }

    /**
     * 换货 如果点击了换货，选择了原商品可用，库存不发生变化，如果选择的是原商品不可用，则库存变化：良品库存减少，残次品库存增加<br/>
     * 点换货后， 订单状态变更为 换货中（EXCHANGING) <br/>
     * 并写订单操作日志
     */

    public void exchangeSalesOrderItems(OrderItemExchageRefundParamDto exchageParamDto) throws Exception {
        // 更新SalesOrderItem
        OrderItem orderItemForUpdate = getById(exchageParamDto.getOrderItemId());
        if (null == orderItemForUpdate) {
            throw new Exception("订单不存在");
        }
        orderItemForUpdate.setStatus(OrderItem.Status.EXCHANGING);
        update(orderItemForUpdate);

        // 库存变化
        if (!exchageParamDto.getCanUse()) {
            skuService.decrStockAndIncrDefectStockOnExchange(orderItemForUpdate.getSku().getId(), exchageParamDto.getQuantity());
        }

        // 插入操作日志
        OrderItemOperationLog operLog = new OrderItemOperationLog();
        // operLog.setOperator(new User(WebUtils.getLoggedUserId()));
        Date orderOpTime = new Date();
        operLog.setLogTime(orderOpTime);
        operLog.setOrderItem(orderItemForUpdate);
        operLog.setOperation(OrderItem.OrderAction.EXCHANGE.getName());

        String notePattern = "换货说明：{0} <br/>  商品是否可用：{1} <br/>  换货数量:{2}";
        String note = MessageFormat.format(notePattern, exchageParamDto.getNote(), exchageParamDto.getCanUse() ? "是" : "否", exchageParamDto.getQuantity());
        operLog.setNote(note);
        operLogDao.insert(operLog);

        // 插入换货记录
        ProdReturnRecord prodReturnRecord = new ProdReturnRecord();
        prodReturnRecord.setCount(exchageParamDto.getQuantity());
        prodReturnRecord.setNote(exchageParamDto.getNote());
        // ShiroUser user = WebUtils.getLoggedUser();
        // prodReturnRecord.setOperatorId(user.getId());
        // prodReturnRecord.setOperatorName(user.getName());
        prodReturnRecord.setOrderCreateTime(orderItemForUpdate.getCreateTime());
        prodReturnRecord.setOrderItem(orderItemForUpdate);
        prodReturnRecord.setOrderOpTime(orderOpTime);
        prodReturnRecord.setStatus(1);
        prodReturnRecord.setType(1);
        prodReturnRecordDao.insert(prodReturnRecord);
    }

    /**
     * 退货 如果点击了退货，选择了原商品可用，增加良品库存 ，如果选择的是原商品不可用，增加残次品库存<br/>
     * 点退货后， 订单状态变更为 REFUNDING("退货中")<br/>
     * 并写订单操作日志
     */
    public void doRefundSalesOrderItems(final OrderItemExchageRefundParamDto refundParamDto) throws Exception {
        // 更新SalesOrderItem
        OrderItem orderItemForUpdate = getById(refundParamDto.getOrderItemId());
        if (null == orderItemForUpdate) {
            throw new IllegalArgumentException("订单【orderItemId:" + refundParamDto.getOrderItemId() + "】不存在");
        }

        orderItemForUpdate.setStatus(OrderItem.Status.REFUNDING);
        update(orderItemForUpdate);

        // 更新库存 原商品可用
        if (refundParamDto.getCanUse()) {
            skuService.increaseStockOnRefundIfCanUse(orderItemForUpdate.getSku().getId(), refundParamDto.getQuantity());
        } else {
            // 原商品不可用
            // skuService.increaseDefectStockOnRefundIfCanotUse(orderItemForUpdate.getSku().getId(),
            // refundParamDto.getQuantity());
        }

        // 退货时，退还积分
        int refundCent = orderItemForUpdate.getCent() * refundParamDto.getQuantity();
        RefundParam refundParam = new RefundParam();
        refundParam.setCent(refundCent);
        refundParam.setEntityId(orderItemForUpdate.getOrder().getId());
        refundParam.setConsumeTypeCode(orderItemForUpdate.getOrderType().getCode());
        refundParam.setUserId(orderItemForUpdate.getUser().getId());
        // centService.refundCent(refundParam);

        // 插入操作日志
        OrderItemOperationLog operLog = new OrderItemOperationLog();
        // operLog.setOperator(new User(WebUtils.getLoggedUserId()));
        Date orderOpTime = new Date();
        operLog.setLogTime(orderOpTime);
        operLog.setOrderItem(orderItemForUpdate);
        operLog.setOperation(OrderItem.OrderAction.REFUND.getName());

        String notePattern = "退货说明：{0} <br/>  商品是否可用：{1} <br/>  退货数量:{2}";
        String note = MessageFormat.format(notePattern, refundParamDto.getNote(), refundParamDto.getCanUse() ? "是" : "否", refundParamDto.getQuantity());
        operLog.setNote(note);
        operLogDao.insert(operLog);

        // 插入换货记录
        ProdReturnRecord prodReturnRecord = new ProdReturnRecord();
        prodReturnRecord.setCount(refundParamDto.getQuantity());
        prodReturnRecord.setNote(refundParamDto.getNote());
        // ShiroUser user = WebUtils.getLoggedUser();
        // prodReturnRecord.setOperatorId(user.getId());
        // prodReturnRecord.setOperatorName(user.getName());
        prodReturnRecord.setOrderCreateTime(orderItemForUpdate.getCreateTime());
        prodReturnRecord.setOrderOpTime(orderOpTime);
        prodReturnRecord.setOrderItem(orderItemForUpdate);
        prodReturnRecord.setStatus(3);
        prodReturnRecord.setType(2);
        prodReturnRecordDao.insert(prodReturnRecord);
    }

    public void buildDetail(OrderItem orderItem) {
        if (orderItem != null) {
            Order order = this.orderService.getById(orderItem.getOrder().getId());
            order.setUser(this.userService.getById(order.getUser().getId()));
            Long addressId = order.getAddressId();
            if (addressId != null) {
	            Address addr = this.addressService.getById(addressId);
	            //this.addressService.buildProvCityCounty(Lists.newArrayList(addr));
	            order.setOrderAddress(addr);
            }
             
            Sku sku = this.skuService.getById(orderItem.getSku().getId());
            // TODO 目前供应商和用户没有关联 后期修改一下下面两行代码
            Product product = this.productService.getById(sku == null ? -1 : sku.getProduct().getId());
            product.setSupplier(supplierService.getById(product == null ? -1 : product.getSupplier().getId()));
            sku.setProduct(product);
            orderItem.setSku(sku);
            orderItem.setOrder(order);
            DictExpress express = orderItem.getExpress();
            if (express != null && express.getId() != null) {
                orderItem.setExpress(this.expressService.getById(express.getId()));
            }
        }
    }

    /**
     * 批量更新发货状态
     *
     * @param canSendOrderDtos
     * @param loginedUserId
     */
    public void updateSalesOrderItemsForBitchSend(final List<OrderBitchSendImportExcelDto> canSendOrderDtos, AdminUser loginedUser) {

        for (OrderBitchSendImportExcelDto sendDto : canSendOrderDtos) {
            OrderItem updateItem = new OrderItem();
            updateItem.setStatus(OrderItem.Status.SENDING);
            updateItem.setExpress(new DictExpress(sendDto.getExpressId()));
            updateItem.setTransportNo(sendDto.getTransportNo());
            updateItem.setId(sendDto.getOrderItem().getId());
            super.update(updateItem);
            this.skuService.decreaseStockAndOccupiedOnSend(sendDto.getOrderItem().getSku().getId(), sendDto.getOrderItem().getQuantity());
        }

        // 清除缓存
        this.skuService.batchClearCachedSkus(Collections3.extractToList(canSendOrderDtos, "orderItem.sku.id"));
        this.batchClearCachedOrderItems(Collections3.extractToList(canSendOrderDtos, "orderItem.id"));

        // 插入操作日志
        List<OrderItemOperationLog> operationLogs = Lists.newArrayList();

        Date logDateTime = new Date();
        for (final OrderBitchSendImportExcelDto dto : canSendOrderDtos) {
            OrderItemOperationLog operLog = new OrderItemOperationLog();
            operLog.setOperator(loginedUser);
            operLog.setLogTime(logDateTime);
            operLog.setOrderItem(dto.getOrderItem());
            operLog.setOperation(OrderItem.OrderAction.BATCH_SEND.getName());
            operLog.setNote("批量发货");
            operationLogs.add(operLog);
        }
        operLogDao.batchInsert(operationLogs);
    }

    private void batchClearCachedOrderItems(List<Long> odrItemIdList) {
        String itemCacheKeyPrev = CacheKeys.SALE_ORDER_ITEM_PREFIX;
        itemCacheKeyPrev = StringUtils.substringBetween(itemCacheKeyPrev, "'");
        List<String> itemCacheKeyList = Lists.newArrayListWithExpectedSize(odrItemIdList.size());
        for (Long itemId : odrItemIdList) {
            itemCacheKeyList.add(itemCacheKeyPrev + itemId);
        }
        jedisTemplate.del(itemCacheKeyList.toArray(new String[]{}));
    }

    /**
     * 查询订单操作记录
     *
     * @param orderItemId 订单id
     */
    public List<OrderItemOperationLog> findOperationLogByOrderItemId(Long orderItemId) {
        return operLogDao.findByOrderItemId(orderItemId);
    }

    public StatusDto importToSend(List<OrderBitchSendImportExcelDto> sendOrderList, AdminUser operator) {

        // 抽出批量查询条件
        List<String> orderCodes = Lists.newArrayList();
        List<String> expressCodes = Lists.newArrayList();
        if (CollectionUtils.isNotEmpty(sendOrderList)) {
            orderCodes = Collections3.extractToList(sendOrderList, "orderCode");
            expressCodes = Collections3.extractToList(sendOrderList, "expressCode");
        } else {
            return StatusDto.buildFailureStatusDto("导入失败! 请填写数据");
        }

        // 查询订单Item列表
        List<OrderItem> orderItems = findDealedOrderItemsByCodesIn(orderCodes);
        if (CollectionUtils.isEmpty(orderItems)) {
            return StatusDto.buildFailureStatusDto("导入失败! 没有可以发货的的订单");
        }
        // 查询快递公司列表
        Map<String, DictExpress> expressMap = expressService.findExpressMapByCodesIn(expressCodes);
        if (expressMap.isEmpty()) {
            return StatusDto.buildFailureStatusDto("导入失败! 没有对应的快递公司记录");
        }

        Map<String, OrderItem> dealedItemMap = Maps.newHashMap();
        for (OrderItem ordItem : orderItems) {
            dealedItemMap.put(ordItem.getOrderCode(), ordItem);
        }

        // 区分可成功导入和不能导入数据
        List<OrderBitchSendImportExcelDto> canUpdateDtos = Lists.newArrayList();
        List<OrderBitchSendImportExcelFailDto> canNotUpdateDtos = Lists.newArrayList();
        for (OrderBitchSendImportExcelDto dto : sendOrderList) {
            int correctFieldCount = 0;
            OrderBitchSendImportExcelFailDto failDto = null;
            StringBuffer errorMsgBuf = new StringBuffer();

            if (StringUtils.isBlank(dto.getOrderCode()) || !dealedItemMap.containsKey(dto.getOrderCode())) {
                errorMsgBuf.append("订单不存在;");
            } else {
                correctFieldCount++;
                OrderItem item = dealedItemMap.get(dto.getOrderCode());
                dto.setOrderItem(item);

                // 非虚拟地址
                // if ((!item.getSku().getProduct().getVisual()) &&
                // (item.getOrder().getAddress() == null ||
                // item.getOrder().getAddress().getId() == null)) {
                // errorMsgBuf.append("订单没有配送地址;");
                // } else {
                // correctFieldCount++;
                // }
                correctFieldCount++;
            }

            if (expressMap.containsKey(dto.getExpressCode()) && expressMap.get(dto.getExpressCode()).getStatus() == Status.OPEN) {
                dto.setExpressId(expressMap.get(dto.getExpressCode()).getId());
                correctFieldCount++;
            } else {
                errorMsgBuf.append("快递公司不存在或已停用;");
            }

            if (StringUtils.isNotEmpty(dto.getTransportNo())) {
                correctFieldCount++;
            } else {
                errorMsgBuf.append("运单号不能为空；");
            }

            if (correctFieldCount == 4) {
                canUpdateDtos.add(dto);
            } else {
                failDto = new OrderBitchSendImportExcelFailDto();
                failDto.setExpressCode(dto.getExpressCode());
                failDto.setOrderCode(dto.getOrderCode());
                failDto.setTransportNo(dto.getTransportNo());
                failDto.setFailMsg(errorMsgBuf.toString());
                canNotUpdateDtos.add(failDto);
            }
        }

        // 执行更新操作
        if (CollectionUtils.isNotEmpty(canUpdateDtos)) {
            updateSalesOrderItemsForBitchSend(canUpdateDtos, operator);
        }

        // 有导入失败记录，导出excel
        if (Collections3.isNotEmpty(canNotUpdateDtos)) {
            StatusDto res = StatusDto.buildFailureStatusDto();
            Map<String, Object> resMap = Maps.newHashMapWithExpectedSize(2);
            resMap.put("canUpdateDtos", canUpdateDtos);
            resMap.put("canNotUpdateDtos", canNotUpdateDtos);
            res.setData(resMap);
            return res;
        }

        return StatusDto.buildSuccessStatusDto("导入完成");
    }

    @Override
    public Map<String, Long> statisOrderNumGroupByStatus(Long userId) {
        Map<String, Long> statusQtyMap = null;
        if (userId > 0) {
            List<Map<String, Object>> orderNumRowList = this.entityDao.statisOrderNumGroupByStatus(userId);
            if (!CollectionUtils.isEmpty(orderNumRowList)) {
                Long totalCnt = 0L;
                statusQtyMap = Maps.newHashMapWithExpectedSize(orderNumRowList.size());
                for (Map<String, Object> rowMapper : orderNumRowList) {
                    String status = rowMapper.get("status").toString();
                    statusQtyMap.put(status, (Long) rowMapper.get("orderNum"));
                    totalCnt += (Long) rowMapper.get("orderNum");
                }
                statusQtyMap.put("total", totalCnt);
            }
        }

        if (statusQtyMap == null) {
            statusQtyMap = Collections.emptyMap();
        }
        return statusQtyMap;
    }

    @Override
    public void insertItems(List<OrderItem> items) {
        for (OrderItem item : items) {
            this.entityDao.insert(item);
        }
    }

    @Override
    @Transactional(isolation = Isolation.READ_COMMITTED, propagation = Propagation.REQUIRED)
    public void batchUpdateStatus(OrderItem.Status outStatus, AdmStatus inStatus, List<Long> ids) {
        if (ids == null || ids.size() == 0)
            throw new ServiceException("子订单id为空！");
        entityDao.batchUpdateStatus(outStatus, inStatus, ids);
    }

    /**
     * 查询订单下的第一条子订单
     *
     * @param orderId 订单id
     * @return
     */
    @Override
    public OrderItem getFirstByOrderId(Long orderId) {
        if (orderId == null)
            return null;
        List<OrderItem> list = this.findByOrderId(orderId);
        if (CollectionUtils.isNotEmpty(list)) {
            return list.get(0);
        }
        return null;
    }
}