package com.rocoinfo.rocomall.service.impl.order;

import com.google.common.collect.Lists;
import com.rocoinfo.rocomall.Constants;
import com.rocoinfo.rocomall.common.service.CrudService;
import com.rocoinfo.rocomall.dto.DeductParam;
import com.rocoinfo.rocomall.dto.StatusDto;
import com.rocoinfo.rocomall.entity.Address;
import com.rocoinfo.rocomall.entity.IdEntity;
import com.rocoinfo.rocomall.entity.Sku;
import com.rocoinfo.rocomall.entity.dict.DictPlatform;
import com.rocoinfo.rocomall.entity.order.Order;
import com.rocoinfo.rocomall.entity.order.OrderItem;
import com.rocoinfo.rocomall.repository.order.OrderDao;
import com.rocoinfo.rocomall.service.ICartService;
import com.rocoinfo.rocomall.service.IUserService;
import com.rocoinfo.rocomall.service.cent.ICentsService;
import com.rocoinfo.rocomall.service.impl.AddressService;
import com.rocoinfo.rocomall.service.impl.product.ProductService;
import com.rocoinfo.rocomall.service.order.IOrderItemService;
import com.rocoinfo.rocomall.service.order.IOrderService;
import com.rocoinfo.rocomall.service.product.ISkuMetaService;
import com.rocoinfo.rocomall.service.product.ISkuService;
import com.rocoinfo.rocomall.service.product.ISupplierService;
import com.rocoinfo.rocomall.utils.CodeGenerator;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springside.modules.utils.Collections3;

import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * <dl>
 * <dd>描述: 主订单service</dd>
 * <dd>公司: 大城若谷信息技术有限公司</dd>
 * <dd>创建时间：15/7/31 下午1:36</dd>
 * <dd>创建人： weiys</dd>
 * </dl>
 * @author Kong
 */
@SuppressWarnings("all")
@Service
public class OrderService extends CrudService<OrderDao, Order> implements IOrderService {

	@Autowired
	private IOrderItemService orderItemService;
	@Autowired
	private AddressService addressService;
	@Autowired
	private ISkuService skuService;
	@Autowired
	private ProductService productService;
	@Autowired
	private ISupplierService supplierService;
	@Autowired
	private ISkuMetaService skuMetaService;
	@Autowired
	private OrderDeliveryService orderDeliveryService;
	@Autowired
	private ICartService cartService;
	@Autowired
	private ICentsService centsService;
	@Autowired
	private IUserService userService;

	@Autowired
	private CodeGenerator codeGenerator;
	/**
	 * 释放或占用订单的“占用库存”
	 * 
	 * @param operateFlag 操作标记 1：释放库存，2：占用库存
	 */
	public void releaseOrOccupyStockByOrderId(Long orderId, int operateFlag) {
		List<OrderItem> itemList = orderItemService.findByOrderId(orderId);
		if (!CollectionUtils.isEmpty(itemList)) {
			List<Long> skuIdList = Lists.newArrayListWithCapacity(itemList.size());
			List<Integer> qtyList = Lists.newArrayListWithCapacity(itemList.size());
			for (OrderItem item : itemList) {
				skuIdList.add(item.getSku().getId());
				if (operateFlag == 1) {
					qtyList.add(-item.getQuantity());
				} else if (operateFlag == 2) {
					qtyList.add(item.getQuantity());
				}
			}
			skuService.incrOrDecrOccupiedStock(skuIdList, qtyList);
		}
	}

	/**
	 * 保存购物车订单
	 */
	public void insert(Order order) {
		super.insert(order);

		String consumeTypeCode = OrderItem.OrderType.common.getCode();

		if (!CollectionUtils.isEmpty(order.getItems())) {
			consumeTypeCode = Collections3.getFirst(order.getItems()).getOrderType().getCode();

			for (OrderItem item : order.getItems()) {
				orderItemService.insert(item);
			}

			List<Long> skuIdList = Lists.newArrayList();
			List<Integer> qtyList = Lists.newArrayList();

			for (OrderItem item : order.getItems()) {
				skuIdList.add(item.getSku().getId());
				qtyList.add(item.getQuantity());
			}

			skuService.batchIncreaseOccupiedStockAndSaleVolumeOnPayed(skuIdList, qtyList);

			orderDeliveryService.insert(order.getDelivery());
			// cartService.removeCartItemUnitProdId(order.getUser().getId(),
			// skuIdList.toArray(new Long[]{}));
		}
		// 支付得调支付接口
		// this.payOrder(order, consumeTypeCode);
	}

	public Page<Order> searchOrder(Map<String, Object> params, Pageable pageable) {
		List<Order> orders = Collections.emptyList();
		Long count = entityDao.searchTotal(params);
		if (count > pageable.getOffset()) {
			params.put(Constants.PAGE_OFFSET, pageable.getOffset());
			params.put(Constants.PAGE_SIZE, pageable.getPageSize());
			params.put(Constants.PAGE_SORT, pageable.getSort());
			orders = entityDao.search(params);
			buildOrderItems(orders, true, params);
		}
		return new PageImpl<Order>(orders, pageable, count);
	}

	/**
	 * 构建订单详情
	 * @param orderList
	 * @param isLoadImg
	 * @param params 参数，包括status
	 */
	public void buildOrderItems(List<Order> orderList, boolean isLoadImg, Map<String, Object> params) {
		if (!CollectionUtils.isEmpty(orderList)) {
			List<OrderItem> itemList = orderItemService.findWithSkuByOrderIdsIn(Collections3.extractToList(orderList, IdEntity.ID_FIELD_NAME));

			final String status = "status";
			List<Sku> skuList = Lists.newArrayList();
			String itemStatus = null;
			if (params != null &&params.containsKey(status)) {
				itemStatus = params.get(status).toString();
			}
			
			for (Order order : orderList) {
				order.setItems(null);
				Iterator<OrderItem> iter = itemList.iterator();
				while (iter.hasNext()) {
					OrderItem orderItem = iter.next();

					if (orderItem.getOrder().getId().equals(order.getId())) {
						Sku itemSku = orderItem.getSku();
						// 规格获取
						itemSku.getProduct().setSkuMeta(skuMetaService.getByProductId(itemSku.getProduct().getId()));
						orderItem.setSku(itemSku);
						// 订单状态过滤
						if (StringUtils.isNotEmpty(itemStatus) && orderItem.getStatus().toString().equals(itemStatus)) {
							order.addItem(orderItem);
						} else if (StringUtils.isEmpty(itemStatus)) {
							order.addItem(orderItem);
						}

						skuList.add(orderItem.getSku());
						iter.remove();
					}
				}
			}

			if (isLoadImg) {
				this.skuService.loadImages(skuList);
			}

			Iterator<Order> iterator = orderList.iterator();
			while (iterator.hasNext()) {
				Order orderInfo = iterator.next();
				Address address = addressService.getById(orderInfo.getAddressId());
				orderInfo.setOrderAddress(address);
				if (Collections3.isEmpty(orderInfo.getItems())) {
					iterator.remove();
					orderList.remove(orderInfo);
				}
			}
		}

	}

	public Order getByOrderNo(final String orderNo) {
		Order order = null;
		if (StringUtils.isNotBlank(orderNo)) {
			order = entityDao.getByOrderNo(orderNo);
		}
		return order;
	}

	@Override
	public void saveCartOrder(Order order, Long loggedUserId) {
		this.entityDao.insert(order);

		String consumeTypeCode = OrderItem.OrderType.common.getCode();

		if (!CollectionUtils.isEmpty(order.getItems())) {
			consumeTypeCode = Collections3.getFirst(order.getItems()).getOrderType().getCode();

			this.orderItemService.insertItems(order.getItems());

			List<Long> skuIdList = Lists.newArrayList();
			List<Integer> qtyList = Lists.newArrayList();

			for (OrderItem item : order.getItems()) {
				skuIdList.add(item.getSku().getId());
				qtyList.add(item.getQuantity());
			}

			skuService.batchIncreaseOccupiedStockAndSaleVolumeOnPayed(skuIdList, qtyList);
			cartService.removeCartItemUnitProdId(order.getUser().getId(), skuIdList.toArray(new Long[]{}));
		}

		this.payOrder(order, consumeTypeCode, loggedUserId);
	}

	@Override
	public Page<Order> searchOrder(Map<String, Object> parameters, PageRequest pageable) {
		List<Order> orders = Collections.emptyList();
		Long count = this.entityDao.searchTotal(parameters);
		if (count > pageable.getOffset()) {
			parameters.put(Constants.PAGE_OFFSET, pageable.getOffset());
			parameters.put(Constants.PAGE_SIZE, pageable.getPageSize());
			parameters.put(Constants.PAGE_SORT, pageable.getSort());
			orders = this.entityDao.search(parameters);
			buildOrderItems(orders, true, parameters);
		}
		return new PageImpl<Order>(orders, pageable, count);
	}

	private StatusDto payOrder(Order order, String consumeTypeCode, Long loggedUserId) {
		DeductParam param = new DeductParam();
		param.setConsumeTypeCode(consumeTypeCode);
		param.setCent(order.getCent());
		param.setEntityId(order.getId());
		param.setUserId(order.getUser().getId());
		param.setPlatformCode(DictPlatform.CENT_PLATFORM.getCode());
		return this.centsService.deductCent(param, loggedUserId);
	}

}