package com.rocoinfo.rocomall.service.impl.product;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.ListMultimap;
import com.google.common.collect.Lists;
import com.rocoinfo.rocomall.Constants;
import com.rocoinfo.rocomall.common.service.CrudService;
import com.rocoinfo.rocomall.common.service.ServiceException;
import com.rocoinfo.rocomall.dto.SkuCentDiscountDto;
import com.rocoinfo.rocomall.entity.*;
import com.rocoinfo.rocomall.entity.ProductImage.ImageType;
import com.rocoinfo.rocomall.redis.CacheKeys;
import com.rocoinfo.rocomall.redis.JedisTemplate;
import com.rocoinfo.rocomall.repository.SkuDao;
import com.rocoinfo.rocomall.service.product.*;
import org.apache.commons.lang3.StringUtils;
import org.apache.ibatis.annotations.Param;
import org.springframework.aop.framework.AopContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.*;

/**
 * <dl>
 * <dd>描述:</dd>
 * <dd>公司: 大城若谷信息技术有限公司</dd>
 * <dd>@创建时间：2015-8-3 下午1:33:04</dd>
 * <dd>@author： 张文山</dd>
 * </dl>
 */
@SuppressWarnings("all")
@Service
public class SkuService extends CrudService<SkuDao, Sku> implements ISkuService {

	@Autowired
	private IDescriptionService descriptionService;
	@Autowired
	private IProductService productService;
	@Autowired
	private ISupplierService supplierService;
	@Autowired
	private IBrandService brandService;
	@Autowired
	private IProductImageService productImageService;
	@Autowired
	private IProductCatalogService productCatalogService;
	@Autowired
	private JedisTemplate jedisTemplate;
	@Autowired
	private ProductCatalogService prodCataService;
	@Autowired
	private IPriceHistoryService priceHistoryService;

	/**
	 * 查询出所有上架的sku,包括sku的Product信息，图片、所属分类(用来在列表页生成js文件)
	 */
	public List<Sku> findAllListedSkuWithProdCatalogAndImgs() {
		List<Sku> skuList = this.entityDao.findAllListedSkuWithProd();
		if (!CollectionUtils.isEmpty(skuList)) {
			this.loadImages(skuList);
			List<ProductCatalog> allProdCatalog = prodCataService.findAll();

			ListMultimap<Long, Catalog> prodMutiCataIdMap = ArrayListMultimap.create();
			for (ProductCatalog prodCata : allProdCatalog) {
				prodMutiCataIdMap.put(prodCata.getProductId(), new Catalog(prodCata.getCatalogId()));
			}

			for (Sku sku : skuList) {
				sku.getProduct().setCatalogs(prodMutiCataIdMap.get(sku.getProduct().getId()));
			}
		}
		return skuList;
	}

	public List<Sku> getByProductId(Long productId) {
		return entityDao.getByProductId(productId);
	}

	public List<Sku> findByIdIn(List<Long> ids) {
		List<Sku> skus = new ArrayList<Sku>(ids.size());
		for (Long id : ids) {
			skus.add(getById(id));
		}
		return skus;
	}

	/**
	 * 获得sku列表（包括sku的商品，和图片）
	 */
	public List<Sku> findSkuWithProdByIdIn(List<Long> skuIds, boolean isLoadSkuImg) {
		if (CollectionUtils.isEmpty(skuIds)) {
			return Collections.emptyList();
		}

		List<Sku> skuList = this.entityDao.findSkuWithProdByIdIn(skuIds);
		if (isLoadSkuImg) {
			this.loadImages(skuList);
		}
		return skuList;
	}

	public Sku buildDetail(Long skuId, boolean isLoadImg) {
		if (skuId == null)
			return null;
		Sku sku = this.entityDao.findSkuWithProdBySkuId(skuId);
		return sku;
	}

	public void buildDetail(Sku sku) {
		if (sku != null) {
			sku.setProductImages(productImageService.getBySkuId(sku.getId()));
			Product product = productService.getById(sku.getProduct().getId());
			if (product != null) {
				product.setSupplier(supplierService.getById(product.getSupplier().getId()));
				if (product.getBrand() != null) {
					product.setBrand(brandService.getById(product.getBrand().getId()));
				}
				product.setCatalogs(productCatalogService.findCatalogByProductId(product.getId()));
				product.setDescription(descriptionService.findByProductId(product.getId()));
				product.setProductImages(productImageService.getProductPrimaryImages(product.getId()));
				sku.setProduct(product);
			}
		}
	}

	public void loadImages(List<Sku> skuList) {

		if (!CollectionUtils.isEmpty(skuList)) {
			List<Long> prodIds = Lists.newArrayList();
			for (Sku sku : skuList) {
				if (!prodIds.contains(sku.getProduct().getId())) {
					prodIds.add(sku.getProduct().getId());
				}
			}

			List<ProductImage> prodImgList = this.productImageService.findByProductIdIn(prodIds);

			if (!CollectionUtils.isEmpty(prodImgList)) {
				for (Sku sku : skuList) {

					for (ProductImage prodImg : prodImgList) {
						if (prodImg.getProduct().getId().equals(sku.getProduct().getId())) {
							if (prodImg.getSku() == null) {
								sku.getProduct().addImage(prodImg);
							} else {
								if (prodImg.getSku().getId().equals(sku.getId())) {
									sku.addImage(prodImg);
								}
							}
						}
					}
				}
			}
		}
	}

	public void buildCartSkuDetail(Sku sku) {
		sku.setProductImages(productImageService.getBySkuId(sku.getId()));
		Product product = productService.getById(sku.getProduct().getId());
		if (product != null) {
			product.setSupplier(supplierService.getById(product.getSupplier().getId()));
			product.setProductImages(productImageService.getProductPrimaryImages(product.getId()));
			sku.setProduct(product);
		}
	}

	@Override
	public List<SkuCentDiscountDto> findSkuCentAndDicountByIdIn(List<Long> skuIds) {
		if (CollectionUtils.isEmpty(skuIds)) {
			return Collections.emptyList();
		}

		List<SkuCentDiscountDto> skuCentDiscDtoList = Lists.newArrayList();

		List<Map<String, Object>> mapList = this.entityDao.findSkuCentAndDicountByIdIn(skuIds);
		for (Map<String, Object> rowItem : mapList) {
			Integer id = (Integer) rowItem.get("id");
			Integer cent = ((BigDecimal) rowItem.get("cent")).intValue();

			SkuCentDiscountDto centDiscountDto = new SkuCentDiscountDto();
			centDiscountDto.setSkuId(id);
			centDiscountDto.setCent(cent);

			skuCentDiscDtoList.add(centDiscountDto);
		}
		return skuCentDiscDtoList;
	}

	public Long getSkuIdByAttributes(Long productId, String attribute1, String attribute2, String attribute3) {
		Map<String, Object> params = new HashMap<String, Object>();
		params.put("productId", productId);
		params.put("attribute1", attribute1);
		if (StringUtils.isNotBlank(attribute2)) {
			params.put("attribute2", attribute2);
		}
		if (StringUtils.isNotBlank(attribute3)) {
			params.put("attribute3", attribute3);
		}
		return entityDao.getSkuIdByAttributes(params);
	}

	@Override
	public void insert(Sku sku) {
		if (countByCode(sku.getCode()) > 0) {
			throw new ServiceException("SKU编码 " + sku.getCode() + " 已存在");
		}
		// insert 操作
		sku.setName(sku.getProductFullName());
		sku.setOrigCashAmt(0d);
		sku.setOrigCent(0);
		sku.setOrigMergeCashAmt(0d);
		sku.setOrigMergeCent(0d);
		super.insert(sku);

		if (sku.getProductImages() != null) {
			for (int i = 0; i < sku.getProductImages().size(); i++) {
				ProductImage productImage = sku.getProductImages().get(i);
				if (StringUtils.isBlank(productImage.getPath())) {
					continue;
				}
				productImage.setSku(sku);
				productImage.setProduct(sku.getProduct());
				productImage.setDisplayOrder(i);
				productImage.setImageType(ImageType.SKU);
				productImageService.insert(productImage);
			}
		}
	}

	public void update(Sku sku, Long userId) {
		if(sku.getId()!=null){
			Sku oldSku = super.getById(sku.getId());
			if (oldSku.getCashAmt() != sku.getCashAmt() || oldSku.getCent() != sku.getCent() || oldSku.getMergeCashAmt() != sku.getMergeCashAmt() || oldSku.getMergeCent() != sku.getMergeCent()) {
				// 价格发生了变化
				sku.setOrigCashAmt(oldSku.getCashAmt());
				sku.setOrigCent(oldSku.getCent());
				sku.setOrigMergeCashAmt(oldSku.getMergeCashAmt());
				sku.setOrigMergeCent(oldSku.getMergeCent());
				insertPriceHistory(oldSku, userId);
			}
		}
		
		sku.setAttribute1(StringUtils.trimToNull(sku.getAttribute1()));
		sku.setAttribute2(StringUtils.trimToNull(sku.getAttribute2()));
		sku.setAttribute3(StringUtils.trimToNull(sku.getAttribute3()));
		super.update(sku);

		// 更新sku图片
		if (sku.getProductImages() != null) {
			List<ProductImage> oldProductImages = productImageService.getBySkuId(sku.getId());
			productImageService.batchUpdate(oldProductImages, sku.getProductImages(), sku.getProduct(), sku);
		}
	}

	private void insertPriceHistory(Sku sku, Long userId) {
		PriceHistory price = new PriceHistory();
		price.setUserId(userId);
		price.setCashAmt(sku.getCashAmt());
		price.setCent(sku.getCent());
		price.setMergeCashAmt(sku.getMergeCashAmt());
		price.setMergeCent(sku.getMergeCent());
		price.setTime(new Date());
		price.setSkuId(sku.getId());
		priceHistoryService.insert(price);
	}

	/**
	 * 增加/减少 占用库存
	 * 
	 * @param skuIdList
	 * @param qtyList 数量>0 表示增加, 数量<0则表示减少
	 */
	public void incrOrDecrOccupiedStock(final List<Long> skuIdList, final List<Integer> qtyList) {
		if (!CollectionUtils.isEmpty(skuIdList) && !CollectionUtils.isEmpty(qtyList) && skuIdList.size() == qtyList.size()) {
			this.entityDao.incrOrDecrOccupiedStock(skuIdList, qtyList);
		}
	}

	/**
	 * 增加|减少 良品库存
	 * 
	 * @param dictWareHouseId 仓库Id,如果是出库时，则传null,入库传仓库id
	 */
	public void incrOrDecrStock(@Param("skuId") Long skuId, @Param("quantity") Integer quantity, @Param("wareHouseId") Long dictWareHouseId) {
		if (skuId != null && skuId > 0 && quantity != null && quantity != 0) {
			this.entityDao.incrOrDecrStock(skuId, quantity, dictWareHouseId);
		}
	}

	/**
	 * 下单后，增加占用库存 & 增加销量
	 */
	@Deprecated
	public void batchIncreaseOccupiedStockAndSaleVolumeOnPayed(final List<Long> skuIdList, final List<Integer> quantityList) {
		if (!CollectionUtils.isEmpty(skuIdList) && !CollectionUtils.isEmpty(quantityList) && skuIdList.size() == quantityList.size()) {
			entityDao.batchIncreaseOccupiedStockAndSaleVolumeOnPayed(skuIdList, quantityList);

			batchClearCachedSkus(skuIdList);
		}
	}

	/**
	 * 订单被取消后, 较少占用库存 & 较少销量
	 */
	@Deprecated
	public void decreaseOccupiedStockAndSaleVolumeOnCancel(final long skuId, final int quantity) {
		if (skuId > 0 && quantity > 0) {
			entityDao.decreaseOccupiedStockAndSaleVolumeOnCancel(skuId, quantity);
		}
	}

	/**
	 * 发货时, 较少占用库存 & 减少良品库存
	 */
	@Deprecated
	public void decreaseStockAndOccupiedOnSend(final long skuId, final int quantity) {
		if (skuId > 0 && quantity > 0) {
			entityDao.decreaseStockAndOccupiedOnSend(skuId, quantity);
		}
	}

	/**
	 * 换货时,如果选择选择原商品不可用， 良品库存减少 & 残次品库存增加
	 */
	@Deprecated
	public void decrStockAndIncrDefectStockOnExchange(final long skuId, final int quantity) {
		if (skuId > 0 && quantity > 0) {
			entityDao.decrStockAndIncrDefectStockOnExchange(skuId, quantity);
		}
	}

	/**
	 * 退货时,如果选择原商品可用, 则商品良品库存增加
	 */
	@Deprecated
	public void increaseStockOnRefundIfCanUse(final long skuId, final int quantity) {
		if (skuId > 0 && quantity > 0) {
			entityDao.increaseStockOnRefundIfCanUse(skuId, quantity);
		}
	}

	/**
	 * 批量导入发货时, 较少占用库存 & 减少良品库存
	 */
	@Deprecated
	public void batchDecreaseStockAndOccupiedOnImport(final List<Long> skuIdList, final List<Integer> quantityList) {
		if (!CollectionUtils.isEmpty(skuIdList) && !CollectionUtils.isEmpty(quantityList) && skuIdList.size() == quantityList.size()) {
			entityDao.batchDecreaseStockAndOccupiedOnImport(skuIdList, quantityList);
			batchClearCachedSkus(skuIdList);
		}
	}

	/**
	 * 批量清除缓存
	 */
	public void batchClearCachedSkus(List<Long> skuIdList) {
		String skuCacheKeyPrev = CacheKeys.SKU_KEY_PREV;
		skuCacheKeyPrev = StringUtils.substringBetween(skuCacheKeyPrev, "'");
		List<String> skuCacheKeyList = Lists.newArrayListWithExpectedSize(skuIdList.size());
		for (Long skuId : skuIdList) {
			skuCacheKeyList.add(skuCacheKeyPrev + skuId);
		}
		jedisTemplate.del(skuCacheKeyList.toArray(new String[]{}));
	}

	public int countByCode(String code) {
		return entityDao.countByCode(code);
	}

	/**
	 * 当活动下架时，将SKU中活动信息字段设为null
	 */
	public void setListedActivityColumnsNull(Long id) {
		Sku sku = super.getById(id);
		entityDao.setListedActivityColumnsNull(id);
		// 写价格更新日志
		// insertPriceHistory(sku);
	}

	public Page<Sku> adminSearchJoin(Map<String, Object> parameters, Pageable pageable) {
		Long total = entityDao.adminSearchTotalJoin(parameters);
		List<Sku> content;
		if (total > pageable.getOffset()) {
			parameters.put(Constants.PAGE_OFFSET, pageable.getOffset());
			parameters.put(Constants.PAGE_SIZE, pageable.getPageSize());
			parameters.put(Constants.PAGE_SORT, pageable.getSort());
			content = entityDao.adminSearchJoin(parameters);
		} else {
			content = Collections.emptyList();
		}
		return new PageImpl<Sku>(content, pageable, total);
	}

	private SkuService getProxy() {
		return (SkuService) AopContext.currentProxy();
	}

	public Sku getByCode(String code) {
		return entityDao.getByCode(code);
	}

	// 是否虚拟商品
	public boolean isVirtualProduct(Long skuId) {
		Sku sku = super.getById(skuId);
		return (sku != null && productService.getById(sku.getProduct().getId()).getVisual());
	}
}