package com.rocogz.syy.operation.entity.quotapply;

import com.baomidou.mybatisplus.annotation.TableField;
import com.rocogz.syy.common.annotation.UniqueField;
import com.rocogz.syy.common.entity.UserTimeEntity;
import com.rocogz.syy.operation.constants.OperationConstant;
import com.rocogz.syy.operation.enums.*;
import lombok.Getter;
import lombok.Setter;
import lombok.experimental.Accessors;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.List;

/**
 * <p>
 * 客户权益管理-> 额度申请表
 * </p>
 *
 * @author zhangmin
 * @since 2021-06-23
 */
@Setter
@Getter
@Accessors(chain = true)
public class OperateQuotaApply extends UserTimeEntity {

    /**
     * 申请单编码
     */
    @UniqueField
    private String code;

    /**
     * 申请单名称
     */
    private String name;

    /**
     * 申请方式:整体或分团队或 分主体
     * 目前 applyWay都是 = ALL
     * @see OperationConstant.DictQuotaApplyWay
     */
    private String applyWay;

    /**
     * 额度费用类型
     * @see OperationConstant.DictQuotaApplyFeeType#UNIVERSAL_USE
     */
    private String feeType;


    /***
     * 申请端入口
     * @see OperationConstant.DictQuotaApplyPortal#B_WEAPP
     */
    private String portal;

    /**
     * 申请模式:运营代申请/主体自己申请
     *
     * @see OperationConstant.DictQuotaApplyMode#TYPE_CODE
     */
    private QuotaApplyModeEnum applyMode;

    /**
     * 额度管理模式
     * @see 备付金电子账户的 字典类型是： ELECTRONIC_ACCT_QUOTA_MODE
     */
    private QuotaManageModeEnum quotaManageMode;

    /**
     * 备付金模式下：充值类型
     * @see 备付金电子账户的 字典类型是： ELECTRONIC_ACCT_PROVISION_TYPE
     */
    private CoverDepositTypeEnum depositType;

    /**
     * 申请单状态
     *
     * @see OperationConstant.DictQuotaApplyStatus
     */
    private String status;

    /**
     * 申请人用户id: reference basic_system_admin_user.id
     */
    private Integer applierUserId;

    /**
     * 申请人用户名: reference basic_system_admin_user.username
     */
    private String applierUser;

    /**
     * 申请人手机号:reference basic_system_admin_user.mobile
     */
    private String applierMobile;

    /**
     * 申请人名字: reference basic_system_admin_user.name
     */
    private String applierName;

    /**
     * 上级审批节点编号,即最后审批/审核 节点编号
     */
    private String lastApproveNodeCode;

    /**
     * 上级审批节点名称,即最后审批/审核 节点名称
     */
    private String lastApproveNodeName;

    /**
     * 上级审批节点类型,即最后审批/审核节点类型
     */
    private QuotaApproveNodeTypeEnum lastApproveNodeType;

    /**
     * 上级审批结果
     *@see OperationConstant.DictApproveResult
     */
    private String lastApproveResult;

    /**
     * 最后审批时间
     */
    private LocalDateTime lastApproveTime;


    /**
     * 当前审批节点编号(还未审)
      */
    private String curNodeCode;

    /**
     * 当前审批节点名称 (还未审)
     */
    private String curNodeName;

    /**
     * 当前审批节点类型,即审批业务类型
     * @see QuotaApproveNodeTypeEnum
     */
    private String curNodeType;

    /**
     * 申请人的 发放主体编号: reference equity_issuing_body.code
     */
    private String issuingBodyCode;

    /**
     * 申请人发放主体的祖先主体编号
     */
    private String topIssuingBodyCode;

    /**
     * 申请额度
     */
    private BigDecimal quota;

    //额度转换备付金 兑换比例
    private BigDecimal quotaConversionRatio;

    /**
     * 备付金：金额
     * （1）如果是额度管理模式  = 额度 * 额度转换备付金的 兑换比例
     * （2）如果是备付金模式  coverAmount = 充值金额 或授信借款金额
     */
    private BigDecimal coverAmount;

    /**
     * 收款到账日期
     */
    private LocalDate receiptDate;

    /**
     * 代理商编号：如果发放主体是代理商
     * 如果发放主体的quity_issuing_body.whether_customer='AGENT'
     */
    private String agentCode;

    /**
     * 合作客户编号：如果发放主体的 equity_issuing_body.whether_customer='CUSTOMER'
     */
    private String customerCode;

    /**
     * 团队/发放主体 数量【团队/发放主体 申请方式专属字段】
     */
    private Integer teamNum;

    /**
     * 最近的操作: 如 审核、审批、处理
     * @see QuotaApplyActionTypeEnum
     */
    private String recentAction;

    /**
     * 最近操作时间：审核时间,完成时间
     */
    private LocalDateTime recentActionTime;

    /**
     * 最近操作人用户名
     */
    private String recentActionUser;

    /**
     * 划拨成功操作人用户名
     */
    private String allocateUser;

    /**
     * 划拨操作人名字
     */
    private String allocateUserFullName;

    /**
     * 划拨操作时间
     */
    private LocalDateTime allocateTime;

    /**
     * 额度账号 的交易号logNo
     * @see com.rocogz.syy.settlement.dto.TradeResp#logNo
     */
    private String quotaAcctLogNo;

    /**
     * 划拨操作,备付金帐户充值交易业务单据号
     */
    private String coverAcctTradeNo;

    /**
     * 申请备注
     */
    private String remark;


    //是否被逻辑删除
    private Boolean deleted;


    /**
     * 是否允许修改
     *
     * @return
     */
    public boolean isModifyAllowed() {
        return OperationConstant.DictQuotaApplyStatus.AUDIT_FAILED.equals(status) ||
                OperationConstant.DictQuotaApplyStatus.APPROVE_FAILED.equals(status) ||
                OperationConstant.DictQuotaApplyStatus.DRAFT.equals(status);
    }


    /**
     * 申请是否可以审批
     */
    public boolean isApproveble() {
        return OperationConstant.DictQuotaApplyStatus.PENDING_AUDIT.equals(status) ||
                OperationConstant.DictQuotaApplyStatus.PENDING_APPROVE.equals(status);
    }


    /**
     * 需要充值的备付金:  coverQuota = quota *（合作客户或代理商）配置的额度转换系数
     */
    public BigDecimal calcCoverQuota() {
        BigDecimal converQuota = quotaConversionRatio.multiply(quota.setScale(3, BigDecimal.ROUND_HALF_UP));
        return converQuota;
    }

    /**
     * 是否是最顶级发放主体
     */
    public boolean isTopLevelIssuingBody() {
        return this.issuingBodyCode.equals(topIssuingBodyCode);
    }

    //包含的团队/发放主体Item
    @TableField(exist = false)
    private List<OperateQuotaApplyItem> teamItemList;

    /**
     * 我最近的审批或审核记录 （小程序端 我已审批申请单列表 使用此字段）
     */
    @TableField(exist = false)
    private OperateQuotaApplyApproveRecord myRecentApproveRecord;


    //申请详情中要展示的 进度时间线
    @TableField(exist = false)
    private List<OperateQuotaApplyProgress> timelineList;


    //附件列表
    @TableField(exist = false)
    private List<OperateQuotaApplyAttach> attachList;


    //===== 以下是 非持久化属性 =====


    /**
     * 审批进度：待审核/待审批的 能 审批/审核的 用户名列表
     */
    private transient List<String> curApproverUserList;


    /**
     * 充值类型：名字
     */
    private transient String depositTypeLabel;

    //发放主体名称,页面显示使用
    private transient String issuingBodyName;

    //合作客户名称,页面显示
    private transient String customerName;


    //是否能被 当前登录用户执行划拨操作 【申请单列表,判定登录用户是否有"划拨额度"操作权限】
    private transient Boolean hasAppropriatePerm;

    /**
     * 是否能撤回：申请列表页面 控制是否能看见 撤回按钮
     */
    private transient Boolean canRecall;


    /**
     *登录账号是否 是该申请单的拥有者,即是申请创建者
     */
    private transient Boolean owner;

    /**
     * 当前审批节点 审批人账号类型
     */
    private transient QuotaApplyNodeApproverTypeEnum approverType;

}
