package com.rocogz.syy.infrastructure.util;

import com.rocogz.syy.common.response.Response;
import com.rocogz.syy.infrastructure.dto.trace.TraceDto;
import org.springframework.beans.BeanUtils;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.concurrent.TimeUnit;

/**
 * @author zhangmin
 * @date 2020/5/26
 */
public final class TraceUtils {
    static final String RESPONSE_NULL = "返回Response对象为NULL";

    private TraceUtils() {
    }

    /**
     * 获取错误的概要描述信息
     * 如果没有错误，则返回null
     */
    public static String getErrorMessage(Response resp, Throwable ex) {
        if (ex != null) {
            return ex.getMessage();
        } else {
            return getRespErrorMsg(resp);
        }
    }

    /**
     */
    public static TraceDto getTraceDto(Response resp, Throwable ex) {
        TraceDto traceDto = new TraceDto();
        //没有错误信息
        if(resp == null) {
            traceDto.setRespMsg(RESPONSE_NULL);
        }else  {
            traceDto.setRespCode(resp.getCode());
            traceDto.setRespMsg(resp.getMessage());
        }

        TraceDto errorTraceDto = getTraceDto(ex);
        BeanUtils.copyProperties(errorTraceDto,traceDto);
        return traceDto;
    }

    /**
     */
    public static TraceDto getTraceDto(LocalDateTime callStartTime, Response resp, Throwable ex) {
        TraceDto traceDto = getTraceDto(resp,ex);
        traceDto.setStartTimeAndCalcSpend(callStartTime);
        return traceDto;
    }

    public static TraceDto getTraceDto(Throwable ex) {
        if(ex == null) {
            return TraceDto.EMPTY_TRACE_DTO;
        }


        StackTraceElement targetStack = null;
        StackTraceElement[] traceElems= Thread.currentThread().getStackTrace();

        for(int i=1; i<traceElems.length; i++) {
            if(!TraceUtils.class.getName().equals(traceElems[i].getClassName())) {
                targetStack = traceElems[i];
                break;
            }
        }

        TraceDto traceDto = new TraceDto();
        if(targetStack!=null) {
            traceDto.setErrorLine(targetStack.getLineNumber());
            traceDto.setErrorMethod(targetStack.getMethodName());
            traceDto.setErrorClassName(targetStack.getClassName());
        }

        traceDto.setStackTrace(getStackTrace(ex));
        return traceDto;

    }

    public static TraceDto getTraceDto(LocalDateTime callStartTime, Throwable ex) {
        if(ex == null) {
            TraceDto traceDto = new TraceDto();
            traceDto.setStartTimeAndCalcSpend(callStartTime);
            return traceDto;
        }

        TraceDto traceDto = getTraceDto(ex);
        traceDto.setStartTimeAndCalcSpend(callStartTime);
        return traceDto;
    }


    /**
     * 获取错误详细原因, stack跟踪信息
     * 如果没有错误，则返回null
     */
    public static String getErrorTrace(Response resp, Throwable ex) {
        if (ex != null) {
            return getStackTrace(ex);
        } else {
            return getRespErrorMsg(resp);
        }
    }


    public static String getStackTrace(Throwable throwable) {
        StringWriter sw = new StringWriter();
        PrintWriter pw = new PrintWriter(sw, true);
        throwable.printStackTrace(pw);
        return sw.getBuffer().toString();
    }


    //如果成功，则返回的错误消息为NULL
    private static String getRespErrorMsg(Response resp) {
        if (resp == null) {
            return RESPONSE_NULL;
        }
        if (!resp.isSucceed()) {
            return resp.getMessage();
        }

        return null;
    }


    public static final String prettyMillis(long millisSeconds) {
        final int divided = 1000;
        if (millisSeconds < divided)  {
            return millisSeconds + "毫秒";
        }
        long seconds =  TimeUnit.MILLISECONDS.toSeconds(millisSeconds);
        long mod = millisSeconds % divided;
        if(mod!=0) {
             double result =  seconds +  (mod*1.0 / divided);
             //3.45秒
            return new BigDecimal(String.valueOf(result)).divide(BigDecimal.ONE, 2, BigDecimal.ROUND_HALF_UP) +"秒";
        }else {
            return seconds + "秒";
        }
    }

}
