package com.rocogz.syy.common.tencent;

import com.rocogz.syy.common.tencent.config.QQMapProperties;
import com.rocogz.syy.common.tencent.resp.*;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;
import org.springframework.web.client.RestTemplate;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 腾讯地图
 * @author zhangmin
 * @date 2020/4/8
 */
public class QQMapService {

    @Autowired
    private RestTemplate restTemplate;

    private QQMapProperties props;

    public QQMapService(QQMapProperties props) {
     this.props = props;
    }

    /**
     * 文档地址：https://lbs.qq.com/webservice_v1/guide-geocoder.html
     * 地址解析成 => 经纬度坐标
     * @param address 地址（注：地址中请包含城市名称，否则会影响解析效果）例如：address=北京市海淀区彩和坊路海淀西大街74号
     * @return 请求失败 返回 null
     */
    public QQLngLatResp getLngAndLatByAddress(String address) {
        if (StringUtils.isBlank(address)) {
            return null;
        }

        //https://apis.map.qq.com/ws/geocoder/v1/?address=%E5%8C%97%E4%BA%AC%E5%B8%82%E6%B5%B7%E6%B7%80%E5%8C%BA%E5%BD%A9%E5%92%8C%E5%9D%8A%E8%B7%AF%E6%B5%B7%E6%B7%80%E8%A5%BF%E5%A4%A7%E8%A1%9774%E5%8F%B7&key=OB4BZ-D4W3U-B7VVO-4PJWW-6TKDJ-WPB77
        String geoUrl ="https://apis.map.qq.com/ws/geocoder/v1/?address=%s&key=%s";
        String url = String.format(geoUrl, address,props.getKey());
        QQLngLatResp resp =  restTemplate.getForObject(url,QQLngLatResp.class);
        return resp;
    }


    /**
     * 将坐标点（经纬度）转换为对应位置信息
     * @param lngLat
     */
    public QQLocationResp getLocation(LngLat lngLat) {
        if(lngLat == null) {
            return null;
        }

        return this.getLocation(lngLat.toLatLngString());
    }


    /**
     * 文档：https://lbs.qq.com/webservice_v1/guide-gcoder.html
     * 将坐标点（经纬度）转换为对应位置信息（如所在行政区划，周边地标点分布）功能。
     * @param latLng 纬经值 = "lat<纬度>,lng<经度>"
     */
    public QQLocationResp getLocation(String latLng) {

        //location=lat<纬度>,lng<经度>
        //例如:https://apis.map.qq.com/ws/geocoder/v1/?location=39.984154,116.307490&key=OB4BZ-D4W3U-B7VVO-4PJWW-6TKDJ-WPB77&get_poi=1

        String queryLocationUrl = "https://apis.map.qq.com/ws/geocoder/v1/?location=%s&key=%s&get_poi=1";
        String url = String.format(queryLocationUrl, latLng,props.getKey());

        QQLocationResp location = restTemplate.getForObject(url,QQLocationResp.class);

        return location;
    }




    /**
     * 计算两点之间的距离
     * @param origin      实时定位的用户位置：经纬度
     * @param destination 目的地位置：经纬度
     */
   public QQUniqueDistanceResp getUniqueDistanceBetween(LngLat origin, LngLat destination) {

       if(destination == null) {
           return null;
       }

       List<LngLat> to = new ArrayList<>();
       to.add(destination);
       QQSingleOriginMultiDestDistanceResp tempResult = getMultiDistanceBetween(origin,to);
       if(tempResult == null) {
           return null;
       }

       QQUniqueDistanceResp resp = new QQUniqueDistanceResp();

       copyStatusMessage(tempResult,resp);

       if(tempResult.isSuccess()) {
           resp.setResult(tempResult.getResult().get(0));
       }

       return resp;
   }


    /**
     *  计算 （单出发地-> 多目的地） 距离
      * @param origin 出发地
     * @param destinationList 多个目的地
     * @return
     */
   public QQSingleOriginMultiDestDistanceResp getMultiDistanceBetween(LngLat origin,List<LngLat> destinationList) {
       if(origin == null) {
           return null;
       }

       List<LngLat> from = new ArrayList<>();
       from.add(origin);

       QQMultiOriginMultiDestDistanceResp tempResult = getMultiDistanceBetween(from,destinationList);
       if(tempResult == null) {
           return null;
       }

       QQSingleOriginMultiDestDistanceResp resp = new QQSingleOriginMultiDestDistanceResp();

       copyStatusMessage(tempResult,resp);

       if(tempResult.isSuccess()) {
           resp.setResult(tempResult.getResult().getOriginDistanceList().get(0).getDistanceList());
       }

       return resp;
   }



    /**
     * 文档：https://lbs.qq.com/webservice_v1/guide-distancematrix.html
     * @param originList 出发地
     * @param destinationList 目的地
     */
    public QQMultiOriginMultiDestDistanceResp getMultiDistanceBetween(List<LngLat> originList, List<LngLat> destinationList) {

        if(CollectionUtils.isEmpty(originList) || CollectionUtils.isEmpty(destinationList)) {
            return null;
        }


        //例如： http://apis.map.qq.com/ws/distance/v1/matrix/?mode=driving&from=39.984092,116.306934;40.007763,116.353798&to=39.981987,116.362896;39.949227,116.394310&key=OB4BZ-D4W3U-B7VVO-4PJWW-6TKDJ-WPB77
        final String calDistanceUrl = "http://apis.map.qq.com/ws/distance/v1/matrix/?mode=%s&from=%s&to=%s&key=%s";

        final String delimiter = ";";

        String from = originList.stream().map(LngLat::toLatLngString).collect(Collectors.joining(delimiter));

        String to= destinationList.stream().map(LngLat::toLatLngString).collect(Collectors.joining(delimiter));

        String url = String.format(calDistanceUrl,DistanceCalcModeEnum.driving.name(), from,to,props.getKey());

        QQMultiOriginMultiDestDistanceResp resp = restTemplate.getForObject(url,QQMultiOriginMultiDestDistanceResp.class);


        if(resp!=null && resp.isSuccess()) {

            final int rowCount = originList.size();
            final int columnCount = destinationList.size();

            for(int rowIdx =0; rowIdx<rowCount; rowIdx++) {
                QQMultiOriginMultiDestDistanceResp.OneOriginDistance distanceDto = resp.getResult().getOriginDistanceList().get(rowIdx);
                distanceDto.setOrigin(originList.get(rowIdx));

                for(int colIdx = 0; colIdx<columnCount; colIdx++) {
                    distanceDto.getDistanceList().get(colIdx).setDest(destinationList.get(colIdx));
                }
            }
        }
        return resp;
    }


    private void copyStatusMessage(QQMapBaseResp from, QQMapBaseResp target) {
        target.setStatus(from.getStatus());
        target.setMessage(from.getMessage());
    }

}
