package com.rocogz.syy.common.web;

import com.rocogz.syy.common.response.Response;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.propertyeditors.StringTrimmerEditor;
import org.springframework.stereotype.Component;
import org.springframework.validation.BindException;
import org.springframework.validation.BindingResult;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.ServletRequestDataBinder;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ResponseBody;

import javax.servlet.http.HttpServletRequest;
import javax.validation.ValidationException;
import java.util.HashMap;
import java.util.Map;

/**
 * 统一异常处理：包含如下功能：
 * （1）统一处理：controller 中Bean 加了 @Valid 校验错误信息收集
 *  (2)统一处理：提交的查询参数进行了trim()
 * @author zhangmin
 * @date 2020/4/12
 */
@Component(value="commonGobalExceptionHandler")
@ControllerAdvice
@ResponseBody
public class GlobalExceptionHandler {

    private Logger log = LoggerFactory.getLogger(GlobalExceptionHandler.class);

    /**
     * 统一处理 controller中方法校验异常 @Valid
     * @param ex
     * @return
     */
    @ExceptionHandler({MethodArgumentNotValidException.class,ValidationException.class,BindException.class})
    public Response handleValidateException(Exception ex) {
        Map<String,String> errorData = new HashMap<>();
        if(ex instanceof MethodArgumentNotValidException) {
            BindingResult bindingResult = ((MethodArgumentNotValidException)ex).getBindingResult();
            if(bindingResult.hasErrors()) {
                bindingResult.getFieldErrors().forEach(errorField -> {
                    errorData.put(errorField.getField(),errorField.getDefaultMessage());
                });
            }
        }else  {
            errorData.put("errorMsg",ex.getMessage());
        }

        Response failResp = Response.failure("提交的数据格式不正确");
        failResp.setData(errorData);
        return failResp;
    }

    /**
     * 参数不合法异常
     * @return
     */
    @ExceptionHandler(IllegalArgumentException.class)
    public Object illegalArgumentException(IllegalArgumentException ex) {
        log.error(ExceptionUtils.getStackTrace(ex));
        return Response.failure(ex.getMessage());
    }

    /**
     * 统一异常处理
     *
     * @param ex
     * @return
     */
    @ExceptionHandler({Exception.class})
    public Object exception(Exception ex) {
        log.error(ExceptionUtils.getStackTrace(ex));
        return Response.failure("操作失败！" +ex.getMessage());
    }

    /**
     * 注册自定义属性编辑器
     */
    @InitBinder
    protected void initBinder(HttpServletRequest request, ServletRequestDataBinder binder) throws Exception {
        binder.registerCustomEditor(String.class,new StringTrimmerEditor(true));
    }

}
