package com.rocogz.syy.common.web;

import com.fasterxml.jackson.datatype.jdk8.Jdk8Module;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.fasterxml.jackson.datatype.jsr310.deser.LocalDateTimeDeserializer;
import com.fasterxml.jackson.datatype.jsr310.ser.LocalDateSerializer;
import com.fasterxml.jackson.datatype.jsr310.ser.LocalDateTimeSerializer;
import com.fasterxml.jackson.module.paramnames.ParameterNamesModule;
import org.springframework.boot.autoconfigure.jackson.Jackson2ObjectMapperBuilderCustomizer;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import java.time.format.DateTimeFormatter;
import java.util.TimeZone;

/**
  @author zhangmin
  @date 2020/4/12
 */
@Configuration
public class JacksonCustomizer {

    private static final String dateFormat = "yyyy-MM-dd";
    private static final String dateTimeFormat = "yyyy-MM-dd HH:mm:ss";

    /**
     * 定制化json输出格式:
     * 解决如下问题：
     * （1）java8中日期与时间 json序列号与反序列化问题
     * （2）实体类中：日期输出 不用再写 @JsonFormat(pattern = YYYY_MM_DD_HH_MM_SS, timezone = TIME_ZONE)
     *  (3)json反序列化是，全局处理 "忽略json字符串中不识别的属性"
     */
      @Bean
      public Jackson2ObjectMapperBuilderCustomizer jsonCustomizer() {
          return builder -> {
              builder.simpleDateFormat(dateTimeFormat);

              DateTimeFormatter dateTimeFormatter = DateTimeFormatter.ofPattern(dateTimeFormat);
              builder.modulesToInstall(new Jdk8Module(),new JavaTimeModule(),new ParameterNamesModule());
              builder.serializers(new LocalDateSerializer(DateTimeFormatter.ofPattern(dateFormat)));
              builder.serializers(new LocalDateTimeSerializer(dateTimeFormatter));

              //java8的 LocalDateTime的 json反序列化的 默认时间格式是："2019-12-12T12:43:12" 即： LocalDateTimeDeserializer
              builder.deserializers(new LocalDateTimeDeserializer(dateTimeFormatter));

              // 忽略json字符串中不识别的属性
              builder.failOnUnknownProperties(false);
              // 忽略无法转换的对象 “No serializer found for class com.xxx.xxx”
              builder.failOnEmptyBeans(false);
              //指定时区
              builder.timeZone(TimeZone.getTimeZone("GMT+8:00"));

              // NULL不参与序列化
              // builder.serializationInclusion(JsonInclude.Include.NON_NULL);
          };
        }

}
