package com.rocogz.syy.common.tencent;

import com.rocogz.syy.common.tencent.config.QQMapProperties;
import com.rocogz.syy.common.tencent.resp.*;
import org.apache.commons.codec.digest.DigestUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Sort;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.transaction.jta.WebSphereUowTransactionManager;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;
import org.springframework.web.client.RestTemplate;
import sun.misc.Request;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

/**
 * 腾讯地图
 *
 * @author zhangmin
 * @date 2020/4/8
 */
public class QQMapService {

    static final String QQ_MAP_API_DOMAIN = "https://apis.map.qq.com";
    @Autowired
    private RestTemplate restTemplate;
    @Autowired
    private ThreadPoolTaskExecutor executorService;
    private QQMapProperties props;
    public QQMapService(QQMapProperties props) {
        this.props = props;
    }


    /**
     * 文档地址：https://lbs.qq.com/webservice_v1/guide-geocoder.html
     * 地址解析成 => 经纬度坐标
     *
     * @param address 地址（注：地址中请包含城市名称，否则会影响解析效果）例如：address=北京市海淀区彩和坊路海淀西大街74号
     * @return 请求失败 返回 null
     */
    public QQLngLatResp getLngAndLatByAddress(String address) {
        Assert.hasText(address, "address地址不能为空");

        //https://apis.map.qq.com/ws/geocoder/v1/?address=%E5%8C%97%E4%BA%AC%E5%B8%82%E6%B5%B7%E6%B7%80%E5%8C%BA%E5%BD%A9%E5%92%8C%E5%9D%8A%E8%B7%AF%E6%B5%B7%E6%B7%80%E8%A5%BF%E5%A4%A7%E8%A1%9774%E5%8F%B7&key=OB4BZ-D4W3U-B7VVO-4PJWW-6TKDJ-WPB77

        String apiPath = "/ws/geocoder/v1";
        Map<String, String> reqParamMap = new LinkedHashMap<>();
        reqParamMap.put("address", address);
        reqParamMap.put("key", props.getKey());
        String sign = this.calcSign(apiPath, reqParamMap);
        String geoUrl = QQ_MAP_API_DOMAIN + apiPath + "/?" + joinParams(reqParamMap,true) + "&sig=" + sign;
        QQLngLatResp resp = doGetForObject(geoUrl,QQLngLatResp.class);
        return resp;
    }


    /**
     * 将坐标点（经纬度）转换为对应位置信息
     *
     * @param lngLat
     */
    public QQLocationResp getLocation(LngLat lngLat) {
        Assert.notNull(lngLat, "经纬度不能为NULL");
        return this.getLocation(lngLat.toLatLngString());
    }


    /**
     * 文档：https://lbs.qq.com/webservice_v1/guide-gcoder.html
     * 将坐标点（经纬度）转换为对应位置信息（如所在行政区划，周边地标点分布）功能。
     *
     * @param latLng 纬经值 = "lat<纬度>,lng<经度>"
     */
    public QQLocationResp getLocation(String latLng) {
        //location=lat<纬度>,lng<经度>
        //例如:https://apis.map.qq.com/ws/geocoder/v1/?location=39.984154,116.307490&key=OB4BZ-D4W3U-B7VVO-4PJWW-6TKDJ-WPB77&get_poi=1
        Assert.hasText(latLng, "经纬度不能为空");
        String apiPath = "/ws/geocoder/v1";
        Map<String, String> reqParamMap = new LinkedHashMap<>();
        reqParamMap.put("location", latLng);
        reqParamMap.put("key", props.getKey());
        reqParamMap.put("get_poi", "1");

        String sign = this.calcSign(apiPath, reqParamMap);
        String queryLocationUrl = QQ_MAP_API_DOMAIN + apiPath + "/?" + joinParams(reqParamMap,true) + "&sig=" + sign;
        QQLocationResp location = doGetForObject(queryLocationUrl,QQLocationResp.class);
        return location;
    }


    /**
     * 计算两点之间的距离
     *
     * @param origin      实时定位的用户位置：经纬度
     * @param destination 目的地位置：经纬度
     */
    public QQUniqueDistanceResp getUniqueDistanceBetween(LngLat origin, LngLat destination) {

        Assert.notNull(destination, "目的地经纬度不能为NULL");

        List<LngLat> to = new ArrayList<>();
        to.add(destination);
        QQSingleOriginMultiDestDistanceResp tempResult = getMultiDistanceBetween(origin, to);
        if (tempResult == null) {
            return null;
        }

        QQUniqueDistanceResp resp = new QQUniqueDistanceResp();

        copyStatusMessage(tempResult, resp);

        if (tempResult.isSuccess()) {
            resp.setResult(tempResult.getResult().get(0));
        }

        return resp;
    }


    /**
     * 计算 （单出发地-> 多目的地） 距离
     *
     * @param origin          出发地
     * @param destinationList 多个目的地
     */
    public QQSingleOriginMultiDestDistanceResp getMultiDistanceBetween(LngLat origin, List<LngLat> destinationList) {
        Assert.notNull(origin, "源地点经纬度不能为NULL");

        List<LngLat> from = new ArrayList<>();
        from.add(origin);

        QQMultiOriginMultiDestDistanceResp tempResult = getMultiDistanceBetween(from, destinationList);
        if (tempResult == null) {
            return null;
        }

        QQSingleOriginMultiDestDistanceResp resp = new QQSingleOriginMultiDestDistanceResp();

        copyStatusMessage(tempResult, resp);

        if (tempResult.isSuccess()) {
            resp.setResult(tempResult.getResult().getOriginDistanceList().get(0).getDistanceList());
        }

        return resp;
    }


    /**
     * 计算 （单出发地-> 多目的地） 距离
     *
     * @param origin          出发地
     * @param destinationList 多个目的地
     * @param sortDirect      按照距离排序方式
     */
    public QQSingleOriginMultiDestDistanceResp getMultiDistanceBetweenOrderBy(LngLat origin, List<LngLat> destinationList, Sort.Direction sortDirect) {

        QQSingleOriginMultiDestDistanceResp resp = getMultiDistanceBetween(origin, destinationList);
        if (resp != null && resp.isSuccess()) {
            resp.sortByDistance(sortDirect);
        }
        return resp;
    }


    /**
     * @param originList      出发地
     * @param destinationList 目的地
     */
    public QQMultiOriginMultiDestDistanceResp getMultiDistanceBetween(List<LngLat> originList, List<LngLat> destinationList) {
        return this.getMultiDistanceBetweenOrderBy(originList, destinationList, null);
    }


    /**
     * 文档：https://lbs.qq.com/webservice_v1/guide-distancematrix.html
     *
     * @param originList      出发地
     * @param destinationList 目的地
     * @param sortDirect      按照距离排序方式
     */
    public QQMultiOriginMultiDestDistanceResp getMultiDistanceBetweenOrderBy(List<LngLat> originList, List<LngLat> destinationList, Sort.Direction sortDirect) {

        Assert.notEmpty(originList, "源地址经纬度不能为空");
        Assert.notEmpty(destinationList, "目的地经纬度不能为空");

        //例如： http://apis.map.qq.com/ws/distance/v1/matrix/?mode=driving&from=39.984092,116.306934;40.007763,116.353798&to=39.981987,116.362896;39.949227,116.394310&key=OB4BZ-D4W3U-B7VVO-4PJWW-6TKDJ-WPB77

        final String delimiter = ";";
        String from = originList.stream().map(LngLat::toLatLngString).collect(Collectors.joining(delimiter));
        String to = destinationList.stream().map(LngLat::toLatLngString).collect(Collectors.joining(delimiter));

        String apiPath = "/ws/distance/v1/matrix";
        Map<String, String> reqParamMap = new LinkedHashMap<>();
        reqParamMap.put("mode", DistanceCalcModeEnum.driving.name());
        reqParamMap.put("from", from);
        reqParamMap.put("to", to);
        reqParamMap.put("key", props.getKey());
        String sign = this.calcSign(apiPath, reqParamMap);
        String url = QQ_MAP_API_DOMAIN + apiPath + "/?" + joinParams(reqParamMap,true) + "&sig=" + sign;

        QQMultiOriginMultiDestDistanceResp resp = doGetForObject(url,QQMultiOriginMultiDestDistanceResp.class);

        if (resp != null && resp.isSuccess()) {
            final int rowCount = originList.size();
            final int columnCount = destinationList.size();

            //循环出发地列表
            for (int rowIdx = 0; rowIdx < rowCount; rowIdx++) {
                QQMultiOriginMultiDestDistanceResp.OneOriginDistance distanceDto = resp.getResult().getOriginDistanceList().get(rowIdx);
                distanceDto.setOrigin(originList.get(rowIdx));

                for (int colIdx = 0; colIdx < columnCount; colIdx++) {
                    distanceDto.getDistanceList().get(colIdx).setDest(destinationList.get(colIdx));
                }

                //按照距离 由近到源排列
                distanceDto.sortByDistance(sortDirect);
            }
        }
        return resp;
    }


    private <T> T doGetForObject(String url, Class<T> respClass) {
          CompletableFuture<T> resultFuture =
                CompletableFuture.supplyAsync(() -> {
                    return restTemplate.getForObject(url, respClass);
                }, executorService).handle((resp, ex) -> {
                    if(ex!=null) {
                        ex.printStackTrace();
                    }
                    return resp;
                });

        return resultFuture.join();
    }


    private void copyStatusMessage(QQMapBaseResp from, QQMapBaseResp target) {
        target.setStatus(from.getStatus());
        target.setMessage(from.getMessage());
    }


    //拼接成  a=1&b=2&c=3
    private String joinParams(Map<String, String> reqParamMap,Boolean urlEncoding) {
        if (reqParamMap.isEmpty()) {
            return "";
        }
        //首先对参数进行排序：按参数名升序
        List<String> paramNameList = new ArrayList(reqParamMap.keySet());
        Collections.sort(paramNameList);

        StringBuilder paramPairBuf = new StringBuilder();
        int paramCount = paramNameList.size();
        for (int i = 0; i < paramCount; i++) {
            String paramName = paramNameList.get(i);
            paramPairBuf.append(paramName).append("=").append(reqParamMap.get(paramName));
            if (i < (paramCount - 1)) {
                paramPairBuf.append("&");
            }
        }

        return paramPairBuf.toString();
    }


    /**
     * 参考文档：https://lbs.qq.com/FAQ/server_faq.html#3
     * 计算签名:请求路径+”?”+请求参数+SK进行拼接，并计算拼接后字符串md5值，即为签名(sig)：
     */
    private String calcSign(String apiPath, Map<String, String> reqParamMap) {
        if(!props.isValidSign()) {
            return "";
        }

        String reqParams = joinParams(reqParamMap,false);
        //md5("/ws/geocoder/v1?key=5Q5BZ-5EVWJ-SN5F3-*****&location=28.7033487,115.8660847SWvT26ypwq5Nwb5RvS8cLi6NSoH8HlJX")
        String toSignStr = apiPath + "?" + reqParams + props.getSk();
        String sign = DigestUtils.md5Hex(toSignStr);
        return sign;
    }

}
