package com.rocogz.syy.common.tencent;

import com.rocogz.syy.common.tencent.resp.QQMapBaseResp;
import com.rocogz.syy.common.tencent.resp.QQMultiOriginMultiDestDistanceResp;
import com.rocogz.syy.common.tencent.resp.QQSingleOriginMultiDestDistanceResp;
import com.rocogz.syy.common.tencent.resp.QQUniqueDistanceResp;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Sort;
import org.springframework.data.geo.Distance;
import org.springframework.data.geo.Point;
import org.springframework.data.redis.connection.RedisGeoCommands;
import org.springframework.data.redis.core.GeoOperations;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.util.Assert;

import java.util.ArrayList;
import java.util.List;

/**
 * 基于redis实现的 计算距离服务
 * @author zhangmin
 * @date 2020/4/16
 */
public class RocoMapService {

    @Autowired
    private RedisTemplate<String,String> redisTemplate;

    /**
     * 计算两点之间的距离
     * @param origin      实时定位的用户位置：经纬度
     * @param destination 目的地位置：经纬度
     */
    public QQUniqueDistanceResp getUniqueDistanceBetween(LngLat origin, LngLat destination) {

        Assert.notNull(destination,"目的地经纬度不能为NULL");

        List<LngLat> toList = new ArrayList<>();
        toList.add(destination);
        QQSingleOriginMultiDestDistanceResp tempResult = this.getMultiDistanceBetween(origin,toList);
        if(tempResult == null) {
            return null;
        }

        QQUniqueDistanceResp resp = new QQUniqueDistanceResp();

        copyStatusMessage(tempResult,resp);

        if(tempResult.isSuccess()) {
            resp.setResult(tempResult.getResult().get(0));
        }

        return resp;
    }


    /**
     *  计算 （单出发地-> 多目的地） 距离
     * @param origin 出发地
     * @param destinationList 多个目的地
     */
    public QQSingleOriginMultiDestDistanceResp getMultiDistanceBetween(LngLat origin, List<LngLat> destinationList) {
        Assert.notNull(origin,"源地点经纬度不能为NULL");
        Assert.notEmpty(destinationList,"目的地经纬度不能为空");
        String redisKey = "geoDistances:" + System.currentTimeMillis();

        GeoOperations<String, String> geoOpers = redisTemplate.opsForGeo();

        String from = "from";
        String to = "to-";

        List<RedisGeoCommands.GeoLocation<String>> geoList = new ArrayList<>();

        geoList.add(new RedisGeoCommands.GeoLocation(from,new Point(origin.getLng(),origin.getLat())));

        final int length = destinationList.size();

        for(int idx=0; idx<length;idx++) {
            LngLat destination = destinationList.get(idx);
            geoList.add(new RedisGeoCommands.GeoLocation(to+idx,new Point(destination.getLng(),destination.getLat())));
        }

        //把这些位置加入到 geo集合中
        geoOpers.add(redisKey,geoList);


        List<DistanceTime> resultList = new ArrayList<>();

        for(int idx=0; idx<length;idx++) {
            //计算出发点 到目的地到距离,返回距离米
            Distance geoDist = geoOpers.distance(redisKey,from,to+idx, RedisGeoCommands.DistanceUnit.METERS);
            DistanceTime disTime = new DistanceTime();
            disTime.setDistance((int)geoDist.getValue());
            disTime.setDest(destinationList.get(idx));
            resultList.add(disTime);
        }

        QQSingleOriginMultiDestDistanceResp resp = new QQSingleOriginMultiDestDistanceResp();
        resp.setStatus(0);
        resp.setResult(resultList);
        redisTemplate.delete(redisKey);
        return resp;
    }


    /**
     *  计算 （单出发地-> 多目的地） 距离
     * @param origin 出发地
     * @param destinationList 多个目的地
     * @param sortDirect 按照距离排序方式
     */
    public QQSingleOriginMultiDestDistanceResp getMultiDistanceBetweenOrderBy(LngLat origin,List<LngLat> destinationList,Sort.Direction sortDirect) {
        QQSingleOriginMultiDestDistanceResp resp = getMultiDistanceBetween(origin,destinationList);
        if(resp!=null && resp.isSuccess()) {
            resp.sortByDistance(sortDirect);
        }
        return resp;
    }

    /**
     * @param originList 出发地
     * @param destinationList 目的地
     */
    public QQMultiOriginMultiDestDistanceResp getMultiDistanceBetween(List<LngLat> originList, List<LngLat> destinationList) {
        return this.getMultiDistanceBetweenOrderBy(originList,destinationList,null);
    }


    /**
     * @param originList 出发地
     * @param destinationList 目的地
     * @param sortDirect 按照距离排序方式
     */
    public QQMultiOriginMultiDestDistanceResp getMultiDistanceBetweenOrderBy(List<LngLat> originList, List<LngLat> destinationList,Sort.Direction sortDirect) {
        Assert.notEmpty(originList,"源地址经纬度不能为空");

        QQMultiOriginMultiDestDistanceResp multiDestDistanceResp = new QQMultiOriginMultiDestDistanceResp();

        QQMultiOriginMultiDestDistanceResp.MultiOriginDistance multiOrgiginDist  = new QQMultiOriginMultiDestDistanceResp.MultiOriginDistance();

        //循环出发地列表
        for(int rowIdx =0; rowIdx<originList.size(); rowIdx++) {
            LngLat from = originList.get(rowIdx);

            QQSingleOriginMultiDestDistanceResp thisOrginResp = this.getMultiDistanceBetweenOrderBy(from,destinationList,sortDirect);

            QQMultiOriginMultiDestDistanceResp.OneOriginDistance distanceDto = new QQMultiOriginMultiDestDistanceResp.OneOriginDistance();

            distanceDto.setOrigin(from);
            if(thisOrginResp!=null) {
                distanceDto.setDistanceList(thisOrginResp.getResult());
            }

            multiOrgiginDist.addOriginDistance(distanceDto);
        }

        multiDestDistanceResp.setStatus(0);
        multiDestDistanceResp.setResult(multiOrgiginDist);
        return multiDestDistanceResp;
    }



    private void copyStatusMessage(QQMapBaseResp from, QQMapBaseResp target) {
        target.setStatus(from.getStatus());
        target.setMessage(from.getMessage());
    }

}
