package com.rocogz.merchant.entity.scm;

import com.baomidou.mybatisplus.annotation.TableField;
import com.baomidou.mybatisplus.core.toolkit.StringUtils;
import com.rocogz.merchant.constant.Constant;
import com.rocogz.merchant.entity.agent.goods.MerchantAgentGoods;
import com.rocogz.merchant.entity.goods.MerchantGoods;
import com.rocogz.syy.common.entity.IdEntity;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.experimental.Accessors;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.List;

/**
 * <p>
 * 订单信息
 * </p>
 *
 * @author liangyongtong
 * @since 2021-03-09
 */
@Data
@EqualsAndHashCode(callSuper = true)
@Accessors(chain = true)
public class MerchantScmOrderInformation extends IdEntity {

    private static final long serialVersionUID = 1L;

    /**
     * 订单号,与权益系统的清单编号1对1 关联
     */
    private String orderCode;

    /**
     * 下单时间:等于发放时间
     */
    private LocalDateTime orderTime;

    /**
     * 发放时间
     */
    private LocalDateTime grantTime;

    /**
     * 发放成功时间
     */
    private LocalDateTime grantSuccessTime;

    /**
     * 发放总张数 = 清单表中的数量
     * 等于 调用供应链系统 传入的数量 com.rocogz.merchant.dto.scm.ScmSendDto#quantity
     *
     * @see com.rocogz.syy.equity.entity.batchDistributionCouponApply.EquityBatchDistributionCouponApplyDetail#quantity
     */
    private Integer quantity;

    /**
     * 发放成功张数，即发放用户券成功的上游订单数量，会等于 MerchantScmOrderDetai.userCouponGrantStatus = 'SUCCESS' 的数量
     * 上游订单MerchantScmOrderDetail#userCouponGrantStatus='SUCCESS'时, quantitySuccess+1
     * 如果是直充产品,1个下游充值订单 会按照面额分拆成功多个上游订单，所以可能会出现 quantitySuccess > quantity
     */
    private Integer quantitySuccess;

    /**
     * 订单总金额/商品总面值,已经乘过quantity的金额
     * (1)非直充：totalPrice = 平台产品市场价  platformProductMarketPrice * quantity
     * (2)直充： totalPrice = 输入的自定义面额 com.rocogz.merchant.entity.scm.MerchantScmInformation#unitFaceValue * quantity
     */
    private BigDecimal totalPrice;

    /**
     * 发放成功金额
     */
    private BigDecimal successPrice;

    /**
     * 订单发放用户券状态:初始化为 "待发放"状态 (下游订单状态)
     * 下游订单状态包括：PENDING(待发放), IN_PROCESS(发放中)，SUCCESS（已发放),CANCEL(已作废）,BACK(已撤回)
     *
     * @see Constant.DictData.SCM_ORDER_STATUS_PENDING
     */
    private String orderStatus;

    /**
     * 明细订单数量,拆单后分拆的下游订单数量 = 该订单下 merchant_scm_order_detail的行数
     */
    private Integer secondOrderNum;

    /**
     * 矫正执行次数
     */
    private Integer revisionRetryNum;

    /**
     * 领取验证状态
     *
     * @see Constant.DictData#RECEIVE_VALIDATE_STATUS_UNRECEIVED
     */
    private String receiveValidateStatus;

    /**
     * 领取验证时间
     */
    private LocalDateTime receiveValidateTime;

    /**
     * 额度支付状态
     *
     * @see Constant.DictData#SCM_PAY_STATUS_UNPAID
     */
    private String payStatus;

    /**
     * 总额度
     * （1）直充产品：payPrice = { purchasePrice(客户产品或代理商产品采购价)* unitFaceValue(自定义面额) / customerMarketPrice (客户产品市场价) } * quantity
     * (2)非直充： payPrice =  purchasePrice(客户产品或代理商产品采购价) * quantity
     * payPrice = payQuotaPrice + payCashPrice
     */
    private BigDecimal payPrice;

    /**
     * 支付方式:POINT 积分支付的订单，对应的grantWay是 EXCHANGE,包括的orderType:（POINT或ACTIVITY_GIFT）
     *
     * @see Constant.DictData#SCM_PAY_WAY
     */
    private String payWay;

    /**
     * 额度支付尝试次数
     */
    private Integer payRetryNum;

    /**
     * 支付结果提示
     */
    private String payReason;

    /**
     * 额度支付时间
     */
    private LocalDateTime payTime;

    /**
     * 额度领取状态，额度账户中领取额度是否已处理
     * 君同的额度账户中，是否成功增加已领取金额,当上游订单全部领取成功了才调用juntong额度账户，增加已领取金额
     */
    private String receiveQuotaStatus;

    /**
     * 客户备付金扣减时点: DEDUCTION_ON_ISSUE("发放时扣减"),  DEDUCTION_ON_COLLECTION("领取时扣减")
     *
     * @see com.rocogz.syy.equity.entity.batchDistributionCouponApply.EquityBatchDistributionCouponApply#dedutionTimePoint
     * @see com.rocogz.syy.settlement.constant.SettlementConstant.DeductionTimeAppointDictData
     */
    private String dedutionTimePoint;

    /**
     * 备付金账户增加已发放额度状态,如果额度扣减成功了,才更新此字段
     * 如果额度扣减成功了，用来标记：备付金账户额度是否增加成功了
     *
     * @see Constant.DictData#SCM_ORDER_STATUS_PENDING
     */
    private String coverGrantStatus;

    /**
     * 额度退款状态
     *
     * @see Constant.DictData#REFUNDED_STATUS_UNREFUNDED
     */
    private String refundStatus;

    /**
     * 额度退款金额
     */
    private BigDecimal refundPrice;

    /**
     * 额度退款时间
     */
    private LocalDateTime refundTime;

    /**
     * 撤回时间
     */
    private LocalDateTime recallTime;

    /**
     * 订单类型：后台发放/赠送订单/积分兑换
     *
     * @see Constant.DictData#SCM_ORDER_TYPE_SYS
     */
    private String orderType;

    /**
     * 支付额度金额: settle系统中额度账户要扣减的额度
     */
    private BigDecimal payQuotaPrice;

    /**
     * 现金支付金额
     */
    private BigDecimal payCashPrice;

    /**
     * 扣减的合作客户备付金总额,整个下游订单要扣减的客户备付金总额 = customerPayPrice
     * 勾稽金额 就是该金额,收款单要结算的金额
     * 也就是：报表模块中显示的应结算总金额
     * (1)直充产品：payCoverPrice = {purchaseSettlePrice(客户或代理商产品采购结算价)* unitFaceValue(自定义面额) / customerMarketPrice (客户产品市场价) } * quantity
     * (2)非直充： payCoverPrice = purchaseSettlePrice(客户或代理商产品采购结算价) * quantity
     */
    private BigDecimal payCoverPrice;

    /**
     * 结算状态
     *
     * @see Constant.DictSettleStatus#UN_SETTLE
     */
    private String settleStatus;

    /**
     * 完成结算时间
     */
    private LocalDateTime settleTime;

    /**
     * 结算批次号
     */
    private String settleBatchCode;

    /**
     * 本批次结算结果是否同步更新到结算系统中
     */
    private Boolean syncToSettle;


    /**
     * 本订单在某个收款单中占用的 结算的金额[页面展示使用此字段]
     */
    private transient BigDecimal settledAmount;

    /**
     * 服务商备付金支付总金额 = 服务商备付金扣减单价 * 数量 - 现金支付金额
     * 如果是B端的并且是纯现金支付的，则服务商备付金=0
     */
    private BigDecimal agentPayPrice;

    /**
     * 服务商备付金支付状态
     * @see Constant.DictData#SCM_PAY_STATUS_SUCCESS
     */
    private String agentPayStatus;

    /**
     * 服务商备付金支付时间
     */
    private LocalDateTime agentPayTime;

    /**
     * 服务商备付金支付流水
     */
    private String agentPayFlow;

    /**
     * 服务商备付金支付次数
     */
    private Integer agentPayRetryNum;

    /**
     * 服务商备付金扣减时点
     * @see Constant.MerchantAgentGoodsDeductionTime#DEDUCTION_ON_ISSUANCE
     * @see MerchantAgentGoods#deductionTime
     */
    private String agentDeductionTime;

    /**
     * 服务商备付金更新领取金额
     */
    private String agentReceiveStatus;

    /**
     * 初始情况为 INIT
     * 额度扣减成功、同时服务商备付金发放金额增加成功为 SUCCESS
     * 额度扣减成功、同时服务商备付金发放金额增加失败为 FAILURE
     *
     * @see com.rocogz.merchant.constant.Constant.DictData#COMMON_STATUS_INIT
     * @see com.rocogz.merchant.constant.Constant.DictData#COMMON_STATUS_SUCCESS
     * @see com.rocogz.merchant.constant.Constant.DictData#COMMON_STATUS_FAILURE
     * 服务商备付金更新发放金额状态
     */
    private String agentGrantStatus;

    /**
     * 服务商备付金退款状态
     */
    private String agentRefundStatus;

    /**
     * 服务商备付金退款金额
     */
    private BigDecimal agentRefundPrice;

    /**
     * 服务商备付金退款时间
     */
    private LocalDateTime agentRefundTime;

    /**
     * 服务商备付金退款流水
     */
    private String agentRefundFlow;

    /**
     * 合作客户备付金支付金额,也会等于payCoverPrice
     */
    private BigDecimal customerPayPrice;

    /**
     * 合作客户备付金支付状态
     */
    private String customerPayStatus;

    /**
     * 合作客户备付金支付时间
     */
    private LocalDateTime customerPayTime;

    /**
     * 合作客户备付金支付流水
     */
    private String customerPayFlow;


    /**
     * 合作客户备付金支付次数
     */
    private Integer customerPayRetryNum;

    /**
     * 合作客户备付金退款价格
     */
    private BigDecimal customerRefundPrice;

    /**
     * 合作客户备付金退款状态
     */
    private String customerRefundStatus;

    /**
     * 合作客户备付金退款时间
     */
    private LocalDateTime customerRefundTime;

    /**
     * 合作客户备付金退款流水
     */
    private String customerRefundFlow;

    /**
     * 上游供应商券发放模式
     *
     * @see Constant.RedisKey#PLATFORM_GRANT
     * @see MerchantGoods#supplierGrantMode
     */
    private String supplierGrantMode;

    /**
     * 是否需要调用权益系统发放用户券接口 (Y:是, N:否)
     * @see Constant.DictData#COMMON_STATUS_YES
     */
    private String whetherNeedIssueCoupon;

    /**
     * 是否需要库存（Y/N）：占用库存，增加销量等
     * @see Constant.DictData#COMMON_STATUS_YES
     * @see MerchantGoods#whetherNeedStock
     */
    private String whetherNeedStock;

    /**
     * 需要仓库的库存编号
     */
    private String warehouseCode;

    /**
     * 是否需要发放用户券
     */
    public boolean needIssueCoupon() {
        return Constant.DictData.COMMON_STATUS_YES.equals(this.whetherNeedIssueCoupon);
    }

    /**
     * 是否需要走库存逻辑
     * @return
     */
    public boolean needScmStock() {
        return Constant.DictData.COMMON_STATUS_YES.equals(this.whetherNeedStock) &&
                StringUtils.isNotEmpty(warehouseCode);
    }

    /**
     * 是否时领取后发放
     */
    public boolean issueAfterReceive() {
        return Constant.RedisKey.USER_COLLECT_COUPON.equals(this.supplierGrantMode);
    }


    /**
     * 供应链发放信息
     */
    @TableField(exist = false)
    private MerchantScmInformation scmInformation;

    /**
     * 发放的产品信息
     */
    @TableField(exist = false)
    private MerchantScmProductInformation scmProductInformation;

    /**
     * 发放目标用户信息
     */
    @TableField(exist = false)
    private MerchantScmTargetUser scmTargetUser;

    /**
     * 包含多个明细订单
     */
    @TableField(exist = false)
    private List<MerchantScmOrderDetail> orderDetails;


}
