package com.rocogz.account.api.enums.account;


import java.math.BigDecimal;

/**
 * 交易流水类型
 *
 * @author Andrew
 * @create 2020/4/15 9:38 上午
 */
public enum AccountTradeRecordType {


    /**
     * 余额增加
     */
    ADD,


    /**
     * 金额减少
     */
    DECR,

    /**
     * 划拨收回
     */
    TRANSFER_BACK,


    /**
     * 划拨转出
     */
    TRANSFER_OUT,

    /**
     * 金额冻结
     */
    FROZEN,

    /**
     * 金额解冻
     */
    UNFROZEN,

 
    /**
     * 金额冻结后消费
     */
    CONSUME_UNFROZEN,
    
    /**
     * 直接消费
     */
    CONSUME,


    /**
     * 金额消费后退回
     */
    REFUND;


    /**
     * 新增标志
     */
    public static final String ADD_FLAG = "+";
    /**
     * 减少标志
     */
    public static final String DECR_FLAG = "-";


    public static String getFlag(BigDecimal amount, AccountAmountType amountType, String type) {
        try {
            AccountTradeRecordType transRecordType = valueOf(type);

            switch (transRecordType) {
                case ADD:
                    //余额增加
                    return getFlag(AccountAmountType.BALANCE, null, amountType, amount);
                case DECR:
                    //余额减少
                    return getFlag(null, AccountAmountType.BALANCE, amountType, amount);
                case TRANSFER_BACK:
                    //余额增加，   已划拨减少
                    return getFlag(AccountAmountType.BALANCE, AccountAmountType.TRANSFER_AMOUNT, amountType, amount);
                case TRANSFER_OUT:
                    //余额减少，   已划拨增加
                    return getFlag(AccountAmountType.TRANSFER_AMOUNT, AccountAmountType.BALANCE, amountType, amount);
                case FROZEN:
                    //冻结增加， 余额减少
                    return getFlag(AccountAmountType.FROZEN_AMOUNT, AccountAmountType.BALANCE, amountType, amount);
                case UNFROZEN:
                    //余额增减 冻结减少
                    return getFlag(AccountAmountType.BALANCE, AccountAmountType.FROZEN_AMOUNT, amountType, amount);
                case CONSUME_UNFROZEN:
                    //冻结减少， 已用增加
                    return getFlag(AccountAmountType.USED_AMOUNT, AccountAmountType.FROZEN_AMOUNT, amountType, amount);
                case CONSUME:
                    //余额减少， 已用增加
                    return getFlag(AccountAmountType.USED_AMOUNT, AccountAmountType.BALANCE, amountType, amount);
                case REFUND:
                    //余额增加， 已用减少
                    return getFlag(AccountAmountType.BALANCE, AccountAmountType.USED_AMOUNT, amountType, amount);
                default:
                    return null;
            }

        } catch (Exception e) {
            return null;
        }
    }

    /**
     * @param addType  增加的类型
     * @param decrType 减少的类型
     * @param curType  当前的类型
     * @param amount   金额
     * @return
     */
    public static String getFlag(AccountAmountType addType, AccountAmountType decrType, AccountAmountType curType, BigDecimal amount) {
        if (addType == curType) {
            return ADD_FLAG.concat(amount.stripTrailingZeros().toPlainString());
        } else if (decrType == curType) {
            return DECR_FLAG.concat(amount.stripTrailingZeros().toPlainString());
        } else {
            return null;
        }
    }



}
