package com.rocogz.swagger.spring.boot.autoconfigure;

import com.google.common.base.Predicate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.autoconfigure.condition.ConditionalOnWebApplication;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.annotation.AnnotatedElementUtils;
import org.springframework.core.env.ConfigurableEnvironment;
import org.springframework.stereotype.Controller;
import org.springframework.ui.ModelMap;
import org.springframework.util.AntPathMatcher;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.mvc.method.RequestMappingInfo;
import org.springframework.web.servlet.mvc.method.annotation.RequestMappingHandlerMapping;
import springfox.documentation.builders.ApiInfoBuilder;
import springfox.documentation.builders.PathSelectors;
import springfox.documentation.builders.RequestHandlerSelectors;
import springfox.documentation.service.*;
import springfox.documentation.spi.DocumentationType;
import springfox.documentation.spi.service.contexts.SecurityContext;
import springfox.documentation.spring.web.plugins.Docket;
import springfox.documentation.swagger2.annotations.EnableSwagger2;

import javax.annotation.PostConstruct;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import java.util.*;
import java.util.stream.Collectors;

import static com.rocogz.swagger.spring.boot.autoconfigure.SwaggerProperties.CompanyCopyRight;

/**
 * 自动配置成功后访问：http://localhost:port/swagger-ui.html
 * 使用说明：
 * （1）导入 swagger-starter 依赖包以后,在配置application.properties文件中：配置如下：
 *
 *      swagger.basePackage=com.rocogz.demo.controller  //您的controller基础包
 *
 * (2)swagger在导入依赖后,默认开启生成api文档功能,在生产环境应该禁用swagger功能,禁用swagger需配置如下:
 *     swagger.enabled=false
 *
 *  swagger自动配置
 * @author zhangmin
 * @date 2020/1/15
 */
@ConditionalOnWebApplication
@ConditionalOnClass(Controller.class)
@ConditionalOnMissingBean(Docket.class)
@EnableSwagger2
@EnableConfigurationProperties(SwaggerProperties.class)
@Configuration
@ConditionalOnProperty(name = "swagger.enabled", havingValue = "true", matchIfMissing = true)
public class Swagger2AutoConfiguration {

    @Autowired
    private SwaggerProperties swaggerProps;
    @Autowired
    private ConfigurableEnvironment env;
    @Autowired
    private RequestMappingHandlerMapping handlerMapping;

    //允许匿名访问的路径集合,支持Ant 路径Pattern, 多个路径用逗号分割,配置文件中的与注解的配置一起作用（只针对全局配置为授权时有效）
    private List<String> anonApiPathPatternList;

    @PostConstruct
    private void initAnonApiPath() {
        if(CollectionUtils.isEmpty(swaggerProps.getAuthHeaderNameList())) {
            //表示所有的接口 都是匿名访问
            return;
        }

        //如果配置了需要认证访问，则初始化哪些接口可以匿名访问

        anonApiPathPatternList = new ArrayList<>();

        Map<RequestMappingInfo, HandlerMethod> handlerMethodMap = handlerMapping.getHandlerMethods();

        handlerMethodMap.forEach((reqMapping,handlerMethod)-> {
           //handler Bean 是否标注了 ApiAnonAccess 注解
           boolean isAnnoHandlerBean = AnnotatedElementUtils.hasAnnotation(handlerMethod.getBeanType(),ApiAnonAccess.class);
           //handlerMethod 是否标注了 ApiAnonAccess注解
           boolean isAnnoHandlerMethod = AnnotatedElementUtils.hasAnnotation(handlerMethod.getMethod(),ApiAnonAccess.class);

           if(isAnnoHandlerBean || isAnnoHandlerMethod) {
               //获取HandlerMethod上所有的 urlPattern
               Set<String> patterns = reqMapping.getPatternsCondition().getPatterns();
               anonApiPathPatternList.addAll(patterns);
           }
        });

       List<String> configedAnonPathList = this.getConfigedAnonPathPatternList();

       for(String pathPattern : configedAnonPathList) {
           if(!anonApiPathPatternList.contains(pathPattern)) {
               anonApiPathPatternList.add(pathPattern);
           }
       }
    }

    @Bean
    public Docket publicSwaggerRestApi() {
        Docket docket = new Docket(DocumentationType.SWAGGER_2)
                .groupName("匿名访问接口")
                //设置接口中忽略的参数类型
                .ignoredParameterTypes(getIgnoredParameterTypes())
                .apiInfo(apiInfo())
                .select()
                .apis(RequestHandlerSelectors.basePackage(swaggerProps.getBasePackage()))
                .paths(getAnouAccessPathPredicate())
                .build();
        return docket;
    }

    @ConditionalOnProperty(name = "swagger.auth-header-names")
    @Bean
    public Docket authSwaggerRestApi() {
        Docket docket = new Docket(DocumentationType.SWAGGER_2)
                .groupName("认证访问接口")
                //设置接口中忽略的参数类型
                .ignoredParameterTypes(getIgnoredParameterTypes())
                .apiInfo(apiInfo())
                .select()
                .apis(RequestHandlerSelectors.basePackage(swaggerProps.getBasePackage()))
                .paths(getAuthPathPredicate())
                .build()
                .securitySchemes(security())
                .securityContexts(securityContexts());
        return docket;
    }





    private Class[] getIgnoredParameterTypes() {
        return new Class[] {ModelMap.class,
                HttpServletRequest.class,HttpServletResponse.class,
                HttpSession.class
        };
    }

    private ApiInfo apiInfo() {
        ApiInfo apiInfo = new ApiInfoBuilder().title(getTitle())
                .description(getDescription())
                .termsOfServiceUrl(getServiceUrl())
                .contact(getContact())
                .version(getVersion())
                .build();

        return apiInfo;
    }


    private String getTitle() {
        String title = CompanyCopyRight.DEFAULT_COMPANY.getName();
        CompanyCopyRight company = swaggerProps.getCompany();
        if(company!=null && !StringUtils.isEmpty(company.getName())) {
            title = company.getName();
        }
        return title;
    }

    //得到项目描述
    private String getDescription() {
        String description = null;
        CompanyCopyRight company = swaggerProps.getCompany();
        if(company!=null && !StringUtils.isEmpty(company.getDescription())) {
            description = company.getDescription();
        }else {
            //得到系统的名称
            description = env.getProperty("spring.application.name");
        }

        if(StringUtils.isEmpty(description)) {
            description = CompanyCopyRight.DEFAULT_COMPANY.getDescription();
        }
        return description;
    }


    private String getServiceUrl() {
        String serviceUrl = CompanyCopyRight.DEFAULT_COMPANY.getSiteUrl();
        CompanyCopyRight company = swaggerProps.getCompany();
        if(company!=null && !StringUtils.isEmpty(company.getSiteUrl())) {
            serviceUrl = company.getSiteUrl();
        }
        return serviceUrl;
    }


    private String getVersion() {
        String version = CompanyCopyRight.DEFAULT_COMPANY.getVersion();
        CompanyCopyRight company = swaggerProps.getCompany();
        if(company!=null && !StringUtils.isEmpty(company.getVersion())) {
            version = company.getVersion();
        }
        return version;
    }


    private Contact getContact() {
        String contactName = CompanyCopyRight.DEFAULT_COMPANY.getContactName();
        String contactEmail = CompanyCopyRight.DEFAULT_COMPANY.getContactEmail();

        CompanyCopyRight company = swaggerProps.getCompany();
        if(company!=null) {

            if (!StringUtils.isEmpty(company.getContactName())) {
                contactName = company.getContactName();
            }

            if(!StringUtils.isEmpty(company.getContactEmail())) {
                contactEmail = company.getContactEmail();
            }
        }

        return new Contact(contactName,getServiceUrl(),contactEmail);
    }



    private List<ApiKey> security() {
        List<ApiKey> list = new ArrayList<>();
        //Authorization
        String passAs = "header";
        for(String authHeaderName : swaggerProps.getAuthHeaderNameList()) {
            list.add(new ApiKey(authHeaderName, authHeaderName, passAs));
        }
        return list;
    }


    /**
     * 返回匿名接口组中 path 选择器
     * @return
     */
    private Predicate<String> getAnouAccessPathPredicate() {

        if(CollectionUtils.isEmpty(swaggerProps.getAuthHeaderNameList())) {
            //表示所有的接口 都是匿名访问
            return PathSelectors.any();
        }

        //以下处理 authHeader 不为空,则表示需要认证才能访问

        if(CollectionUtils.isEmpty(anonApiPathPatternList)) {
            //没有配置 可以匿名访问的接口，则表示全部需要认证访问
            return PathSelectors.none();
        }

        //过滤出 可以匿名访问都接口
       return new Predicate<String>() {
            public boolean apply(String input) {
                AntPathMatcher matcher = new AntPathMatcher();
                for(String anonPathPattern: anonApiPathPatternList) {
                    //如果匿名可以访问则时间返回true
                    if(matcher.match(anonPathPattern,input)) {
                        return true;
                    }
                }
                return false;
            }
        };
    }

    /**
     * 返回需要经过 token认证的 路径选择器
     * @return
     */
    private Predicate<String> getAuthPathPredicate() {

        if(CollectionUtils.isEmpty(anonApiPathPatternList)) {
            //表示所有的接口都需要授权才能访问
            return PathSelectors.any();
        }


        return new Predicate<String>() {
            public boolean apply(String input) {
                AntPathMatcher matcher = new AntPathMatcher();
                for(String anonPathPattern: anonApiPathPatternList) {
                    if(matcher.match(anonPathPattern,input)) {
                        return false;
                    }
                }
                return true;
            }
        };
    }


    /**
     * 获得配置文件中 配置的允许访问的路径
     * @return
     */
    private List<String> getConfigedAnonPathPatternList() {
        if(!StringUtils.hasText(swaggerProps.getAnonApiPathPatterns())) {
            return Collections.emptyList();
        }

        return Arrays.stream(StringUtils.trimArrayElements(swaggerProps.getAnonApiPathPatterns().split(","))).collect(Collectors.toList());
    }

    private List<SecurityContext> securityContexts() {
        List<SecurityContext> securityContexts= new ArrayList<>();
        securityContexts.add(
                SecurityContext.builder()
                        .securityReferences(defaultAuth())
                        //设置需要认证的路径
                        .forPaths(getAuthPathPredicate())
                        .build());
        return securityContexts;
    }

    private List<SecurityReference> defaultAuth() {

        AuthorizationScope[] authorizationScopes = new AuthorizationScope[]{new AuthorizationScope("global", "accessEverything")} ;

        List<SecurityReference> securityReferences=new ArrayList<>();

        for(String authHeaderName : swaggerProps.getAuthHeaderNameList()) {
            securityReferences.add(new SecurityReference(authHeaderName, authorizationScopes));
        }

        return securityReferences;
    }

}
