package com.rocogz.syy.user.client;

import com.rocogz.syy.common.PageTable;
import com.rocogz.syy.common.response.Response;
import com.rocogz.syy.user.client.agent.ReqAgentParam;
import com.rocogz.syy.user.constant.UserConstant;
import com.rocogz.syy.user.dto.UserFullInfoDto;
import com.rocogz.syy.user.dto.UserMsgWebDto;
import com.rocogz.syy.user.dto.agent.*;
import com.rocogz.syy.user.dto.popularize.UserPopularizeApplyDTO;
import com.rocogz.syy.user.entity.*;
import com.rocogz.syy.user.entity.agent.UserAgentInvite;
import com.rocogz.syy.user.entity.certificate.UserCertificateInfo;
import com.rocogz.syy.user.entity.certificate.UserDriverLicenceInfo;
import com.rocogz.syy.user.entity.certificate.UserDriveringLicenceInfo;
import com.rocogz.syy.user.entity.delivery.address.UserDeliveryAddress;
import com.rocogz.syy.user.entity.identity.UserAgentIdentityAuth;
import com.rocogz.syy.user.entity.identity.UserIdentity;
import com.rocogz.syy.user.entity.popularize.UserPopularizeApply;
import com.rocogz.syy.user.entity.popularize.UserPopularizeInfo;
import org.springframework.web.bind.annotation.*;

import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import java.util.List;

/**
 * The interface User client service.
 *
 * @author：liangyongtong
 * @since：2020/3/25
 */
public interface IUserClientService {

    // =========  start of  用户认证信息 by ranxiangfu   ===========

    /**
     * Save or update user authentication info response.
     *
     * @param userAuthenticationInfo the user authentication info
     * @return the response
     */
    @PostMapping("/api/user/user-authentication-info")
    Response saveOrUpdateUserAuthenticationInfo(@RequestBody UserAuthenticationInfo userAuthenticationInfo);

    /**
     * Update user authentication info by user code response.
     *
     * @param userAuthenticationInfo the user authentication info
     * @return the response
     */
    @PostMapping("/api/user/user-authentication-info/updateByUserCode")
    Response updateUserAuthenticationInfoByUserCode(@RequestBody UserAuthenticationInfo userAuthenticationInfo);

    /**
     * Gets user authentication info by user code.
     *
     * @param userCode the user code
     * @return the user authentication info by user code
     */
    @GetMapping("/api/user/user-authentication-info/getByUserCode")
    Response<UserAuthenticationInfo> getUserAuthenticationInfoByUserCode(@RequestParam("userCode") String userCode);

    /**
     * 根据手机号获取认证信息.
     *
     * @param mobile the mobile
     * @return the user authentication info by mobile
     */
    @GetMapping("/api/user/user-authentication-info/getByMobile")
    Response<UserAuthenticationInfo> getUserAuthenticationInfoByMobile(@RequestParam("mobile") String mobile);

    /**
     * 根据邮箱获取认证信息.
     *
     * @param email the email
     * @return the user authentication info by email
     */
    @GetMapping("/api/user/user-authentication-info/getByEmail")
    Response<UserAuthenticationInfo> getUserAuthenticationInfoByEmail(@RequestParam("email") String email);

    /**
     * 根据这证件类型和证件号码获取认证信息.
     *
     * @param idType the id type
     * @param idCard the id card
     * @return the user authentication info by id type and card
     */
    @GetMapping("/api/user/user-authentication-info/getByIdTypeAndCard")
    Response<UserAuthenticationInfo> getUserAuthenticationInfoByIdTypeAndCard(@RequestParam("idType") String idType,
                                                                              @RequestParam("idCard") String idCard);

    // =========  end of  用户认证信息 by ranxiangfu   ===========

    /**
     * 根据手机号码查询用户的基本信息和认证信息, 此接口是查询车主端用户的信息.
     *
     * @param mobile the mobile
     * @return the response
     */
    @GetMapping("/api/user/user-full-info/getUserBasicAndAuthenticationInfo")
    Response<UserFullInfoDto> getUserBasicAndAuthenticationInfo(@RequestParam(value = "mobile") String mobile);

    /**
     * 根据手机号码和用户来源查询用户的基本信息和认证信息.
     *
     * @param mobile the mobile
     * @param source the source
     * @return the response
     */
    @GetMapping("/api/user/user-full-info/getUserBasicAndAuthenticationInfoByMobileAndSource")
    Response<UserFullInfoDto> getUserBasicAndAuthenticationInfoByMobileAndSource(
            @RequestParam(value = "mobile") String mobile, @RequestParam(value = "source") String source);

    // =========  start of  用户基本信息 by ranxiangfu   ===========

    /**
     * Save or update user basic info response.
     *
     * @param userBasicInfo the user basic info
     * @return the response
     */
    @PostMapping("/api/user/user-basic-info")
    Response<UserBasicInfo> saveOrUpdateUserBasicInfo(@RequestBody UserBasicInfo userBasicInfo);

    /**
     * Update user basic info by user code response.
     *
     * @param userBasicInfo the user basic info
     * @return the response
     */
    @PostMapping("/api/user/user-basic-info/updateByUserCode")
    Response updateUserBasicInfoByUserCode(@RequestBody UserBasicInfo userBasicInfo);

    /**
     * Gets user basic info by openid.
     *
     * @param openid the openid
     * @return the user basic info by openid
     */
    @GetMapping("/api/user/user-basic-info/getByOpenid")
    Response<UserBasicInfo> getUserBasicInfoByOpenid(@RequestParam("openid") String openid);

    /**
     * Gets user basic info by user code.
     *
     * @param userCode the user code
     * @return the user basic info by user code
     */
    @GetMapping("/api/user/user-basic-info/getByUserCode")
    Response<UserBasicInfo> getUserBasicInfoByUserCode(@RequestParam("userCode") String userCode);

    /**
     * Gets user basic info by mobile.
     *
     * @param mobile the mobile
     * @return the user basic info by mobile
     */
    @GetMapping("/api/user/user-basic-info/getByMobile")
    Response<UserBasicInfo> getUserBasicInfoByMobile(@RequestParam("mobile") String mobile);

    /**
     * Gets user basic info by id type and card.
     *
     * @param idType the id type
     * @param idCard the id card
     * @return the user basic info by id type and card
     */
    @GetMapping("/api/user/user-basic-info/getByIdTypeAndCard")
    Response<UserBasicInfo> getUserBasicInfoByIdTypeAndCard(@RequestParam("idType") String idType,
                                                            @RequestParam("idCard") String idCard);

    /**
     * 根据手机号码和用户来源查询用户基本信息.
     *
     * @param mobile the mobile
     * @param source the source
     * @return the user basic info by mobile
     */
    @GetMapping("/api/user/user-basic-info/getByMobileAndSource")
    Response<UserBasicInfo> getUserBasicInfoByMobileAndSource(@RequestParam("mobile") String mobile,
                                                              @RequestParam("source") String source);

    /**
     * 根据证件类型、证件号码和用户来源查询用户基本信息.
     *
     * @param idType the id type
     * @param idCard the id card
     * @param source the source
     * @return the user basic info by id type and card
     */
    @GetMapping("/api/user/user-basic-info/getByIdTypeCardAndSource")
    Response<UserBasicInfo> getUserBasicInfoBydTypeCardAndSource(@RequestParam("idType") String idType,
                                                                 @RequestParam("idCard") String idCard,
                                                                 @RequestParam("source") String source);

    /**
     * 根据用户编码列表获取用户信息,不建议数量过多.
     *
     * @param userCodeList the user code list
     * @return the user basic info by user code list
     */
    @GetMapping("/api/user/user-basic-info/getByUserCodeList")
    Response<List<UserBasicInfo>> getUserBasicInfoByUserCodeList(@NotNull @NotEmpty @RequestParam("userCodeList") List<String> userCodeList);

    /**
     * 分页查询用户基本信息.
     *
     * @param userCode  用户编码
     * @param openid    wx openid
     * @param wxName    微信昵称
     * @param mobile    手机号码
     * @param status    状态
     * @param source    用户来源
     * @param startTime 注册起始时间
     * @param endTime   注册结束时间
     * @param page      the page
     * @param limit     the limit
     * @return the page table
     */
    @GetMapping("/api/user/user-basic-info/pageQuery")
    PageTable<UserBasicInfo> pageQueryUserBasicInfo(@RequestParam(value = "userCode", required = false) String userCode,
                                                    @RequestParam(value = "openid", required = false) String openid,
                                                    @RequestParam(value = "wxName", required = false) String wxName,
                                                    @RequestParam(value = "mobile", required = false) String mobile,
                                                    @RequestParam(value = "status", required = false) String status,
                                                    @RequestParam(value = "source", required = false) String source,
                                                    @RequestParam(value = "startTime", required = false) String startTime,
                                                    @RequestParam(value = "endTime", required = false) String endTime,
                                                    @RequestParam(value = "page", defaultValue = "1") Integer page,
                                                    @RequestParam(value = "limit", defaultValue = "20") Integer limit);

    // =========  end of  用户基本信息 by ranxiangfu   ===========

    // =========  start of  用户登录凭证信息 by ranxiangfu   ===========

    /**
     * Save or update user login credentials response.
     *
     * @param userRegisterInfo the user register info
     * @return the response
     */
    @PostMapping("/api/user/user-login-credentials")
    Response saveOrUpdateUserLoginCredentials(@RequestBody UserLoginCredentials userRegisterInfo);

    /**
     * Update user login credentials by user code response.
     *
     * @param userLoginCredentials the user login credentials
     * @return the response
     */
    @PostMapping("/api/user/user-login-credentials/updateByUserCode")
    Response updateUserLoginCredentialsByUserCode(@RequestBody UserLoginCredentials userLoginCredentials);

    /**
     * Gets user login credentials by credentials.
     *
     * @param source      the source
     * @param credentials the credentials
     * @param type        the type
     * @return the user login credentials by credentials
     */
    @GetMapping("/api/user/user-login-credentials/getByCredentials")
    Response<UserLoginCredentials> getUserLoginCredentialsByCredentials(@RequestParam("source") String source,
                                                                        @RequestParam("credentials") String credentials,
                                                                        @RequestParam("type") String type);

    /**
     * Gets user login credentials by user code.
     *
     * @param userCode the user code
     * @return the user login credentials by user code
     */
    @GetMapping("/api/user/user-login-credentials/getByUserCode")
    Response<UserLoginCredentials> getUserLoginCredentialsByUserCode(@RequestParam("userCode") String userCode);

    // =========  end of  用户登录凭证信息 by ranxiangfu   ===========

    // =========  start of  用户注册信息 by ranxiangfu   ===========

    /**
     * Save or update user register info response.
     *
     * @param userRegisterInfo the user register info
     * @return the response
     */
    @PostMapping("/api/user/user-register-info")
    Response saveOrUpdateUserRegisterInfo(@RequestBody UserRegisterInfo userRegisterInfo);

    /**
     * Update user register info by user code response.
     *
     * @param userRegisterInfo the user register info
     * @return the response
     */
    @PostMapping("/api/user/user-register-info/updateByUserCode")
    Response updateUserRegisterInfoByUserCode(@RequestBody UserRegisterInfo userRegisterInfo);

    /**
     * Gets user register info by openid.
     *
     * @param openid the openid
     * @return the user register info by openid
     */
    @GetMapping("/api/user/user-register-info/getByOpenid")
    Response<UserRegisterInfo> getUserRegisterInfoByOpenid(@RequestParam("openid") String openid);

    /**
     * Gets user register info by user code.
     *
     * @param userCode the user code
     * @return the user register info by user code
     */
    @GetMapping("/api/user/user-register-info/getByUserCode")
    Response<UserRegisterInfo> getUserRegisterInfoByUserCode(@RequestParam("userCode") String userCode);

    // =========  end of  用户注册信息 by ranxiangfu   ===========

    // =========  start of  用户店铺收藏 by ranxiangfu   ===========

    /**
     * 保存用户店铺收藏.
     *
     * @param userStoreCollection the user store collection
     * @return the response
     */
    @PostMapping("/api/user/user-store-collection")
    Response saveUserStoreCollection(@RequestBody UserStoreCollection userStoreCollection);

    /**
     * 查询用户的收藏店铺信息.
     *
     * @param userCode  the user code
     * @param storeCode the store code
     * @return the response
     */
    @GetMapping("/api/user/user-store-collection/list")
    Response<List<UserStoreCollection>> listUserStoreCollection(@RequestParam(value = "userCode", required = false) String userCode,
                                                                @RequestParam(value = "storeCode", required = false) String storeCode);

    /**
     * 批量删除店铺收藏.
     *
     * @param userCode      the user code
     * @param storeCodeList the store code list
     * @return the response
     */
    @DeleteMapping("/api/user/user-store-collection/batchDelete")
    Response batchDeleteUserStoreCollection(@NotEmpty @RequestParam("userCode") String userCode,
                                            @NotNull @NotEmpty @RequestParam("storeCodeList") List<String> storeCodeList);

    /**
     * 根据id删除店铺收藏.
     *
     * @param id the id
     * @return the response
     */
    @DeleteMapping("/api/user/user-store-collection")
    Response deleteUserStoreCollectionById(@NotNull @RequestParam("id") Integer id);

    // =========  end of  用户店铺收藏 by ranxiangfu   ===========

    // =========  start of  用户登录日志 by ranxiangfu   ===========

    /**
     * 保存用户登录日志.
     *
     * @param userLoginLog the user login log
     * @return the response
     */
    @PostMapping("/api/user/user-login-log")
    Response saveUserLoginLog(@RequestBody UserLoginLog userLoginLog);

    /**
     * 分页查询用户登录日志.
     *
     * @param userCode  the user code
     * @param openid    the openid
     * @param startTime the start time
     * @param endTime   the end time
     * @param page      the page
     * @param limit     the limit
     * @return the page table
     */
    @GetMapping("/api/user/user-login-log/pageQuery")
    PageTable<UserLoginLog> pageQueryUserLoginLog(@RequestParam(value = "userCode", required = false) String userCode,
                                                  @RequestParam(value = "openid", required = false) String openid,
                                                  @RequestParam(value = "startTime", required = false) String startTime,
                                                  @RequestParam(value = "endTime", required = false) String endTime,
                                                  @RequestParam(value = "page", defaultValue = "1") Integer page,
                                                  @RequestParam(value = "limit", defaultValue = "20") Integer limit);

    // =========  end of  用户登录日志 by ranxiangfu   ===========

    //  =========  start of  certificateInfo 模块   ===========

    /**
     * 新增或编辑用户证件信息
     *
     * @param userCertificateInfo 用户证件信息
     * @return response response
     */
    @PostMapping("/api/user/certificateInfo/saveOrUpdateUserCertificateInfo")
    Response saveOrUpdateUserCertificateInfo(@RequestBody UserCertificateInfo userCertificateInfo);

    /**
     * 根据id查询用户证件信息
     *
     * @param id 用户证件信息id
     * @return user certificate info by id
     */
    @GetMapping("/api/user/certificateInfo/{id}")
    Response<UserCertificateInfo> getUserCertificateInfoById(@PathVariable("id") Integer id);

    /**
     * 根据用户code查询用户证件信息
     *
     * @param userCode 用户code
     * @return user certificate info by user code
     */
    @GetMapping("/api/user/certificateInfo/getByUserCode/{userCode}")
    Response<UserCertificateInfo> getUserCertificateInfoByUserCode(@PathVariable("userCode") String userCode);

    /**
     * 根据用户code和证件类型查询用户证件信息
     *
     * @param userCode        用户code
     * @param certificateType 证件类型
     * @return user certificate info by user code and type
     */
    @GetMapping("/api/user/certificateInfo/getByUserCodeAndType")
    Response<UserCertificateInfo> getUserCertificateInfoByUserCodeAndType(@RequestParam("userCode") String userCode, @RequestParam("certificateType") String certificateType);

    /**
     * 根据用户code和证件类型查询不是该类型的用户证件信息
     *
     * @param userCode        用户code
     * @param certificateType 证件类型
     * @return user certificate info by user code and not type
     */
    @GetMapping("/api/user/certificateInfo/getByUserCodeAndNotType")
    Response<List<UserCertificateInfo>> getUserCertificateInfoByUserCodeAndNotType(@RequestParam("userCode") String userCode, @RequestParam("certificateType") String certificateType);

    /**
     * 根据用户code删除用户证件信息
     *
     * @param userCode 用户code
     * @return response response
     */
    @PostMapping("/api/user/certificateInfo/deleteByUserCode/{userCode}")
    Response deleteUserCertificateInfoByUserCode(@PathVariable("userCode") String userCode);

    /**
     * 根据用户id删除用户证件信息
     *
     * @param id 用户id
     * @return response response
     */
    @PostMapping("/api/user/certificateInfo/deleteById/{id}")
    Response deleteUserCertificateInfoById(@PathVariable("id") Integer id);
    // =========  end of  certificateInfo 模块   ===========

    //  =========  start of  UserDriveringLicenceInfo 模块   ===========

    /**
     * 新增或编辑用户行驶证件信息
     *
     * @param userDriveringLicenceInfo 用户行驶证件信息
     * @return response response
     */
    @PostMapping("/api/user/driveringLicenceInfo/saveOrUpdateUserDriveringLicenceInfo")
    Response saveOrUpdateUserDriveringLicenceInfo(@RequestBody UserDriveringLicenceInfo userDriveringLicenceInfo);

    /**
     * 根据id查询用户行驶证件信息
     *
     * @param id 用户行驶证件信息id
     * @return user drivering licence info by id
     */
    @GetMapping("/api/user/driveringLicenceInfo/{id}")
    Response<UserDriveringLicenceInfo> getUserDriveringLicenceInfoById(@PathVariable("id") Integer id);

    /**
     * 根据用户code查询用户行驶证件信息
     *
     * @param userCode 用户code
     * @return user drivering licence info by user code
     */
    @GetMapping("/api/user/driveringLicenceInfo/getByUserCode/{userCode}")
    Response<List<UserDriveringLicenceInfo>> getUserDriveringLicenceInfoByUserCode(@PathVariable("userCode") String userCode);

    /**
     * 根据用户code删除用户行驶证件信息
     *
     * @param userCode 用户code
     * @return response response
     */
    @PostMapping("/api/user/driveringLicenceInfo/deleteByUserCode/{userCode}")
    Response deleteUserDriveringLicenceInfoByUserCode(@PathVariable("userCode") String userCode);

    /**
     * 根据用户id删除用户行驶证件信息
     *
     * @param id 用户id
     * @return response response
     */
    @PostMapping("/api/user/driveringLicenceInfo/deleteById/{id}")
    Response deleteUserDriveringLicenceInfoById(@PathVariable("id") Integer id);
    // =========  end of  UserDriveringLicenceInfo 模块   ===========

    //  =========  start of  UserDriverLicenceInfo 模块   ===========

    /**
     * 新增或编辑用户驾驶证件信息
     *
     * @param userDriverLicenceInfo 用户驾驶证件信息
     * @return response response
     */
    @PostMapping("/api/user/driverLicenceInfo/saveOrUpdateUserDriverLicenceInfo")
    Response saveOrUpdateUserDriverLicenceInfo(@RequestBody UserDriverLicenceInfo userDriverLicenceInfo);

    /**
     * 根据id查询用户驾驶证件信息
     *
     * @param id 用户驾驶证件信息id
     * @return user driver licence info by id
     */
    @GetMapping("/api/user/driverLicenceInfo/{id}")
    Response<UserDriverLicenceInfo> getUserDriverLicenceInfoById(@PathVariable("id") Integer id);

    /**
     * 根据用户code查询用户驾驶证件信息
     *
     * @param userCode 用户code
     * @return user driver licence info by user code
     */
    @GetMapping("/api/user/driverLicenceInfo/getByUserCode/{userCode}")
    Response<UserDriverLicenceInfo> getUserDriverLicenceInfoByUserCode(@PathVariable("userCode") String userCode);

    /**
     * 根据用户code删除用户驾驶证件信息
     *
     * @param userCode 用户code
     * @return response response
     */
    @PostMapping("/api/user/driverLicenceInfo/deleteByUserCode/{userCode}")
    Response deleteUserDriverLicenceInfoByUserCode(@PathVariable("userCode") String userCode);

    /**
     * 根据用户id删除用户驾驶证件信息
     *
     * @param id 用户id
     * @return response response
     */
    @PostMapping("/api/user/driverLicenceInfo/deleteById/{id}")
    Response deleteUserDriverLicenceInfoById(@PathVariable("id") Integer id);
    // =========  end of  UserDriverLicenceInfo 模块   ===========

    //  =========  start of  UserCarInformation 模块   ===========

    /**
     * 用户车辆信息更新和保存
     *
     * @param userCarInformation the user car information
     * @return response response
     */
    @PostMapping("/api/user/user-car-information")
    Response saveOrUpdateUserCarInformation(@RequestBody UserCarInformation userCarInformation);

    /**
     * 用户车辆列表
     *
     * @param code the code
     * @return response response
     */
    @GetMapping("/api/user/user-car-information/findUserCarInfomation")
    Response<List<UserCarInformation>> findUserCarInfomation(@RequestParam(name = "code") String code);

    /**
     * 根据编码获取车辆信息
     *
     * @param code the code
     * @return user car infomation by code
     */
    @GetMapping("/api/user/user-car-information/getUserCarInfomationByCode")
    Response<UserCarInformation> getUserCarInfomationByCode(@RequestParam("code") String code);

    /**
     * 根据车辆code,用户编号删除车辆信息
     *
     * @param code     the code
     * @param userCode the user code
     * @return response response
     */
    @PostMapping("/api/user/user-car-information/deleteMyCarInfor")
    Response deleteMyCarInfor(@RequestParam(name = "code") String code,@RequestParam(name = "userCode") String userCode);

    /**
     * 新增用户车辆信息(比较用户编号,车牌号,车辆型号,车辆后6位和发动机代号后六位)
     *
     * @param userCarInformation the user car information
     * @return response
     */
    @PostMapping("/api/user/user-car-information/saveUserCarInformation")
    Response saveUserCarInformation(@RequestBody UserCarInformation userCarInformation);

    /**
     * 删除车牌信息，匹配各输入条件和用户号
     *
     * @param userCarInformation the user car information
     * @return response
     */
    @PostMapping("/api/user/user-car-information/deleteCarInfo")
    Response deleteCarInfoWithSelfConditions(@RequestBody UserCarInformation userCarInformation);
    // =========  end of  UserCarInformation 模块   ===========

    // =========  start of  小程序推荐码扫描记录 by ranxiangfu   ===========

    /**
     * 根据openid获取最新一条小程序推荐码扫描记录信息.
     *
     * @param openid the openid
     * @return the response
     */
    @GetMapping("/api/user/recommend-code-scanning-record/getLatestByWxOpenid")
    Response<RecommendCodeScanningRecord> getLatestRecommendCodeScanningRecordByWxOpenid(@RequestParam("openid") String openid);

    /**
     * 保存小程序推荐码扫描记录.
     *
     * @param scanningRecord the scanning record
     * @return the response
     */
    @PostMapping("/api/user/recommend-code-scanning-record")
    Response saveOrUpdateRecommendCodeScanningRecord(@RequestBody RecommendCodeScanningRecord scanningRecord);

    /**
     * 更新小程序推荐码扫描记录的手机号信息.
     *
     * @param scanningRecord the scanning record
     * @return the response
     */
    @PostMapping("/api/user/recommend-code-scanning-record/updateMobileByUserCode")
    Response updateMobileOfRecommendCodeScanningRecordByUserCode(@RequestBody RecommendCodeScanningRecord scanningRecord);

    /**
     * 更新最新一条小程序推荐码扫描记录的注册信息.
     *
     * @param scanningRecord the scanning record
     * @return the response
     */
    @PostMapping("/api/user/recommend-code-scanning-record/updateLatestByWxOpenid")
    Response<RecommendCodeScanningRecord> updateLatestRecommendCodeScanningRecordByWxOpenid(@RequestBody RecommendCodeScanningRecord scanningRecord);

    /**
     * 分页查询小程序推荐码扫描记录.
     *
     * @param userCode      the user code
     * @param openid        the openid
     * @param mobile        the mobile
     * @param recommendCode the recommend code
     * @param startTime     the start time
     * @param endTime       the end time
     * @param page          the page
     * @param limit         the limit
     * @return the page table
     */
    @GetMapping("/api/user/recommend-code-scanning-record/pageQuery")
    PageTable<RecommendCodeScanningRecord> pageQueryRecommendCodeScanningRecord(@RequestParam(value = "userCode", required = false) String userCode,
                                                                                @RequestParam(value = "openid", required = false) String openid,
                                                                                @RequestParam(value = "mobile", required = false) String mobile,
                                                                                @RequestParam(value = "recommendCode", required = false) String recommendCode,
                                                                                @RequestParam(value = "startTime", required = false) String startTime,
                                                                                @RequestParam(value = "endTime", required = false) String endTime,
                                                                                @RequestParam(value = "page", defaultValue = "1") Integer page,
                                                                                @RequestParam(value = "limit", defaultValue = "20") Integer limit);

    /**
     * 根据推荐码查询是否有扫码记录，有扫码记录时返回true，没有则返回false.
     *
     * @param recommendCode the recommend code
     * @return the response
     */
    @GetMapping("/api/user/recommend-code-scanning-record/checkHasScanningRecord")
    Response<Boolean> checkHasRecommendCodeScanningRecord(@RequestParam(value = "recommendCode") String recommendCode);

    /**
     * 分页查询注册成功的小程序推荐码扫描记录.
     *
     * @param recommendCode the recommend code
     * @param startTime     the start time
     * @param endTime       the end time
     * @param page          the page
     * @param limit         the limit
     * @return the page table
     */
    @GetMapping("/api/user/recommend-code-scanning-record/pageQuerySuccRegisterRecord")
    PageTable<RecommendCodeScanningRecord> pageQuerySuccRegisterRecord(@RequestParam(value = "recommendCode") String recommendCode,
                                                                       @RequestParam(value = "startTime", required = false) String startTime,
                                                                       @RequestParam(value = "endTime", required = false) String endTime,
                                                                       @RequestParam(value = "page", defaultValue = "1") Integer page,
                                                                       @RequestParam(value = "limit", defaultValue = "20") Integer limit);

    /**
     * 根据推荐码统计注册成功的扫码记录数目.
     *
     * @param recommendCode the recommend code
     * @param startTime     the start time
     * @param endTime       the end time
     * @return the page table
     */
    @GetMapping("/api/user/recommend-code-scanning-record/countSuccRegisterRecord")
    Response<Integer> countSuccRegisterRecord(@RequestParam(value = "recommendCode") String recommendCode,
                                              @RequestParam(value = "startTime", required = false) String startTime,
                                              @RequestParam(value = "endTime", required = false) String endTime);
    // =========  end of  小程序推荐码扫描记录 by ranxiangfu   ===========

    // =========  start of 用户模板消息   ===========

    /**
     * 保存模板消息
     *
     * @param userMsg the user msg
     * @return Response response
     */
    @PostMapping("/api/user/msg/saveUserMsg")
    Response saveUserMsg(@RequestBody UserMsg userMsg);

    /**
     * 获取用户消息列表，不分页
     *
     * @param userCode the user code
     * @return response
     */
    @GetMapping("/api/user/msg/listUserMsg/{userCode}")
    Response<List<UserMsg>> findUserMsgList(@PathVariable(value = "userCode") String userCode);

    /**
     * 获取用户消息列表，分页
     *
     * @param userCode the user code
     * @param limit    the limit
     * @param page     the page
     * @return page table
     */
    @GetMapping("/api/user/msg/pageUserMsgList/{userCode}")
    PageTable<UserMsg> pageUserMsgList(@PathVariable(value = "userCode") String userCode,
                                       @RequestParam(value = "limit", defaultValue = "20") int limit,
                                       @RequestParam(value = "page", defaultValue = "1") int page);

    /**
     * 阅读消息 通过code
     *
     * @param code the code
     * @return response
     */
    @GetMapping("/api/user/msg/readUserMsgByCode/{code}")
    Response<UserMsg> readUserMsgByCode(@PathVariable(value = "code") String code);

    /**
     * 清空消息（不传消息类型，消息全部清空；传消息类型，清空所有的消息类型）
     * 清空消息的同时，需要更新，相关的已读未读清空
     *
     * @param userCode 用户Code
     * @param msgType  消息类型（非必填）
     * @return response
     */
    @GetMapping("/api/user/msg/emptyUserMsgByUserCode/{userCode}")
    Response emptyUserMsgByUserCode(@PathVariable("userCode") String userCode,
                                    @RequestParam(value = "msgType", required = false) String msgType);

    /**
     * web端, 用户消息列表
     *
     * @param userCode   the user code
     * @param code       the code
     * @param senderCode the sender code
     * @param bindMobile the bind mobile
     * @param msgType    the msg type
     * @param hasRead    the has read
     * @param delStatus  the del status
     * @param startTime  the start time
     * @param endTime    the end time
     * @param page       the page
     * @param limit      the limit
     * @return page table
     */
    @GetMapping("/api/user/msg/pageWebQueryPageUserMsg")
    PageTable<UserMsgWebDto> pageWebQueryPageUserMsg(@RequestParam(value = "userCode", required = false) String userCode,
                                                     @RequestParam(value = "code", required = false) String code,
                                                     @RequestParam(value = "senderCode", required = false) String senderCode,
                                                     @RequestParam(value = "bindMobile", required = false) String bindMobile,
                                                     @RequestParam(value = "msgType", required = false) String msgType,
                                                     @RequestParam(value = "hasRead", required = false) Integer hasRead,
                                                     @RequestParam(value = "delStatus", required = false) Integer delStatus,
                                                     @RequestParam(value = "startTime", required = false) String startTime,
                                                     @RequestParam(value = "endTime", required = false) String endTime,
                                                     @RequestParam(value = "page", defaultValue = "1") Integer page,
                                                     @RequestParam(value = "limit", defaultValue = "20") Integer limit);
    // =========  end of  用户模板消息   ===========

    // =========  start of  用户收货地址   ===========

    /**
     * List user delivery address response.
     *
     * @param userCode the user code
     * @param deleted  the deleted
     * @return the response
     */
    @GetMapping("/api/user/user-delivery-address/list")
    Response<List<UserDeliveryAddress>> listUserDeliveryAddress(@RequestParam("userCode") String userCode,
                                                                @RequestParam(value = "deleted", required = false) Boolean deleted);

    /**
     * Save or update user delivery address response.
     *
     * @param userDeliveryAddress the user delivery address
     * @return the response
     */
    @PostMapping("/api/user/user-delivery-address")
    Response<Integer> saveOrUpdateUserDeliveryAddress(@RequestBody UserDeliveryAddress userDeliveryAddress);

    /**
     * Gets user delivery address by id.
     *
     * @param id the id
     * @return the user delivery address by id
     */
    @GetMapping("/api/user/user-delivery-address")
    Response<UserDeliveryAddress> getUserDeliveryAddressById(@RequestParam("id") Integer id);

    /**
     * Gets one user delivery address.
     *
     * @param userCode the user code
     * @return the one user delivery address
     */
    @GetMapping("/api/user/user-delivery-address/getOne")
    Response<UserDeliveryAddress> getOneUserDeliveryAddress(@RequestParam("userCode") String userCode);

    // =========  end of  用户收货地址   ===========

    // =========  start of  用户身份   ===========

    /**
     * Save or update user identity response.
     *
     * @param userIdentity the user identity
     * @return the response
     */
    @PostMapping("/api/user/user-identity")
    Response saveOrUpdateUserIdentity(@RequestBody UserIdentity userIdentity);

    /**
     * List user identity response.
     *
     * @param userCode         the user code
     * @param openid           the openid
     * @param mobile           the mobile
     * @param identityTypeCode the identity type code
     * @return the response
     */
    @GetMapping("/api/user/user-identity/list")
    Response<List<UserIdentity>> listUserIdentity(@RequestParam(value = "userCode", required = false) String userCode,
                                                  @RequestParam(value = "openid", required = false) String openid,
                                                  @RequestParam(value = "mobile", required = false) String mobile,
                                                  @RequestParam(value = "status", required = false) String status,
                                                  @RequestParam(value = "identityTypeCode", required = false) String identityTypeCode);

    /**
     * 有多个结果的时候只返回一个, 精确查找时建议参数都传.
     *
     * @param userCode         the user code
     * @param status           the status
     * @param identitySource   the identity source
     * @param identityTypeCode the identity type code
     * @param identityCode     the identity code
     * @return the user identity
     */
    @GetMapping("/api/user/user-identity/getOne")
    Response<UserIdentity> getUserIdentity(@RequestParam(value = "userCode") String userCode,
                                           @RequestParam(value = "status", required = false) String status,
                                           @RequestParam(value = "identitySource", required = false) String identitySource,
                                           @RequestParam(value = "identityTypeCode", required = false) String identityTypeCode,
                                           @RequestParam(value = "identityCode") String identityCode);
    // =========  end of  用户身份   ===========

    // =========  start of  代理人身份认证   ===========
    /**
     * Save or update user agent identity auth response.
     *
     * @param userAgentIdentityAuth the user agent identity auth
     * @return the response
     */
    @PostMapping("/api/user/user-agent-identity-auth")
    Response saveOrUpdateUserAgentIdentityAuth(@RequestBody UserAgentIdentityAuth userAgentIdentityAuth);

    /**
     * Gets last user agent identity auth.
     *
     * @param userCode the user code
     * @return the last user agent identity auth
     */
    @GetMapping("/api/user/user-agent-identity-auth/getLast")
    Response<UserAgentIdentityAuth> getLastUserAgentIdentityAuth(@RequestParam(value = "userCode", required = false) String userCode);

    /**
     * Page query user agent identity auth page table.
     *
     * @param userCode   the user code
     * @param userOpenid the user openid
     * @param userMobile the user mobile
     * @param name       the name
     * @param authWay    the auth way
     * @param status     the status
     * @param startTime  the start time
     * @param endTime    the end time
     * @param page       the page
     * @param limit      the limit
     * @return the page table
     */
    @GetMapping("/api/user/user-agent-identity-auth/pageQuery")
    PageTable<UserAgentIdentityAuth> pageQueryUserAgentIdentityAuth(@RequestParam(value = "userCode", required = false) String userCode,
                                                                    @RequestParam(value = "userOpenid", required = false) String userOpenid,
                                                                    @RequestParam(value = "userMobile", required = false) String userMobile,
                                                                    @RequestParam(value = "name", required = false) String name,
                                                                    @RequestParam(value = "authWay", required = false) String authWay,
                                                                    @RequestParam(value = "status", required = false) String status,
                                                                    @RequestParam(value = "startTime", required = false) String startTime,
                                                                    @RequestParam(value = "endTime", required = false) String endTime,
                                                                    @RequestParam(value = "page", defaultValue = "1") Integer page,
                                                                    @RequestParam(value = "limit", defaultValue = "20") Integer limit);

    /**
     * 导出代理人身份
     *
     * @param userCode
     * @param userOpenid
     * @param userMobile
     * @param name
     * @param authWay
     * @param status
     * @param startTime
     * @param endTime
     * @return
     */
    @GetMapping("/api/user/user-agent-identity-auth/exportUserAgentIdentityAuth")
    Response<List<UserAgentIdentityAuth>> exportUserAgentIdentityAuth(@RequestParam(value = "userCode", required = false) String userCode,
                                                            @RequestParam(value = "userOpenid", required = false) String userOpenid,
                                                            @RequestParam(value = "userMobile", required = false) String userMobile,
                                                            @RequestParam(value = "name", required = false) String name,
                                                            @RequestParam(value = "authWay", required = false) String authWay,
                                                            @RequestParam(value = "status", required = false) String status,
                                                            @RequestParam(value = "startTime", required = false) String startTime,
                                                            @RequestParam(value = "endTime", required = false) String endTime);
    // =========  end of  代理人身份认证   ===========

    // ========= liangyongtong start of  邀请人关系   ===========
    /**
     * 保存初始的邀请关系
     * @param userAgentInvite
     * @return
     */
    @PostMapping("/api/user/agent/user-agent-invite/saveInitInvite")
    Response saveInvite(@RequestBody UserAgentInvite userAgentInvite);

    /**
     * 更新被邀请人信息
     */
    @PostMapping("/api/user/agent/user-agent-invite/updateBeInvite")
    Response updateBeInviteUserInfo(@RequestBody AgentInviteUpdateInfoDto updateInfo);

    /**
     * 查询各性质的数量
     * @param userCode 邀请人用户编码
     * @return
     */
    @GetMapping("/api/user/agent/user-agent-invite/nature/count")
    Response<AgentNatureCountDto> agentInviteNatureCount(@RequestParam(value = "userCode") String userCode);

    /**
     * 我的邀请汇总
     * @param userCode 会员编码
     * @return
     */
    @GetMapping("/api/user/agent/user-agent-invite/myInviteSummary")
    Response<AgentInviteIndexDto> myInviteSummary(@RequestParam(value = "userCode") String userCode);

    /**
     * 我的邀请列表
     * @param userCode 邀请人用户编码
     * @param nature 性质, 默认注册
     * @param page 当前页
     * @param limit 每页条数
     * @return
     */
    @GetMapping("/api/user/agent/user-agent-invite/myInviteListForPage")
    PageTable<UserAgentInvite> registerList(@RequestParam(value = "userCode") String userCode,
                                            @RequestParam(value = "keyword") String keyword,
                                                   @RequestParam(value = "nature", defaultValue = UserConstant.UserAgentInviteNature.REGISTER) String nature,
                                                   @RequestParam(value = "page", defaultValue = "1") Integer page,
                                                   @RequestParam(value = "limit", defaultValue = "20") Integer limit);

    /**
     * 我的团队情况
     * @param userCode 邀请人编码
     * @return
     */
    @GetMapping("/api/user/agent/user-agent-invite/myteam/info")
    Response<MyTeamDto> agentMyTeam(@RequestParam(value = "userCode") String userCode);

    /**
     * 我的排行情况
     * @param userCode 邀请人用户编码
     * @param startTime 注册开始时间
     * @param endTime 注册结束时间
     * @return
     */
    @GetMapping("/api/user/agent/user-agent-invite/ranking/info")
    Response<AgentInviteRankingDto> rankingInfo(@RequestParam(value = "userCode") String userCode,
                                                       @RequestParam(value = "startTime", required = false) String startTime,
                                                       @RequestParam(value = "endTime", required = false) String endTime);

    /**
     * 排行榜
     * @param startTime 开始时间
     * @param endTime 结束时间
     * @param page 页数
     * @param limit 每页条数
     * @return
     */
    @GetMapping("/api/user/agent/user-agent-invite/rankingList")
    PageTable<UserAgentInvite> rankingList(@RequestParam(value = "startTime", required = false) String startTime,
                                                  @RequestParam(value = "endTime", required = false) String endTime,
                                                  @RequestParam(value = "page", defaultValue = "1") Integer page,
                                                  @RequestParam(value = "limit", defaultValue = "20") Integer limit);

    /**
     * 根据被邀请人用户编码查询邀请关系
     * @param beInviteUserCode
     * @return
     */
    @GetMapping("/api/user/agent/user-agent-invite/inviteByBeInviteUserCode")
    Response<UserAgentInvite> getAgentInviteByBeInviteUserCode(@RequestParam(value = "beInviteUserCode") String beInviteUserCode);

    /**
     * 被邀请人详情
     * @param param
     * @return
     */
    @PostMapping("/api/user/agent/user-agent-invite/agent/detail")
    Response<AgentDetailDto> getAgentDetail(@RequestBody ReqAgentParam param);
    // =========  end of  邀请人关系   ===========

    // ========= luoyc start of  推广码   ===========
    /**
     * 创建推广申请 - 待审核
     * @param apply
     * @return
     */
    @PostMapping("/api/user/popularize/apply/saveApply")
    Response<UserPopularizeApply> saveApply(@RequestBody UserPopularizeApply apply);

    /**
     * 更新推广申请 - 通过/不通过
     * @param applyDTO
     * @return
     */
    @PostMapping("/api/user/popularize/apply/updateApply")
    Response updateApply(@RequestBody UserPopularizeApplyDTO applyDTO);


    /**
     * 分页查询推广信息
     *
     * @param page
     * @param limit
     * @return
     */
    @GetMapping("/api/user/popularize/info/pageQuery")
    PageTable<UserPopularizeInfo> pageQueryUserPopularizeInfo(
                                                              @RequestParam(value = "page", defaultValue = "1") Integer page,
                                                              @RequestParam(value = "limit", defaultValue = "20") Integer limit);

    /**
     * 创建推广信息
     *
     * @param info
     * @return
     */
    @PostMapping("/api/user/popularize/info/createInfo")
    Response createInfo(@RequestBody UserPopularizeInfo info);
    // ========= luoyc end of  推广码   ===========
}
