package com.rocogz.syy.user.client;

import com.rocogz.syy.common.PageTable;
import com.rocogz.syy.common.response.Response;
import com.rocogz.syy.user.dto.UserFullInfoDto;
import com.rocogz.syy.user.entity.*;
import com.rocogz.syy.user.entity.certificate.UserCertificateInfo;
import com.rocogz.syy.user.entity.certificate.UserDriverLicenceInfo;
import com.rocogz.syy.user.entity.certificate.UserDriveringLicenceInfo;
import org.springframework.web.bind.annotation.*;

import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import java.time.LocalDateTime;
import java.util.List;

/**
 * The interface User client service.
 *
 * @author：liangyongtong
 * @since：2020/3/25
 */
public interface IUserClientService {

    /**
     * Save or update user authentication info response.
     *
     * @param userAuthenticationInfo the user authentication info
     * @return the response
     */
    @PostMapping("/api/user/user-authentication-info")
    Response saveOrUpdateUserAuthenticationInfo(@RequestBody UserAuthenticationInfo userAuthenticationInfo);

    /**
     * Update user authentication info by user code response.
     *
     * @param userAuthenticationInfo the user authentication info
     * @return the response
     */
    @PostMapping("/api/user/user-authentication-info/updateByUserCode")
    Response updateUserAuthenticationInfoByUserCode(@RequestBody UserAuthenticationInfo userAuthenticationInfo);

    /**
     * Gets user authentication info by user code.
     *
     * @param userCode the user code
     * @return the user authentication info by user code
     */
    @GetMapping("/api/user/user-authentication-info/getByUserCode")
    Response<UserAuthenticationInfo> getUserAuthenticationInfoByUserCode(@RequestParam("userCode") String userCode);

    /**
     * 根据手机号获取认证信息.
     *
     * @param mobile the mobile
     * @return the user authentication info by mobile
     */
    @GetMapping("/api/user/user-authentication-info/getByMobile")
    Response<UserAuthenticationInfo> getUserAuthenticationInfoByMobile(@RequestParam("mobile") String mobile);

    /**
     * 根据邮箱获取认证信息.
     *
     * @param email the email
     * @return the user authentication info by email
     */
    @GetMapping("/api/user/user-authentication-info/getByEmail")
    Response<UserAuthenticationInfo> getUserAuthenticationInfoByEmail(@RequestParam("email") String email);

    /**
     * 根据这证件类型和证件号码获取认证信息.
     *
     * @param idType the id type
     * @param idCard the id card
     * @return the user authentication info by id type and card
     */
    @GetMapping("/api/user/user-authentication-info/getByIdTypeAndCard")
    Response<UserAuthenticationInfo> getUserAuthenticationInfoByIdTypeAndCard(@RequestParam("idType") String idType,
                                                                              @RequestParam("idCard") String idCard);

    /**
     * Save or update user basic info response.
     *
     * @param userBasicInfo the user basic info
     * @return the response
     */
    @PostMapping("/api/user/user-basic-info")
    Response<UserBasicInfo> saveOrUpdateUserBasicInfo(@RequestBody UserBasicInfo userBasicInfo);

    /**
     * Update user basic info by user code response.
     *
     * @param userBasicInfo the user basic info
     * @return the response
     */
    @PostMapping("/api/user/user-basic-info/updateByUserCode")
    Response updateUserBasicInfoByUserCode(@RequestBody UserBasicInfo userBasicInfo);

    /**
     * Gets user basic info by openid.
     *
     * @param openid the openid
     * @return the user basic info by openid
     */
    @GetMapping("/api/user/user-basic-info/getByOpenid")
    Response<UserBasicInfo> getUserBasicInfoByOpenid(@RequestParam("openid") String openid);

    /**
     * Gets user basic info by user code.
     *
     * @param userCode the user code
     * @return the user basic info by user code
     */
    @GetMapping("/api/user/user-basic-info/getByUserCode")
    Response<UserBasicInfo> getUserBasicInfoByUserCode(@RequestParam("userCode") String userCode);

    /**
     * 根据用户编码列表获取用户信息,不建议数量过多.
     *
     * @param userCodeList the user code list
     * @return the user basic info by user code list
     */
    @GetMapping("/api/user/user-basic-info/getByUserCodeList")
    Response<List<UserBasicInfo>> getUserBasicInfoByUserCodeList(@NotNull @NotEmpty @RequestParam("userCodeList") List<String> userCodeList);

    /**
     * 分页查询用户基本信息.
     *
     * @param userCode  用户编码
     * @param openid    wx openid
     * @param wxName    微信昵称
     * @param mobile    手机号码
     * @param status    状态
     * @param startTime 注册起始时间
     * @param endTime   注册结束时间
     * @param page      the page
     * @param limit     the limit
     * @return the page table
     */
    @GetMapping("/api/user/user-basic-info/pageQuery")
    PageTable<UserBasicInfo> pageQueryUserBasicInfo(@RequestParam(value = "userCode", required = false) String userCode,
                                                    @RequestParam(value = "openid", required = false) String openid,
                                                    @RequestParam(value = "wxName", required = false) String wxName,
                                                    @RequestParam(value = "mobile", required = false) String mobile,
                                                    @RequestParam(value = "status", required = false) String status,
                                                    @RequestParam(value = "startTime", required = false) String startTime,
                                                    @RequestParam(value = "endTime", required = false) String endTime,
                                                    @RequestParam(value = "page", defaultValue = "1") Integer page,
                                                    @RequestParam(value = "limit", defaultValue = "20") Integer limit);

    /**
     * Save or update user login credentials response.
     *
     * @param userRegisterInfo the user register info
     * @return the response
     */
    @PostMapping("/api/user/user-login-credentials")
    Response saveOrUpdateUserLoginCredentials(@RequestBody UserLoginCredentials userRegisterInfo);

    /**
     * Update user login credentials by user code response.
     *
     * @param userLoginCredentials the user login credentials
     * @return the response
     */
    @PostMapping("/api/user/user-login-credentials/updateByUserCode")
    Response updateUserLoginCredentialsByUserCode(@RequestBody UserLoginCredentials userLoginCredentials);

    /**
     * Gets user login credentials by credentials.
     *
     * @param credentials the credentials
     * @param type        the type
     * @return the user login credentials by credentials
     */
    @GetMapping("/api/user/user-login-credentials/getByCredentials")
    Response<UserLoginCredentials> getUserLoginCredentialsByCredentials(@RequestParam("credentials") String credentials,
                                                                               @RequestParam("type") String type);

    /**
     * Gets user login credentials by user code.
     *
     * @param userCode the user code
     * @return the user login credentials by user code
     */
    @GetMapping("/api/user/user-login-credentials/getByUserCode")
    Response<UserLoginCredentials> getUserLoginCredentialsByUserCode(@RequestParam("userCode") String userCode);

    /**
     * Save or update user register info response.
     *
     * @param userRegisterInfo the user register info
     * @return the response
     */
    @PostMapping("/api/user/user-register-info")
    Response saveOrUpdateUserRegisterInfo(@RequestBody UserRegisterInfo userRegisterInfo);

    /**
     * Update user register info by user code response.
     *
     * @param userRegisterInfo the user register info
     * @return the response
     */
    @PostMapping("/api/user/user-register-info/updateByUserCode")
    Response updateUserRegisterInfoByUserCode(@RequestBody UserRegisterInfo userRegisterInfo);

    /**
     * Gets user register info by openid.
     *
     * @param openid the openid
     * @return the user register info by openid
     */
    @GetMapping("/api/user/user-register-info/getByOpenid")
    Response<UserRegisterInfo> getUserRegisterInfoByOpenid(@RequestParam("openid") String openid);

    /**
     * Gets user register info by user code.
     *
     * @param userCode the user code
     * @return the user register info by user code
     */
    @GetMapping("/api/user/user-register-info/getByUserCode")
    Response<UserRegisterInfo> getUserRegisterInfoByUserCode(@RequestParam("userCode") String userCode);

    /**
     * 保存用户店铺收藏.
     *
     * @param userStoreCollection the user store collection
     * @return the response
     */
    @PostMapping("/api/user/user-store-collection")
    Response saveUserStoreCollection(@RequestBody UserStoreCollection userStoreCollection);

    /**
     * 查询用户的收藏店铺信息.
     *
     * @param userCode  the user code
     * @param storeCode the store code
     * @return the response
     */
    @GetMapping("/api/user/user-store-collection/list")
    Response<List<UserStoreCollection>> listUserStoreCollection(@RequestParam(value = "userCode", required = false) String userCode,
                                                                @RequestParam(value = "storeCode", required = false) String storeCode);

    /**
     * 批量删除店铺收藏.
     *
     * @param userCode      the user code
     * @param storeCodeList the store code list
     * @return the response
     */
    @DeleteMapping("/api/user/user-store-collection/batchDelete")
    Response batchDeleteUserStoreCollection(@NotEmpty @RequestParam("userCode") String userCode,
                                            @NotNull @NotEmpty @RequestParam("storeCodeList") List<String> storeCodeList);

    /**
     * 根据id删除店铺收藏.
     *
     * @param id the id
     * @return the response
     */
    @DeleteMapping("/api/user/user-store-collection")
    Response deleteUserStoreCollectionById(@NotNull @RequestParam("id") Integer id);

    /**
     * 保存用户登录日志.
     *
     * @param userLoginLog the user login log
     * @return the response
     */
    @PostMapping("/api/user/user-login-log")
    Response saveUserLoginLog(@RequestBody UserLoginLog userLoginLog);

    /**
     * 分页查询用户登录日志.
     *
     * @param userCode  the user code
     * @param openid    the openid
     * @param startTime the start time
     * @param endTime   the end time
     * @param page      the page
     * @param limit     the limit
     * @return the page table
     */
    @GetMapping("/api/user/user-login-log/pageQuery")
    PageTable<UserLoginLog> pageQueryUserLoginLog(@RequestParam(value = "userCode", required = false) String userCode,
                                                  @RequestParam(value = "openid", required = false) String openid,
                                                  @RequestParam(value = "startTime", required = false) String startTime,
                                                  @RequestParam(value = "endTime", required = false) String endTime,
                                                  @RequestParam(value = "page", defaultValue = "1") Integer page,
                                                  @RequestParam(value = "limit", defaultValue = "20") Integer limit);

    /**
     * 根据手机号码查询用户的基本信息和认证信息.
     *
     * @param mobile the mobile
     * @return the response
     */
    @GetMapping("/api/user/user-full-info/getUserBasicAndAuthenticationInfo")
    Response<UserFullInfoDto> getUserBasicAndAuthenticationInfo(@RequestParam(value = "mobile") String mobile);

    //  =========  start of  cleanSystemMessage 模块   ===========
    /**
     * 记录用户清空系统消息时间
     *
     * @param userCode 用户编码
     * @param now      当前时间
     * @return
     */
    @GetMapping("/api/user/cleanSystemMessage/cleanSystemMessageByUserCode")
    Response cleanSystemMessageByUserCode(@RequestParam("userCode") String userCode, @RequestParam("now") LocalDateTime now);

    /**
     * 根据用户编码获取用户清空系统消息时间
     *
     * @param userCode 用户编码
     * @return
     */
    @GetMapping("/api/user/cleanSystemMessage/getCleanSystemMessageByUserCode")
    Response<CleanSystemMessage> getCleanSystemMessageByUserCode(@RequestParam("userCode") String userCode);

    // =========  end of  cleanSystemMessage 模块   ===========

    //  =========  start of  certificateInfo 模块   ===========
    /**
     * 新增或编辑用户证件信息
     *
     * @param userCertificateInfo 用户证件信息
     * @return
     */
    @PostMapping("/api/user/certificateInfo/saveOrUpdateUserCertificateInfo")
    Response saveOrUpdateUserCertificateInfo(@RequestBody UserCertificateInfo userCertificateInfo);

    /**
     * 根据id查询用户证件信息
     *
     * @param id 用户证件信息id
     * @return
     */
    @GetMapping("/api/user/certificateInfo/{id}")
    Response<UserCertificateInfo> getUserCertificateInfoById(@PathVariable("id") Integer id);

    /**
     * 根据用户code查询用户证件信息
     *
     * @param userCode 用户code
     * @return
     */
    @GetMapping("/api/user/certificateInfo/getByUserCode/{userCode}")
    Response<UserCertificateInfo> getUserCertificateInfoByUserCode(@PathVariable("userCode") String userCode);

    /**
     * 根据用户code和证件类型查询用户证件信息
     *
     * @param userCode 用户code
     * @param certificateType 证件类型
     * @return
     */
    @GetMapping("/api/user/certificateInfo/getByUserCodeAndType")
    Response<UserCertificateInfo> getUserCertificateInfoByUserCodeAndType(@RequestParam("userCode") String userCode, @RequestParam("certificateType") String certificateType);

    /**
     * 根据用户code和证件类型查询不是该类型的用户证件信息
     *
     * @param userCode 用户code
     * @param certificateType 证件类型
     * @return
     */
    @GetMapping("/api/user/certificateInfo/getByUserCodeAndNotType")
    Response<List<UserCertificateInfo>> getUserCertificateInfoByUserCodeAndNotType(@RequestParam("userCode") String userCode, @RequestParam("certificateType") String certificateType);

    /**
     * 根据用户code删除用户证件信息
     *
     * @param userCode 用户code
     * @return
     */
    @PostMapping("/api/user/certificateInfo/deleteByUserCode/{userCode}")
    Response deleteUserCertificateInfoByUserCode(@PathVariable("userCode") String userCode);

    /**
     * 根据用户id删除用户证件信息
     *
     * @param id 用户id
     * @return
     */
    @PostMapping("/api/user/certificateInfo/deleteById/{id}")
    Response deleteUserCertificateInfoById(@PathVariable("id") Integer id);
    // =========  end of  certificateInfo 模块   ===========

    //  =========  start of  UserDriveringLicenceInfo 模块   ===========
    /**
     * 新增或编辑用户行驶证件信息
     *
     * @param userDriveringLicenceInfo 用户行驶证件信息
     * @return
     */
    @PostMapping("/api/user/driveringLicenceInfo/saveOrUpdateUserDriveringLicenceInfo")
    Response saveOrUpdateUserDriveringLicenceInfo(@RequestBody UserDriveringLicenceInfo userDriveringLicenceInfo);

    /**
     * 根据id查询用户行驶证件信息
     *
     * @param id 用户行驶证件信息id
     * @return
     */
    @GetMapping("/api/user/driveringLicenceInfo/{id}")
    Response<UserDriveringLicenceInfo> getUserDriveringLicenceInfoById(@PathVariable("id") Integer id);

    /**
     * 根据用户code查询用户行驶证件信息
     *
     * @param userCode 用户code
     * @return
     */
    @GetMapping("/api/user/driveringLicenceInfo/getByUserCode/{userCode}")
    Response<List<UserDriveringLicenceInfo>> getUserDriveringLicenceInfoByUserCode(@PathVariable("userCode") String userCode);

    /**
     * 根据用户code删除用户行驶证件信息
     *
     * @param userCode 用户code
     * @return
     */
    @PostMapping("/api/user/driveringLicenceInfo/deleteByUserCode/{userCode}")
    Response deleteUserDriveringLicenceInfoByUserCode(@PathVariable("userCode") String userCode);

    /**
     * 根据用户id删除用户行驶证件信息
     *
     * @param id 用户id
     * @return
     */
    @PostMapping("/api/user/driveringLicenceInfo/deleteById/{id}")
    Response deleteUserDriveringLicenceInfoById(@PathVariable("id") Integer id);
    // =========  end of  UserDriveringLicenceInfo 模块   ===========

    //  =========  start of  UserDriverLicenceInfo 模块   ===========
    /**
     * 新增或编辑用户驾驶证件信息
     *
     * @param userDriverLicenceInfo 用户驾驶证件信息
     * @return
     */
    @PostMapping("/api/user/driverLicenceInfo/saveOrUpdateUserDriverLicenceInfo")
    Response saveOrUpdateUserDriverLicenceInfo(@RequestBody UserDriverLicenceInfo userDriverLicenceInfo);

    /**
     * 根据id查询用户驾驶证件信息
     *
     * @param id 用户驾驶证件信息id
     * @return
     */
    @GetMapping("/api/user/driverLicenceInfo/{id}")
    Response<UserDriverLicenceInfo> getUserDriverLicenceInfoById(@PathVariable("id") Integer id);

    /**
     * 根据用户code查询用户驾驶证件信息
     *
     * @param userCode 用户code
     * @return
     */
    @GetMapping("/api/user/driverLicenceInfo/getByUserCode/{userCode}")
    Response<UserDriverLicenceInfo> getUserDriverLicenceInfoByUserCode(@PathVariable("userCode") String userCode);

    /**
     * 根据用户code删除用户驾驶证件信息
     *
     * @param userCode 用户code
     * @return
     */
    @PostMapping("/api/user/driverLicenceInfo/deleteByUserCode/{userCode}")
    Response deleteUserDriverLicenceInfoByUserCode(@PathVariable("userCode") String userCode);

    /**
     * 根据用户id删除用户驾驶证件信息
     *
     * @param id 用户id
     * @return
     */
    @PostMapping("/api/user/driverLicenceInfo/deleteById/{id}")
    Response deleteUserDriverLicenceInfoById(@PathVariable("id") Integer id);
    // =========  end of  UserDriverLicenceInfo 模块   ===========

    //  =========  start of  UserCarInformation 模块   ===========
    /**
     * 用户车辆信息更新和保存
     * @param userCarInformation
     * @return
     */
    @PostMapping("/api/user/user-car-information")
    Response saveOrUpdateUserCarInformation(@RequestBody UserCarInformation userCarInformation);

    /**
     * 用户车辆列表
     * @param code
     * @return
     */
    @GetMapping("/api/user/user-car-information/findUserCarInfomation")
    Response<List<UserCarInformation>> findUserCarInfomation(@RequestParam(name = "code") String code);
    // =========  end of  UserCarInformation 模块   ===========
}
