package com.rocoinfo.aop.logger;

import com.rocoinfo.aop.logger.container.BeanContainer;
import com.rocoinfo.aop.logger.entity.LoggerEntity;
import com.rocoinfo.aop.logger.principal.PrincipalConstants;
import com.rocoinfo.aop.logger.principal.PrincipalHandler;
import com.rocoinfo.aop.logger.principal.PrincipalHandlerHelper;
import com.rocoinfo.aop.logger.principal.ServletRequestPrincipalHandler;
import eu.bitwalker.useragentutils.UserAgent;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.annotation.PostConstruct;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import java.lang.reflect.Method;
import java.util.*;

/**
 * <dl>
 * <dd>Description: 日志切面</dd>
 * <dd>Company: 大城若谷信息技术有限公司</dd>
 * <dd>@date：2017/5/17 上午11:48</dd>
 * <dd>@author：Aaron</dd>
 * </dl>
 */
@Aspect
@Component
public class LoggerAspect {

    @Value("${aspect.logger.app.name:}")
    private String app;

    /**
     * 获取当前登录用户的类的class
     */
    @Value("${aspect.logger.principal.handler:}")
    private String beanClazz;

    /**
     * 日志类的名称
     */
    @Value("${aspect.logger.name:com.rocoinfo.api}")
    private String loggerName;

    /**
     * 实际记录日志的logger
     */
    private org.slf4j.Logger slf4jLogger;

    /**
     * 默认的日志类名称
     */
    private static final String DEFALUT_LOGGER_NAME = "com.rocoinfo.api";

    @PostConstruct
    public void init() {
        // 创建slf4jLogger对象
        if (loggerName == null || "".equals(loggerName.trim())) {
            loggerName = DEFALUT_LOGGER_NAME;
        }
        slf4jLogger = LoggerFactory.getLogger(loggerName);

        if (beanClazz == null || "".equals(beanClazz.trim())) {
            return;
        }
        BeanContainer.registerBean(PrincipalConstants.HANDLER_NAME, beanClazz);
        BeanContainer.addBean(PrincipalConstants.HELPER_NAME, new BeanContainer.BeanHandler() {
            @Override
            public Object get() {
                Object o = BeanContainer.get(PrincipalConstants.HANDLER_NAME);
                if (o.getClass().equals(PrincipalHandler.class)) {
                    return PrincipalHandlerHelper.getInstance((PrincipalHandler) o);
                }
                if (o instanceof ServletRequestPrincipalHandler) {
                    return PrincipalHandlerHelper.getInstance((ServletRequestPrincipalHandler) o);
                }
                return null;
            }
        });
//        try {
//            Object o = Class.forName(beanClazz).newInstance();
//            if (o instanceof PrincipalHandler) {
//                principalHandlerHelper = PrincipalHandlerHelper.getInstance((PrincipalHandler) o);
//            }
//
//            if (o instanceof ServletRequestPrincipalHandler) {
//                principalHandlerHelper = PrincipalHandlerHelper.getInstance((ServletRequestPrincipalHandler) o);
//            }
//        } catch (InstantiationException | IllegalAccessException | ClassNotFoundException e) {
//            e.printStackTrace();
//        }
    }

    /**
     * 定义切点
     */
    @Pointcut("within(com.rocoinfo..*) && @annotation(com.rocoinfo.aop.logger.Logger)")
    public void logger() {
    }

    @Around("logger()")
    public Object around(ProceedingJoinPoint jp) throws Throwable {

        // 开始时间
        Long startTime = System.currentTimeMillis();
        LoggerEntity loggerEntity = this.buildLoggerEntity(jp);
        // 执行方法
        Object result = null;
        try {
            result = jp.proceed();
        } catch (Throwable throwable) {
            loggerEntity.setSuccess(Boolean.FALSE)
                    .setMessage(throwable.getMessage());
            slf4jLogger.error(LoggerJsonUtils.toJson(loggerEntity));
        }

        // 方法执行结束时间
        Long endTime = System.currentTimeMillis();
        loggerEntity.setSuccess(Boolean.TRUE)
                .setResult(result)
                .setExecTime(endTime - startTime);
        // 记录日志
        slf4jLogger.info(LoggerJsonUtils.toJson(loggerEntity));
        return result;
    }

    /**
     * 排除指定类型的参数 ServletRequest 及 ServletResponse
     *
     * @param args 待过滤的参数
     * @return 返回过滤后的参数列表
     */
    private Object[] excludeType(Object[] args) {
        List<Object> argsList = new ArrayList<>();
        if (args.length > 0) {
            for (Object arg : args) {
                if (arg instanceof ServletRequest || arg instanceof ServletResponse) {
                    continue;
                }
                argsList.add(arg);
            }
        }
        return argsList.toArray();
    }

    /**
     * 构建日志实体对象
     *
     * @return 返回日志实体类
     */
    private LoggerEntity buildLoggerEntity(ProceedingJoinPoint jp) {
        // 获取方法参数,过滤掉 request和response
        Object[] args = excludeType(jp.getArgs());
        // 获取方法的签名
        MethodSignature signature = (MethodSignature) jp.getSignature();
        // 获取方法参数名
        String[] paramNames = signature.getParameterNames();
        Method method = signature.getMethod();
        Logger logger = method.getAnnotation(Logger.class);
        //获取代理方法签名
        Class clazz = method.getDeclaringClass();
        String methodSignature = clazz.getName() + "." + method.getName();
        // 获取请求的ip url
        String ip = this.getIp();
        String url = this.getRequestUrl();
        //获取当前登录用户名
        String username = this.getUsername();
        //获取请求类型 POST GET DELETE 等
        String requestMethod = this.getRequestMethod();
        //获取浏览器类型
        String browserType = this.getBrowser();

        //构建方法参数名->值
        Map<String, Object> params = this.builderParams(paramNames, args);
        LoggerEntity loggerEntity = new LoggerEntity()
                .setApp(app)
                .setModule(logger.module())
                .setIp(ip)
                .setUrl(url)
                .setDate(new Date())
                .setRequestMethod(requestMethod)
                .setMethodignature(methodSignature)
                .setParams(params)
                .setUsername(username)
                .setBrowserType(browserType);
        return loggerEntity;
    }

    /**
     * 构建方法参数
     *
     * @param paramNames 方法参数名
     * @param args       方法参数值
     * @return 返回方法参数名->参数值的map
     */
    private Map<String, Object> builderParams(String[] paramNames, Object[] args) {
        if (paramNames != null && args != null && paramNames.length > 0 && args.length > 0) {
            int size = paramNames.length > args.length ? args.length : paramNames.length;
            Map<String, Object> params = new HashMap<>(size);
            for (int i = 0; i < size; i++) {
                params.put(paramNames[i], args[i]);
            }
            return params;
        }
        return new HashMap<>();
    }

    /**
     * 获取Request
     *
     * @return
     */
    private HttpServletRequest getHttpServletRequest() {
        return ((ServletRequestAttributes) RequestContextHolder.currentRequestAttributes()).getRequest();
    }

    /**
     * 获取当前ip
     *
     * @return
     */
    private String getIp() {
        HttpServletRequest request = this.getHttpServletRequest();
        return request.getRemoteAddr();
    }

    /**
     * 获取当前访问的url
     *
     * @return
     */
    private String getRequestUrl() {
        HttpServletRequest request = this.getHttpServletRequest();
        return request.getRequestURI();
    }

    /**
     * 获取请求方式
     *
     * @return 返回请求方式
     */
    private String getRequestMethod() {
        HttpServletRequest request = this.getHttpServletRequest();
        return request.getMethod();
    }

    /**
     * 获取浏览器类型
     *
     * @return 返回浏览器类型
     */
    private String getBrowser() {
        HttpServletRequest request = this.getHttpServletRequest();
        UserAgent userAgent = UserAgent.parseUserAgentString(request.getHeader("User-Agent"));
        return userAgent.getBrowser().toString();
    }

    /**
     * 获取当前登录用户名
     *
     * @return
     */
    private String getUsername() {
        PrincipalHandlerHelper principalHandlerHelper = this.getPrincipalHandlerHelper();
        if (principalHandlerHelper != null) {
            HttpServletRequest request = this.getHttpServletRequest();
            return principalHandlerHelper.getUsername(request);
        }
        return "";
    }

    private PrincipalHandlerHelper getPrincipalHandlerHelper() {
        return BeanContainer.getBean(PrincipalConstants.HELPER_NAME, PrincipalHandlerHelper.class);
    }

}
