package com.rocoinfo.utils;

import org.apache.commons.lang3.RandomStringUtils;
import org.apache.commons.lang3.Validate;

import java.util.Random;
import java.util.UUID;


/**@author zhangmin
 * 随机数帮助类**/
public final class RandomUtil {
	
	
	private static final String RAND_CHARS = "0123456789abcdefghigklmnopqrstuvtxyzABCDEFGHIGKLMNOPQRSTUVWXYZ";
	
	private static final int[] PREFIX = { 1, 2, 3, 4, 5, 6, 7, 8, 9 };
	
	private static Random random = new Random();


    /**
     * 封装JDK自带的UUID, 通过Random数字生成, 中间有-分割.
     */
    public static String uuid() {
        return UUID.randomUUID().toString();
    }

    /**
     * 封装JDK自带的UUID, 通过Random数字生成, 中间无-分割.
     */
    public static String uuid2() {
        return uuid().replaceAll("-", "");
    }


    /**
     * 返回0到Intger.MAX_VALUE的随机Int
     */
    public static int rand() {
        int n = random.nextInt();
        if (n == Integer.MIN_VALUE) {
            n = 0; // corner case
        } else {
            n = Math.abs(n);
        }
        return n;
    }
	
	/***产生[0-max)范围的随机数**/
	public static int rand(int max) {
		return random.nextInt(max);
	}

    /***产生[min-max)范围的随机数
     *如果@param min>=max 直接返回 min
     *如果 min<max && min<0 则返回[0-max)范围的随机数
     */
    public static int rand(int min, int max){
        if(min<max){
            if(min>0){
                return rand(max-min)+min;
            }else{
                return rand(max);
            }
        }else{
            return min;
        }
    }

    /**
     * 返回0－Long.MAX_VALUE间的随机Long, 使用ThreadLocalRandom.
     */
    public static long randLong() {
        long n = random.nextLong();
        if (n == Long.MIN_VALUE) {
            n = 0; // corner case
        } else {
            n = Math.abs(n);
        }
        return n;
    }

    /**
     * 返回min－max间的随机Long, 使用ThreadLocalRandom.
     *
     * min必须大于0.
     */
    public static long randLong(long min, long max) {
        Validate.isTrue(max >= min, "Start value must be smaller or equal to end value.");
        if (min == max) {
            return min;
        }
        return (long) (min + ((max - min) * random.nextDouble()));
    }


    /**
     * 随机产生最大为18位的long型数据(long型数据的最大值是9223372036854775807,共有19位)
     *
     * @param digit   用户指定随机数据的位数
     */
    public static long randLongFixLength(int digit) {
        if (digit >= 19 || digit <= 0) {
            throw new IllegalArgumentException("digit should between 1 and 18(1<=digit<=18)");
        }
        String s = RandomStringUtils.randomNumeric(digit - 1);
        return Long.parseLong(getPrefix() + s);
    }

    /**
     * 随机产生在指定位数之间的long型数据,位数包括两边的值,minDigit<=maxDigit
     *
     * @param minDigit  用户指定随机数据的最小位数 minDigit>=1
     * @param maxDigit 用户指定随机数据的最大位数 maxDigit<=18
     */
    public static long randLongRandomLength(int minDigit, int maxDigit)   {
        if (minDigit > maxDigit) {
            throw new IllegalArgumentException("minDigit > maxDigit");
        }
        if (minDigit <= 0 || maxDigit >= 19) {
            throw new IllegalArgumentException("minDigit <=0 || maxDigit>=19");
        }

        return randLongFixLength(minDigit + getDigit(maxDigit - minDigit));
    }

    /**返回长度为length的随机字符串
	 * @param length 要生成随机字符串的长度
	 * @param randType==0,则包括所有的字符的字符串,randType==1，返回字母串, randType==2 返回数字串
	 */
	public static String getRandStr(int length, int randType) {
		StringBuffer hash = new StringBuffer(length);
		if(randType==1) {//返回字母串
			for (int i = 0; i < length; i++) {
                hash.append(RAND_CHARS.charAt(10 + random.nextInt(52)));
            }
		}else if(randType==2) {//返回数字串
			for (int i = 0; i < length; i++) {
                hash.append(RAND_CHARS.charAt(random.nextInt(10)));
            }
		}else{
			for (int i = 0; i < length; i++) {
                hash.append(RAND_CHARS.charAt(random.nextInt(62)));
            }
		}
		return hash.toString();
	}

	/** 
     * 根据数字生成一个定长的字符串，长度不够前面补0
     * @param num   数字  例如：12
     * @param fixedLength 字符串长度  例如：4
     * @return 定长的字符串  例如返回：0012
     */ 
    public static String toFixdLengthString(int num, int fixedLength) {
            StringBuffer sb = new StringBuffer(); 
            String strNum = String.valueOf(num); 
            if (fixedLength - strNum.length() >= 0) {
                    sb.append(generateZeroString(fixedLength - strNum.length()));
            } else { 
                    throw new RuntimeException("将数字" + num + "转化为长度为" + fixedLength + "的字符串发生异常！");
            } 
            sb.append(strNum); 
            return sb.toString(); 
    }
	   

    private static int getDigit(int max) {    
        return random.nextInt(max + 1);    
     }    
   
    /** 
      * 保证第一位不是零 
      * @return 
      */   
    private static String getPrefix() {    
        return PREFIX[random.nextInt(PREFIX.length)] + "";
    }
    
    /** 
    * 生成一个定长的纯0字符串 
    * @param length 字符串长度 
    * @return 纯0字符串 
    */ 
   private static String generateZeroString(int length){ 
       StringBuffer sb = new StringBuffer(); 
       for(int i = 0; i < length; i++){ 
           sb.append('0'); 
       } 
       return sb.toString(); 
   }


    /**
     * 随机字母或数字，随机长度
     */
    public static String randomStringRandomLength(int minLength, int maxLength) {
        return RandomStringUtils.random(rand(minLength, maxLength), 0, 0, true, true, null,random);
    }


    /**
     * @function 生成length位的随机字符串(数字、大小写字母随机混排)
     * @param length
     */
    public static String randBigSmallLetterNumberFixLength(int length) {
        String str = "";
        for(int i=0;i < length;i++){
            int intVal=(int)(Math.random()*58+65);
            if(intVal >= 91 && intVal <= 96){
                i--;
            }
            if(intVal < 91 || intVal > 96){
                if(intVal%2==0){
                    str += (char)intVal;
                }else{
                    str += (int)(Math.random()*10);
                }
            }
        }
        return str;
    }

    /**
     * @function 生成length位的随机字符串(数字、小写字母随机混排)
     */
    public static String randSmallLetterNumberFixLength(int length) {

        String str = "";
        for(int i=0;i < length;i++){
            int intVal=(int)(Math.random()*26+97);
            if(intVal%2==0){
                str += (char)intVal;
            }else{
                str += (int)(Math.random()*10);
            }
        }
        return str;
    }

    /**
     * @function 生成length位的随机字符串(大写字母与数字混排)
     */
    public static String randBigLetterNumberFixLength(int length) {
        String str = "";
        for(int i=0;i < length;i++){
            int intVal=(int)(Math.random()*26+65);
            if(intVal%2==0){
                str += (char)intVal;
            }else{
                str += (int)(Math.random()*10);
            }
        }
        return str;
    }

}