package com.rocoinfo.utils.base;

import com.rocoinfo.utils.io.FileUtil;
import com.rocoinfo.utils.io.URLResourceUtil;
import com.rocoinfo.utils.text.Charsets;
import org.apache.commons.lang3.math.NumberUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.util.Date;
import java.util.Properties;

/**
 * 关于Properties的工具类
 * 
 * 1. 统一风格读取Properties到各种数据类型
 * 
 * 2. 从文件或字符串装载Properties
 */
public class PropertiesUtil {

	private static final Logger logger = LoggerFactory.getLogger(PropertiesUtil.class);

	/////////////////// 读取Properties ////////////////////


	public static Boolean getBoolean(Properties p, String name, Boolean defaultValue) {
        return BooleanUtil.toBooleanObject(p.getProperty(name), defaultValue);
	}

	public static Integer getInt(Properties p, String name, Integer defaultValue) {
		return NumberUtils.toInt(p.getProperty(name), defaultValue);
	}

	public static Long getLong(Properties p, String name, Long defaultValue) {
		return NumberUtils.toLong(p.getProperty(name), defaultValue);
	}

	public static Double getDouble(Properties p, String name, Double defaultValue) {
		return NumberUtils.toDouble(p.getProperty(name), defaultValue);
	}

	public static String getString(Properties p, String name, String defaultValue) {
		return p.getProperty(name, defaultValue);
	}

	/////////// 加载Properties////////
	/**
	 * 从文件路径加载properties. 默认使用utf-8编码解析文件
	 * 
	 * 路径支持从外部文件或resources文件加载, "file://"或无前缀代表外部文件, "classpath://"代表resources
	 */
	public static Properties loadFromFile(String generalPath) {
		Properties p = new Properties();
		try (Reader reader = new InputStreamReader(URLResourceUtil.asStream(generalPath), Charsets.UTF_8)) {
			p.load(reader);
		} catch (IOException e) {
			logger.warn("Load property from " + generalPath + " failed", e);
		}
		return p;
	}

	/**
	 * 从字符串内容加载Properties
	 */
	public static Properties loadFromString(String content) {
		Properties p = new Properties();
		try (Reader reader = new StringReader(content)) {
			p.load(reader);
		} catch (IOException ignored) { // NOSONAR
		}
		return p;
	}


	/**
	 * @param props
	 * @param filePath 要加载的属性配置文件的完整名称
	 */
	public static void storePropertiesToFile(Properties props, String filePath) {
		OutputStream fos = null;
		try {
			fos = new FileOutputStream(filePath);
			props.store(fos, "By CommonUtils.storePropertiesToFile, " + new Date());
		} catch (Exception e) {
			throw new RuntimeException(e);
		}finally {
			FileUtil.closeIO(fos);
		}
	}
	
}