package com.rocoinfo.utils.img;

import com.google.common.collect.Maps;
import com.google.zxing.BarcodeFormat;
import com.google.zxing.EncodeHintType;
import com.google.zxing.MultiFormatWriter;
import com.google.zxing.client.j2se.MatrixToImageWriter;
import com.google.zxing.common.BitMatrix;
import com.google.zxing.qrcode.decoder.ErrorCorrectionLevel;
import com.rocoinfo.utils.io.FileUtil;
import com.rocoinfo.utils.text.Charsets;
import org.apache.commons.io.IOUtils;

import java.io.OutputStream;
import java.util.Map;


/**
 * <dl>
 * <dd>Description: 用于生成二位码的工具类</dd>
 * <dd>Company: 大城若谷信息技术有限公司</dd>
 * <dd>@date：2015/10/27 16:21</dd>
 * <dd>@author：Aaron</dd>
 * </dl>
 */
public final class QrCodeUtil {

    /**
     * 私有化构造方法,禁止创建实例
     */
    private QrCodeUtil() {
    }

    /**
     * 默认宽度
     */
    private static final int DEFAULT_WIDTH = 300;

    /**
     * 默认长度
     */
    private static final int DEFAULT_HEIGHT = 300;

    /**
     * 默认格式类型
     */
    private static final String DEFAULT_FORMAT = "png";

    /**
     * 二维码周围白边的宽度
     */
    private static final Integer DEFAULT_MARGIN = 0;

    /**
     * 默认纠错等级
     * 纠错登记越高可携带内容越少
     */
    private static final ErrorCorrectionLevel DEFAULT_ECL = ErrorCorrectionLevel.L;

    /**
     * 生成二维码图片
     *
     * @param content 二维码携带内容
     * @param width   二维码宽度
     * @param height  二维码高度
     * @param format  二维码格式类型
     * @param margin  二维码内容距离边框的距离(白边的宽度)
     * @param ecl     纠错等级,等级越高,携带内容越少
     * @param os      输出流,二维码图片会写入到指定的输出流
     * @return 生成二维码图片的url
     */
    public static boolean generateQRCode(String content, int width, int height, String format, Integer margin, ErrorCorrectionLevel ecl, OutputStream os) {
        Map<EncodeHintType, Object> hint = Maps.newConcurrentMap();
        hint.put(EncodeHintType.CHARACTER_SET, Charsets.UTF_8_NAME);
        hint.put(EncodeHintType.ERROR_CORRECTION, ecl);
        hint.put(EncodeHintType.MARGIN, margin);
        try {
            BitMatrix matrix = new MultiFormatWriter().encode(content, BarcodeFormat.QR_CODE, width, height, hint);
            MatrixToImageWriter.writeToStream(matrix, format, os);
        } catch (Exception e) {
            return false;
        } finally {
            FileUtil.closeIO(os);
        }
        // 如果失败,并且没有抛出异常,返回null
        return false;
    }

    /**
     * 生成二维码
     *
     * @param content 二维码携带内容
     * @param os      输出流,二维码图片会写入到指定的输出流
     * @return
     */
    public static boolean generateQRCode(String content, OutputStream os) {
        return generateQRCode(content, DEFAULT_WIDTH, DEFAULT_HEIGHT, DEFAULT_FORMAT, DEFAULT_MARGIN, DEFAULT_ECL, os);
    }
}
